#!/bin/bash

PROJECT="nextcloud"

BSDIFF=$(which bsdiff)

if [[ "$?" -ne 0 ]]
then
    echo "Les commandes bsdiff et bspatch sont nécessaires pour la compression et la décompression des sources"
    exit 1
fi

BSPATCH=$(which bspatch)

SOURCES_DIR=$(dirname $0)/src/share 

CMD=$1
OPTION=$2


function usage() {
    echo $0 "expand|contract"
}


function apply_next_patch() {
    ref_source=$1
    source_dirname=$(dirname "$ref_source")
    source_basename=$(basename "$ref_source")
    source_basename_wo_ext=${source_basename%.tar}

    patch=$(find "$SOURCES_DIR" -type f -name "${source_basename_wo_ext}*.patch")

    if [[ -z "$patch" ]]
    then
        next_source=""
        return 0
    fi

    next_source=${patch%.patch}
    next_source="${SOURCES_DIR}/${next_source#*-to-}.tar"

    bspatch "$ref_source" "${next_source}" "$patch" >/dev/null 2>&1

    echo $next_source

}


function expand_sources() {
    patches=$(find "$SOURCES_DIR" -name "${PROJECT}-*.patch")
    ref_source=$(find "$SOURCES_DIR" -name "${PROJECT}-*.tar")
    echo "La version de base est $ref_source"

    base_source="$ref_source"
    while [[ -n "$base_source" ]]
    do
        base_source="$(apply_next_patch "$base_source")"
        echo "$base_source"
    done

    pushd "$SOURCES_DIR" >/dev/null
    find "./" -type f -name "${PROJECT}-*.tar" -exec tar xf {} --owner=0 --group=0 --no-same-owner --no-same-permissions --no-acls --no-xattrs \;
    find "./" -type f -name "${PROJECT}-*.tar" -exec rm {} \;
    popd >/dev/null
    
}


function compute_next_patch() {
    base_source=$1
    next_source=$2

    source_basename=$(basename "$base_source")
    source_basename=${source_basename%.tar}
    next_basename=$(basename "$next_source")
    next_basename=${next_basename%.tar}

    patch="${SOURCES_DIR}/${source_basename}-to-${next_basename}.patch"

    echo "Computing patch for $next_source based on $base_source"

    bsdiff "$base_source" "$next_source" "$patch"
}


function contract_sources() {
    base=$1
    if [[ "$base" = "first" ]]
    then
        sort_cmd="sort"
    else
        sort_cmd="sort -r"
    fi

    echo "Archiving sources"
    pushd "$SOURCES_DIR" >/dev/null
    find "./" -type d -name "${PROJECT}-*" -exec tar cf {}.tar {} --owner=0 --group=0 --no-same-owner --no-same-permissions --no-acls --no-xattrs \;
    popd >/dev/null

    echo "Calculating patches should take some time!"
    sources=$(find "$SOURCES_DIR" -type f -name "${PROJECT}-*.tar" | $sort_cmd )
    ref_source=$(find "$SOURCES_DIR" -type f -name "${PROJECT}-*.tar" | $sort_cmd | head -n 1)
    echo $ref_source

    for base_source in $sources
    do
        next_source=$(echo ${sources#*${base_source}})
        next_source=${next_source%% *}
        if [[ -n "$next_source" ]]
        then
            compute_next_patch "$base_source" "$next_source" &
        else
            break
        fi
    done
    wait
}


case $CMD in
    expand)
        expand_sources
        ;;
    contract)
        contract_sources "$OPTION"
        ;;
    *)
        usage
        exit 1
        ;;
esac


exit 0
