<?php
/**
 * @copyright Copyright (c) 2016, ownCloud, Inc.
 *
 * @author Arthur Schiwon <blizzz@arthur-schiwon.de>
 * @author Joas Schilling <coding@schilljs.com>
 * @author Morris Jobke <hey@morrisjobke.de>
 * @author Roeland Jago Douma <roeland@famdouma.nl>
 * @author Roger Szabo <roger.szabo@web.de>
 * @author Vinicius Cubas Brand <vinicius@eita.org.br>
 *
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License, version 3,
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 *
 */

namespace OCA\OCCTools\Jobs;

use OC\BackgroundJob\TimedJob;
use OCA\User_LDAP\Helper;
use OCA\User_LDAP\LDAP;
use OCA\User_LDAP\Group_Proxy;
use OCA\User_LDAP\Mapping\GroupMapping;

/**
 * Remaned Group and flush the cache
 *
 * @package OCA\User_LDAP\Jobs;
 */
class SyncGroups extends TimedJob {
    protected $db;
    protected $gproxy;
    protected $gmapping;
    protected $version;

	public function __construct() {
        $this->interval = self::getRefreshInterval();
	}

	public function setArguments() {
        $ldapHelper     = new Helper(\OC::$server->getConfig());
        $ldap           = new LDAP();

        $this->db       = \OC::$server->getDatabaseConnection();
        $this->gmapping = new GroupMapping($this->db);
        $this->gproxy   = new Group_Proxy($ldapHelper->getServerConfigurationPrefixes(true),$ldap,\OC::$server->query('LDAPGroupPluginManager'));  
        $this->version  = \OC::$server->getConfig()->getSystemValue('version', '');
	}

	public function run($argument) {
		$this->setArguments();

        // A partir de la version 19 : il n'y a plus de problème de renommage des groupes on bypass ce job
        if($this->version>="19.0.0.0") return;
                
        // On récupère l'ensemble des clés dn<>cn
        $ldapgroups=$this->getAccess()->searchGroups("cn=*",["cn","dn"]);
        $tbname=[];
        foreach($ldapgroups as $group) {
            $tbname[$group["dn"][0]] = $group["cn"][0];
        }

        // On recherche l'ensemble des groupes enregistrer dans la table ldap_group_mapping
		$groups = $this->gmapping->getList();
		foreach($groups as $group) {
            // On récupère name et dn et realname
            $name=$group["name"];
            $dn=$group["dn"];
            $realname=(array_key_exists($dn,$tbname)?$tbname[$dn]:null);
            
            // Si le groupe n'a pas de vrai nom ldap c'est qu'il n'existe plus
            if(is_null($realname)) continue;

            // Y a-t-il une différence de nom entre le cn et nom nextcloud
            if($name!=$realname) {
                // oc_ldap_group_members
                $update = $this->db->getQueryBuilder();
                $update ->update('ldap_group_members')
                        ->set('owncloudname', $update->createNamedParameter($realname))
                        ->where($update->expr()->eq('owncloudname', $update->createNamedParameter($name)));
                $update->execute();

                // oc_share
                $update = $this->db->getQueryBuilder();
                $update ->update('share')
                        ->set('share_with', $update->createNamedParameter($realname))
                        ->where($update->expr()->eq('share_type', $update->createNamedParameter("1")))
                        ->andWhere($update->expr()->eq('share_with', $update->createNamedParameter($name)));
                $update->execute();
                
                // oc_dav_shares
                $update = $this->db->getQueryBuilder();
                $update ->update('dav_shares')
                        ->set('principaluri', $update->createNamedParameter("principals/groups/".$realname))
                        ->where($update->expr()->eq('principaluri', $update->createNamedParameter("principals/groups/".$name)));
                $update->execute();

                
                // oc_group_folders_groups
                $update = $this->db->getQueryBuilder();
                $update ->update('group_folders_groups')
                        ->set('group_id', $update->createNamedParameter($realname))
                        ->where($update->expr()->eq('group_id', $update->createNamedParameter($name)));
                $update->execute();

                // oc_ldap_group_mapping
                $update = $this->db->getQueryBuilder();
                $update ->update('ldap_group_mapping')
                        ->set('owncloud_name', $update->createNamedParameter($realname))
                        ->where($update->expr()->eq('owncloud_name', $update->createNamedParameter($name)));
                $update->execute();

                // Il y a eu changement on  clear le cache de l'ensemble des membres du groupe
                $users=$this->gproxy->usersInGroup($realname);
                foreach($users as $user) {
                    $cacheKey = 'getUserGroups'.$user;
                    $this->gproxy->clearCache($cacheKey);
                }
            }
        }
    }
    
    private function getAccess() {
        $ldapWrapper = new LDAP();
        $helper = new Helper(\OC::$server->getConfig());
        $serverConfigPrefixes = $helper->getServerConfigurationPrefixes(true);

        $userManager = new \OCA\User_LDAP\User\Manager(
            \OC::$server->getConfig(),
            new \OCA\User_LDAP\FilesystemHelper(),
            new \OCA\User_LDAP\LogWrapper(),
            \OC::$server->getAvatarManager(),
            new \OCP\Image(),
            \OC::$server->getDatabaseConnection(),
            \OC::$server->getUserManager(),
            \OC::$server->getNotificationManager());

		foreach($serverConfigPrefixes as $configPrefix) {
            $configuration = new \OCA\User_LDAP\Configuration($configPrefix);
            $con = new \OCA\User_LDAP\Connection($ldapWrapper, $configPrefix, null);
            $con->setConfiguration($configuration->getConfiguration());
            $con->ldapConfigurationActive = true;
            $con->setIgnoreValidation(true);
            
            return new \OCA\User_LDAP\Access(
                $con,
                $ldapWrapper,
                $userManager,
                new \OCA\User_LDAP\Helper(\OC::$server->getConfig()),
                \OC::$server->getConfig(),
                \OC::$server->getUserManager()
            );
        }    
    }    

	static private function getRefreshInterval() {
		return \OC::$server->getConfig()->getAppValue('user_ldap', 'bgjRefreshInterval', 3600);
	}    
}
