# -*- coding:utf-8 -*-
from sqlalchemy.types import UnicodeText, Binary, Boolean, Text, Integer
from sqlalchemy import Column, ForeignKeyConstraint, ForeignKey
from sqlalchemy.orm import relationship, backref
from os.path import join
from os import unlink
from IPy import IP
from .elixir import Entity
from arv.lib.util import (valid, try_unique_column, get_keyid_in_certif,
    get_keyid_from_certifstring, trace, bin_encoding, suppress_colon,
    cred_end_validity_date, escape_special_characters, der_to_pem,
    split_pkcs7, is_ca)
from creole import cert
from arv.config import (x509_default_key_bits, ssl_default_cert_time,
    ca_conf_file, conf_file_orig, ssl_dir)
from arv.lib.logger import logger


class TmplEdge(Entity):
    __tablename__ = 'arv_db_edge_tmpledge'
    id = Column("id", Integer, primary_key=True)
    name = Column("name", UnicodeText, unique=True)
    tmpl_vertex_a_id = Column(Integer, index=True)
    tmpl_vertex_b_id = Column(Integer, index=True)
    tmpl_connect_id = Column(Integer, index=True)
    state = Column("state", Integer, default=0)
    __table_args__ = (ForeignKeyConstraint(['tmpl_vertex_a_id'], [u'arv_db_node_tmplvertex.id'], **{'name': u'arv_db_edge_tmpledge_tmpl_vertex_a_id_fk'}),
                      ForeignKeyConstraint(['tmpl_vertex_b_id'], [u'arv_db_node_tmplvertex.id'], **{'name': u'arv_db_edge_tmpledge_tmpl_vertex_b_id_fk'}),
                      ForeignKeyConstraint(['tmpl_connect_id'], [u'arv_db_edge_tmplconnect.id'], **{'name': u'arv_db_edge_tmpledge_tmpl_connect_id_fk'})
    )

    def add_edge(self, connect):
        try:
            return Edge(tmpl_edge=self, connect=connect)
        except Exception as e:
            raise ValueError(str(e))

    def mod_name(self, name):
        name = valid(name, 'string')
        if name != self.name:
            if TmplEdge.query.filter_by(name=name).first() != None:
                raise Exception('Name {0} already used'.format(name))
            self.name = name

    def mod_tmpl_vertex1(self, tmpl_vertex1):
        if self.tmpl_vertex_a !=  tmpl_vertex1:
            if self.edges != []:
                raise Exception('This TmplEdge is already used, cannot modify tmpl_vertex')
            self.tmpl_vertex_a = tmpl_vertex1

    def mod_tmpl_vertex2(self, tmpl_vertex2):
        if self.tmpl_vertex_b !=  tmpl_vertex2:
            if self.edges != []:
                raise Exception('This TmplEdge is already used, cannot modify tmpl_vertex')
            self.tmpl_vertex_b = tmpl_vertex2

@trace()
def get_tmpl_edges(tmpl_connect):
    try:
        return TmplEdge.query.filter_by(tmpl_connect=tmpl_connect).all()
    except:
        raise Exception('error in get_tmpl_edges')

@trace()
def del_tmpl_edge(tmpl_edge, tmpl_connect=None):
    for connect in tmpl_edge.tmpl_connect.connects:
        for edge in connect.edges:
            if edge.tmpl_edge == tmpl_edge:
                logger.debug("edge {0} between {1} and {2}".format(tmpl_edge.name, connect.head_node.name, connect.tail_node.name))
                raise Exception("Can't suppress edge template used by connect !!!")
    try:
        tmpl_edge.delete()
    except:
        raise Exception("error in del_tmpl_edge")

@trace()
def del_tmpl_vertex(tmpl_vertex):
    if tmpl_vertex.tmpl_edges_a != [] or tmpl_vertex.tmpl_edges_b != []:
        raise Exception('TmplVertex is already used by a TmplEdge, you cannot delete it')
    if tmpl_vertex.vertices != []:
        for vertex in tmpl_vertex.vertices:
            vertex.delete()
    try:
        tmpl_vertex.delete()
    except:
        raise Exception("error in del_tmpl_vertex")


class TmplConnect(Entity):
    __tablename__ = 'arv_db_edge_tmplconnect'
    id = Column("id", Integer, primary_key=True)
    name = Column("name", UnicodeText, unique=True)
    tmpl_node_a_id = Column(Integer, index=True, nullable=False)
    tmpl_node_b_id = Column(Integer, index=True, nullable=False)
    cred_auth_id = Column(Integer, index=True)
    leftsendcert = Column("leftsendcert", Text, default='always')
    fragmentation = Column("fragmentation", Text, default='no')
    state = Column("state", Integer, default=0)
    tmpl_edges = relationship('TmplEdge', backref="tmpl_connect", primaryjoin="TmplEdge.tmpl_connect_id == TmplConnect.id", uselist=True)
    __table_args__ = (ForeignKeyConstraint(['tmpl_node_a_id'], [u'arv_db_node_tmplnode.id'], **{'name': u'arv_db_edge_tmplconnect_tmpl_node_a_id_fk'}),
                      ForeignKeyConstraint(['tmpl_node_b_id'], [u'arv_db_node_tmplnode.id'], **{'name': u'arv_db_edge_tmplconnect_tmpl_node_b_id_fk'}),
                      ForeignKeyConstraint(['cred_auth_id'], [u'arv_db_edge_credential.id'], **{'name': u'arv_db_edge_tmplconnect_cred_auth_id_fk'})
    )

    def _add_tmpl_edge(self, name, tmpl_vertex_a, tmpl_vertex_b):
        name = valid(name, 'string')
        tmpl = TmplEdge(name=name, tmpl_vertex_a=tmpl_vertex_a, tmpl_vertex_b=tmpl_vertex_b, tmpl_connect=self)
        self.tmpl_edges.append(tmpl)
        return tmpl

    def add_tmpl_edge(self, name, tmpl_vertex_a, tmpl_vertex_b):
        if TmplEdge.query.filter_by(tmpl_vertex_a=tmpl_vertex_a, tmpl_vertex_b=tmpl_vertex_b, tmpl_connect=self).first() != None:
            raise Exception('TmplEdge already exists')
        return try_unique_column('add_tmpl_edge', self._add_tmpl_edge,
                name=name, tmpl_vertex_a=tmpl_vertex_a,
                tmpl_vertex_b=tmpl_vertex_b)

    def add_connect(self, tail_node, head_node, tail_extr, head_extr, tail_cred, head_cred,
                    leftsendcert=None, fragmentation=None, rw_dns=None):
        try:
            if tail_node == head_node or tail_extr == head_extr:
                raise Exception("wrong argument on add_connect")
            connects = get_connects()
            for connect in connects:
                if ((head_node, tail_node) == (connect.head_node, connect.tail_node) or \
                    (head_node, tail_node) == (connect.tail_node, connect.head_node)) and \
                   self.cred_auth == connect.tmpl_connect.cred_auth:
                    raise Exception("Connect with this CA already exists for these nodes")
            if leftsendcert is not None:
                final_leftsendcert = leftsendcert
            else:
                final_leftsendcert = self.leftsendcert

            if fragmentation is not None:
                final_fragmentation = fragmentation
            else:
                final_fragmentation = self.fragmentation

            if head_node.tmpl_node.mimetype == 'roadwarrior' or tail_node.tmpl_node.mimetype == 'roadwarrior':
                try:
                    if isinstance(rw_dns, bytes):
                        rw_dns = rw_dns.decode()
                    IP("{0}/32".format(rw_dns))
                except:
                    raise Exception("IP address must be set for source IP and DNS")

            return Connect(tmpl_connect=self, tail_node=tail_node,
                    head_node=head_node, tail_extr=tail_extr,
                    head_extr=head_extr, tail_cred=tail_cred,
                    head_cred=head_cred, leftsendcert=final_leftsendcert,
                    fragmentation=final_fragmentation, rw_dns=rw_dns)
        except Exception as e:
            raise ValueError(str(e))

@trace()
def get_local_ca():
    return Credential.query.filter_by(ca=True, local=True).first()

@trace()
def get_credentials_auth():
    try:
        return Credential.query.filter_by(ca=True).all()
    except:
        raise Exception("error in get_credentials_auth")

@trace()
def get_all_credentials():
    try:
        return Credential.query.all()
    except:
        raise Exception("error in get_all_credentials")

@trace()
def get_credential_auth(credential):
    try:
        return Credential.query.filter_by(credential=credential).first()
    except:
        return None

@trace(hide_args=[1], hide_kwargs=['passwd'])
def add_credential_auth(credential, passwd=None, local=False):
    if isinstance(credential, str):
        credential = credential.encode()
    try:
        subject = cert.get_subject(cert=credential)
        name = str(subject[1])
        suffix_cred = subject[0]
        suffix_cred = valid(suffix_cred, 'string')
        subject = bin_encoding("{0}, CN = {1}".format(suffix_cred, name))
        name = valid(name, 'string')
        expiration_date = cred_end_validity_date(credential)
        if get_credential(name) is not None:
            raise Exception('Credential with same name already exists')
        if local:
            # get the keyid from the private key
            fh = open(cert.ca_keyfile, 'rb')
            private_key = fh.read()
            fh.close()
            subjkey, keyid_cred = get_keyid_from_certifstring(private_key, passwd, mode='rsa')
            subjkey = suppress_colon(subjkey)
            keyid_cred = suppress_colon(keyid_cred)
            return Credential(name=name, suffix_cred=suffix_cred,
                        subject=subject, private_key=private_key,
                        credential=credential, expiration_date=expiration_date,
                        ca=True, local=local, keyid_cred=keyid_cred,
                        subjkey=subjkey)
        else:
            #FIXME: calculer les subjkey et keyid
            subjkey, keyid_cred = get_keyid_from_certifstring(credential, mode='x509')
            subjkey = suppress_colon(subjkey)
            keyid_cred = suppress_colon(keyid_cred)
            return Credential(name=name, suffix_cred=suffix_cred, subject=subject,
                              credential=credential, ca=True, local=local,
                              expiration_date=expiration_date,
                              keyid_cred=keyid_cred, subjkey=subjkey)
    except UnicodeDecodeError as e:
        raise Exception("Problème de lecture du fichier : %s" % e)
    except Exception as e:
        raise Exception("error in add_cred_auth: %s" % e)

@trace(hide_args=[1], hide_kwargs=['passwd'])
def add_certificate_authorities(certif, cert_type=None):
    """Add certificate autorities :
        - certif is in DER or PEM format
        - type_cert is 'pkcs7' or 'x509'
        - Returns last intermediate CA certificate
    """
    try:
        ca = None
        if cert_type is None:
            cert_type, certificate = der_to_pem(certif)
        else:
            certificate = certif
        if cert_type == "pkcs7":
            ca_cred, certificate = split_pkcs7(certificate)
            for ca_cert in ca_cred:
                subject = cert.get_subject(cert=ca_cert)
                name = str(subject[1])
                suffix_cred = subject[0]
                suffix_cred = valid(suffix_cred, 'string')
                subject = bin_encoding("{0}, CN = {1}".format(suffix_cred, name))
                ca = Credential.query.filter_by(subject=subject).first()
                if ca == None:
                    ca = add_credential_auth(ca_cert)
            if ca is None:
                raise Exception('unexpected error, cannot find a CA')
        elif cert_type == "x509":
            if is_ca(certificate):
                subject = cert.get_subject(cert=certificate)
                name = str(subject[1])
                suffix_cred = subject[0]
                suffix_cred = valid(suffix_cred, 'string')
                subject = bin_encoding("{0}, CN = {1}".format(suffix_cred, name))
                ca = Credential.query.filter_by(subject=subject).first()
                if ca == None:
                    ca = add_credential_auth(certificate)
            else:
                raise Exception('the certificate is not a CA')
        else:
            raise Exception("Unknown certificate type")
        return ca
    except Exception as e:
        raise ValueError(str(e))

@trace()
def add_tmpl_connect(name, node_a, node_b, cred_auth, leftsendcert='always', fragmentation='no'):
    try:
        name = valid(name, 'string')

        if not cred_auth or not cred_auth.ca:
            raise Exception('add_tmpl_connect: attribute cred_auth not a credential authority')
    except Exception as err:
        raise ValueError(err)
    if TmplConnect.query.filter_by(name=name).first() != None:
        raise Exception("Name should be unique")
    try:
        return TmplConnect(name=name, tmpl_node_a=node_a, tmpl_node_b=node_b, cred_auth=cred_auth, leftsendcert=leftsendcert, fragmentation=fragmentation)
    except:
        raise Exception("error in add_tmpl_connect")

@trace()
def get_tmpl_connects(tmpl_node_a=None, tmpl_node_b=None):
    try:
        if tmpl_node_a == None:
            return TmplConnect.query.all()
        else:
            #tmpl_node_a could be node_a or node_b and tmpl_node_b could be node_a or node_b
            tmpl_connects = set(TmplConnect.query.filter_by(tmpl_node_a=tmpl_node_a,
                    tmpl_node_b=tmpl_node_b).all())
            union = tmpl_connects | set(TmplConnect.query.filter_by(tmpl_node_b=tmpl_node_a,
                    tmpl_node_a=tmpl_node_b).all())
            return list(union)
    except Exception as e:
        raise Exception('error in get_tmpl_connects : %s '% str(e))

@trace()
def del_tmpl_connect(tmpl_connect):
    if Connect.query.filter_by(tmpl_connect=tmpl_connect).first():
        raise Exception("TmplConnect is already used by a Connect, you cannot delete it")
    try:
        for tmpledge in tmpl_connect.tmpl_edges:
            del_tmpl_edge(tmpledge, tmpl_connect)
    except:
        import traceback
        traceback.print_exc()
        raise Exception("error when delete tmpledges")
    try:
        tmpl_connect.delete()
    except:
        raise Exception("error in del_tmpl_connect")

# ____________________________________________________________
class Edge(Entity):
    __tablename__ = 'arv_db_edge_edge'
    id = Column("id", Integer, primary_key=True)
    tmpl_edge_id = Column(Integer, index=True)
    tmpl_edge = relationship('TmplEdge', backref="edges", primaryjoin="Edge.tmpl_edge_id == TmplEdge.id", uselist=False)
    connect_id = Column(Integer, index=True)
    state = Column("state", Integer, default=0)
    __table_args__ = (ForeignKeyConstraint(['tmpl_edge_id'], [u'arv_db_edge_tmpledge.id'], **{'name': u'arv_db_edge_edge_tmpl_edge_id_fk'}),
                      ForeignKeyConstraint(['connect_id'], [u'arv_db_edge_connect.id'], **{'name': u'arv_db_edge_edge_connect_id_fk'})
    )


@trace()
def get_edges(nodea=None, nodeb=None, tmpl_connect=None):
    if nodea == None and nodeb == None and tmpl_connect == None:
        return Edge.query.all()
    tmpl_node_a = nodea.tmpl_node
    tmpl_node_b = nodeb.tmpl_node
    edges=[]
    if tmpl_connect.edges != None:
        for tmpledge in tmpl_connect.edges:
            #nodea nodeb
            edges.extend(tmpledge.edges)
    return edges

@trace()
def del_edge(edge):
    try:
        edge.delete()
    except:
        raise Exception("error in del_edge")

class Connect(Entity):
    __tablename__ = 'arv_db_edge_connect'
    id = Column("id", Integer, primary_key=True)
    tail_node_id = Column(Integer, index=True)
    head_node_id = Column(Integer, index=True)
    tail_extr_id = Column(Integer, index=True)
    head_extr_id = Column(Integer, index=True)
    tail_cred_id = Column(Integer, index=True)
    head_cred_id = Column(Integer, index=True)
    tmpl_connect_id = Column(Integer, index=True, nullable=False)
    tmpl_connect = relationship('TmplConnect', backref="connects", primaryjoin="Connect.tmpl_connect_id == TmplConnect.id", uselist=False)
    leftsendcert = Column("leftsendcert", Text, default='always')
    fragmentation = Column("fragmentation", Text, default='no')
    rw_dns = Column("rw_dns", UnicodeText, )
    state = Column("state", Integer, default=0)
    edges = relationship("Edge", backref="connect", primaryjoin="Edge.connect_id == Connect.id", uselist=True)
    __table_args__ = (ForeignKeyConstraint(['tail_node_id'], [u'arv_db_node_node.id'], **{'name': u'arv_db_edge_connect_tail_node_id_fk'}),
                      ForeignKeyConstraint(['head_node_id'], [u'arv_db_node_node.id'], **{'name': u'arv_db_edge_connect_head_node_id_fk'}),
                      ForeignKeyConstraint(['tail_extr_id'], [u'arv_db_node_extremity.id'], **{'name': u'arv_db_edge_connect_tail_extr_id_fk'}),
                      ForeignKeyConstraint(['head_extr_id'], [u'arv_db_node_extremity.id'], **{'name': u'arv_db_edge_connect_head_extr_id_fk'}),
                      ForeignKeyConstraint(['tail_cred_id'], [u'arv_db_edge_credential.id'], **{'name': u'arv_db_edge_connect_tail_cred_id_fk'}),
                      ForeignKeyConstraint(['head_cred_id'], [u'arv_db_edge_credential.id'], **{'name': u'arv_db_edge_connect_head_cred_id_fk'}),
                      ForeignKeyConstraint(['tmpl_connect_id'], [u'arv_db_edge_tmplconnect.id'], **{'name': u'arv_db_edge_connect_tmpl_connect_id_fk'})
    )

    def mod_tail_extr(self, extr):
        if extr != self.tail_extr:
            self.tail_extr = extr

    def mod_head_extr(self, extr):
        if extr != self.head_extr:
            self.head_extr = extr

    def mod_tail_cred(self, cred):
        if cred != self.tail_cred:
            self.tail_cred = cred

    def mod_head_cred(self, cred):
        if cred != self.head_cred:
            self.head_cred = cred

@trace()
def get_connects(nodea=None, nodeb=None):
    try:
        if nodea == None and nodeb == None:
            return Connect.query.all()
        if nodea == None:
            connect = Connect.query.filter_by(head_node=nodeb).all()
            connect.extend(Connect.query.filter_by(tail_node=nodeb).all())
            return connect
        if nodeb == None:
            connect = Connect.query.filter_by(tail_node=nodea).all()
            connect.extend(Connect.query.filter_by(head_node=nodea).all())
            return connect
        connect = Connect.query.filter_by(tail_node=nodea, head_node=nodeb).all()
        connect.extend(Connect.query.filter_by(head_node=nodea, tail_node=nodeb).all())
        return connect
    except Exception as e:
        raise ValueError('error in get_connects: '+str(e))

@trace()
def get_connected_nodes(nodea):
    try:
        nodeb = []
        connects = Connect.query.filter_by(head_node=nodea).all()
        for connect in connects:
            nodeb.append(connect.tail_node)
        connects = Connect.query.filter_by(tail_node=nodea).all()
        for connect in connects:
            nodeb.append(connect.head_node)
        return nodeb
    except Exception as e:
        raise ValueError('error in get_connected_nodes: '+str(e))

@trace()
def del_connect(connect):
    try:
        connect.delete()
    except:
        raise Exception("error in del_connect")


class Credential(Entity):
    __tablename__ = 'arv_db_edge_credential'
    id = Column("id", Integer, primary_key=True)
    name = Column("name", UnicodeText, unique=True)
    credential = Column("credential", Binary)
    unsigned_credential = Column("unsigned_credential", Binary)
    ca = Column("ca", Boolean)
    # only if ca = False
    private_key = Column("private_key", Binary, unique=True)
    node_id = Column(Integer, index=True)
    # only if ca = True
    suffix_cred = Column("suffix_cred", Text)
    subject = Column("subject", Binary, unique=True)
    local = Column("local", Boolean, default=False)
    ## only if ca = False
    cred_auth_id = Column(Integer, index=True)
    keyid_cred = Column("keyid_cred", Binary)
    subjkey = Column("subjkey", Binary)
    state = Column("state", Integer, default=0)
    expiration_date = Column("expiration_date", Text, default='')
    head_conns = relationship('Connect', backref="head_cred", primaryjoin="Connect.head_cred_id == Credential.id", uselist=True)
    tail_conns = relationship('Connect', backref="tail_cred", primaryjoin="Connect.tail_cred_id == Credential.id", uselist=True)
    tmpl_connects = relationship('TmplConnect', backref="cred_auth", primaryjoin="TmplConnect.cred_auth_id == Credential.id", uselist=True)
    ## only if ca = True
    credentials = relationship('Credential', backref=backref("cred_auth", uselist=False, remote_side=[id]), primaryjoin="Credential.cred_auth_id == Credential.id", uselist=True)
    __table_args__ = (ForeignKeyConstraint(['node_id'], [u'arv_db_node_node.id'], **{'name': u'arv_db_edge_credential_node_id_fk'}),
                      ForeignKeyConstraint(['cred_auth_id'], [u'arv_db_edge_credential.id'], **{'name': u'arv_db_edge_credential_cred_auth_id_fk'})
    )

    @trace(hide_args=[2], hide_kwargs=['passwd'])
    def mod_keys(self, credential, passwd, private_key=None):
        """
        """
        if self.ca:
            raise Exception('This credential is a CA')
        if not private_key:
            private_key = self.private_key
        self.node.import_credential(private_key, credential, passwd,
                                    old_credential=self, old_ca=self.cred_auth)

    @trace()
    def mod_ca(self, credential):
        """
        """
        if not self.ca:
            raise Exception('This credential is not a CA')
        subject = cert.get_subject(cert=credential)
        name = str(subject[1])
        suffix_cred = subject[0]
        suffix_cred = valid(suffix_cred, 'string')
        subject = bin_encoding("{0}, CN = {1}".format(suffix_cred, name))
        if self.subject != subject:
            raise Exception("Not same subject as old one's")
        subjkey, keyid_cred = get_keyid_from_certifstring(credential, mode='x509')
        subjkey = suppress_colon(subjkey)
        keyid_cred = suppress_colon(keyid_cred)
        if self.subjkey != subjkey:
            raise Exception("Not same subjkey as old one's")
        if self.keyid_cred != keyid_cred:
            raise Exception("Not same keyid as old one's")
        expiration_date = cred_end_validity_date(credential)
        self.credential = credential
        self.expiration_date = expiration_date

@trace()
def del_cred(credential):
    if credential.ca:
        if credential.credentials != []:
            raise Exception('CA is already use by a Credential, cannot delete it')
    else:
        if credential.head_conns != [] or credential.tail_conns != []:
            raise Exception('Credential is already used by a Connect, you cannot delete it')
    credential.delete()


#Table Crl: liste des crl d'une CA
class Crl(Entity):
    __tablename__ = 'arv_db_edge_crl'
    id = Column("id", Integer, primary_key=True)
    uri = Column("uri", Text)
    ca_id = Column(Integer, index=True)
    ca = relationship('Credential', backref="clr_uris", primaryjoin="Crl.ca_id == Credential.id", uselist=False)
    __table_args__ = (ForeignKeyConstraint(['ca_id'], [u'arv_db_edge_credential.id'], **{'name': u'arv_db_edge_crl_ca_id_fk'}),)


@trace()
def add_crl_uri(cred_auth, uri):
    return Crl(ca=cred_auth, uri=uri)

@trace()
def get_credential_by_ca_node(cred_auth, node):
    return Credential.query.filter_by(cred_auth=cred_auth, node=node).first()

@trace()
def get_credential(name):
    return Credential.query.filter_by(name=name).first()

@trace()
def add_credential(name, passwd, node, cred_priv_type):
    try:
        if Credential.query.filter_by(name=name.encode()).first() != None:
            raise Exception('Name should be unique')
        uname = valid(name, 'string')
        #passwd  = valid(passwd, 'string')
        #passwd = str(passwd)
        conf_file = '/tmp/%s.conf'%name
        certfile = join(ssl_dir, 'certs', '%s.pem'%name)
        keyfile = join(ssl_dir, 'private', 'priv-%s.pem'%name)
        reqfile = join(ssl_dir, 'req', '%s.p10'%name)
        #copy(conf_file_orig, conf_file)
        fh = open(conf_file_orig, 'r')
        content = fh.read()
        fh.close()
        content = content.replace('@@DEFAULT_DAYS@@', ssl_default_cert_time)
        content = content.replace('@@CN-SERVEUR@@', name)
        content = content.replace('@@PASSWD@@', escape_special_characters(passwd))
        if node.tmpl_node.mimetype == 'sphynx':
            content = content.replace('encrypt_key\t\t= yes', 'encrypt_key\t\t= no')
        fh = open(conf_file, 'w')
        logger.debug('%s content: %s'%(conf_file, content))
        if isinstance(content, bytes):
            content = content.decode()
        fh.write(content)
        fh.close()
        cert.load_conf({'ssl_dir': ssl_dir, 'start_index': "01", 'conf_file': conf_file,
            'ca_conf_file': ca_conf_file, 'ssl_default_cert_time': ssl_default_cert_time,
            'ssl_default_key_bits': x509_default_key_bits})
        if cred_priv_type == 'autosigned':
            cert.gen_certif(certfile=certfile,
                            keyfile=keyfile,
                            del_passwd=False,
                            regen=True)

            fh = open(certfile, 'rb')
            credential = fh.read()
            fh.close()
            fh = open(keyfile, 'rb')
            private_key = fh.read()
            fh.close()
            local_ca = get_local_ca()
            if local_ca == None:
                raise Exception('No local CA found')
            # get the keyid from the certificate
            subjkey, keyid_cred = get_keyid_in_certif(certfile, passwd=passwd, certiftype='notca')
            subjkey = suppress_colon(subjkey)
            keyid_cred = suppress_colon(keyid_cred)
            node.add_credential(name=uname,
                                private_key=private_key,
                                credential=credential,
                                cred_auth=local_ca,
                                keyid_cred=keyid_cred,
                                subjkey=subjkey,
                                passwd=passwd)
            unlink(certfile)
            unlink(keyfile)
            unlink(reqfile)
            unlink(conf_file)
            return True
        elif cred_priv_type == 'cred_request':
            cert.gen_certif(certfile=certfile,
                            keyfile=keyfile,
                            del_passwd=False,
                            signe_req=False,
                            regen=True)
            fh = open(keyfile, 'rb')
            private_key = fh.read()
            fh.close()
            fh = open(reqfile, 'rb')
            req_credential = fh.read()
            fh.close()
            unlink(conf_file)
            return req_credential
        else:
            raise Exception('Unknown cred_priv_type %s, should be autosigned or cred_request'%cred_priv_type)

        unlink(conf_file)
    except UnicodeDecodeError as e:
        raise Exception("Problème de lecture du fichier : %s" % e)


