# -*- coding: utf-8 -*-

"""Test L{creole.client} API
"""

import sys
import unittest

from creole.client import CreoleClient
from creole.client import NotFoundError

good_variable_name = u'eole_version'
good_variable_path = u'/creole/general/{0}'.format(good_variable_name)
good_variable_key = good_variable_path.lstrip(u'/').replace(u'/', u'.')

unknown_variable_name = u'absolutely_inexistant'
unknown_variable_path = u'/creole/general/{0}'.format(unknown_variable_name)

# z_stats is not available on Zéphir module
root_services = [u'cron', u'ead-server', u'eoleflask', u'nginx', u'bastion']

if sys.version_info[0] >= 3:
    unicode = str

class TestCreoleClient(unittest.TestCase):
    """CreoleClient Tests

    """

    client = CreoleClient()


    def check_services(self, services, first=None, last=None, contains=None):
        """Check services
        FIXME (first, last) : with upstart Creole ordrer doesn't matter

        """
        self.assertIsInstance(services, list)
        self.assertGreaterEqual(len(services), 1)
        self.assertIsInstance(services[0], dict)
        names = [ s[u'name'] for s in services ]

        if first is not None:
            self.assertEqual(names[0],
                             first,
                             msg=u'“{0}” must be first service'.format(first))

        if last is not None:
            # z_stats can be the last service (#9486)
            if names[-1] == u'z_stats':
                place = -2
            else:
                place = -1
            self.assertEqual(names[place],
                             last,
                             msg='“{0}” must be last service'.format(last))

        if contains is not None:
            if not isinstance(contains, list):
                contains = [contains]
            for service in contains:
                self.assertIn(service, names)


    def test_get_root(self):
        """Get root configuration.

        """
        config = self.client.get('/')
        self.assertIsInstance(config, dict)
        self.assertIn(good_variable_key, config)


    def test_list_root(self):
        """List configuration root path.

        """
        root = self.client.list('/')
        self.assertIsInstance(root, list)
        self.assertIn(u'creole/', root)


    def test_get_variable(self):
        """Get variable with full path.

        """
        version = self.client.get(good_variable_path)
        self.assertIsInstance(version, unicode)


    def test_list_variable(self):
        """Listing a variable result in a list of its value.

        """
        version = self.client.get(good_variable_path)
        version_list = self.client.list(good_variable_path)
        self.assertIsInstance(version_list, list)
        self.assertEqual(len(version_list), 1)
        self.assertEqual(version_list[0], version)


    def test_get_creole(self):
        """Get full creole configuration

        """
        creole = self.client.get_creole()
        self.assertIsInstance(creole, dict)
        self.assertIn(good_variable_name, creole)


    def test_get_creole_variable(self):
        """Get creole variable

        """
        version = self.client.get(good_variable_path)
        creole_version = self.client.get_creole(good_variable_name)
        self.assertIsInstance(version, unicode)
        self.assertEqual(creole_version, version)


    def test_get_NotFoundError(self):
        """Unknown variable must raise :exc:`NotFoundError`

        """
        self.assertRaises(NotFoundError, self.client.get, path=unknown_variable_path)


    def test_get_creole_NotFoundError(self):
        """Unknown variable must raise :exc:`NotFoundError`

        """
        self.assertRaises(NotFoundError, self.client.get_creole, name=unknown_variable_name)


    def test_get_services(self):
        """Ordering of services with :meth:`CreoleClient.get_services()`

        """
        services = self.client.get_services()
        self.check_services(services, u'cron', contains=root_services)


    def test_root_container_services(self):
        """Ordering of services for ``root`` container

        This check :meth:`CreoleClient.get_container_infos()`.

        """
        root = self.client.get_container_infos(u'root')
        services = root['services']
        self.check_services(services, u'cron', contains=root_services)


    def test_root_group_services(self):
        """Ordering of services for ``root`` group

        This check :meth:`CreoleClient.get_group_infos()`.

        """
        root = self.client.get_group_infos(u'root')
        services = root['services']
        self.check_services(services, u'cron', contains=root_services)

