# Copyright (C) 2010-2012 Team Gaspacho (see README for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from os.path import join, isdir, dirname
from os import makedirs
from shutil import rmtree

from gaspacho import get_choices, get_softwares, get_oses, get_platforms
from gaspacho.group import get_all_groups
from gaspacho.config import store_dir_apply, default_separator
from gaspacho.conflevel import get_confcomputer, get_confuser, get_confcontext
from gaspacho.log import trace #, logger

try:
    import json
except:
    #for python 2.5
    import simplejson as json

@trace
def get_plugins_apply(group, user, platform, conflevel):
    confcomputer = get_confcomputer()
    ret = {'plugins': {}}
    path = platform.path.name
    osname = platform.osversion.os.name
    osversion = platform.osversion.name
    extension = platform.path.extension
    info = platform.path.info
    for rule, choice in get_choices(group, user, [platform],
                            force_level=conflevel).items():
        choice, hchoice = choice
        if choice == None:
            choice = hchoice
        if choice == None:
            raise Exception('choice must not be None')
        tchoice = choice.state
        tvalue = choice.value
        rtype = rule.type
        conf = rule.conflevels
        if rtype == u'multi':
            mseparator = rule.options.get(u'separator', default_separator)
            tmvalue = tvalue.split(mseparator)
            i = 0
            mvalue = {}
            for val in rule.options[u'values']:
                mvalue[val[u'name']] = tmvalue[i]
                i+=1
        for variable in rule.vars:
            if variable.conflevel != conflevel:
                continue
            if platform not in variable.plats:
                continue
            von = variable.value_on
            voff = variable.value_off
            key = variable.name
            typ = variable.type
            if tchoice == u'on':
                if von == None:
                    value = tvalue
                else:
                    if rtype == u'multi':
                        value = von
                        for k, v in mvalue.items():
                            value = value.replace(k, v)
                    else:
                        value = von
            elif tchoice == u'off':
                value = voff
            elif tchoice == u'free':
                continue
                #FIXME : avant value = '', mais pourquoi ???
            else:
                raise Exception('unknown choice: %s'%tchoice)
            if value == u'IGNORE':
                #if value == IGNORE, rule must not be changed
                continue
            if osversion == u'ALL':
                osversions = []
                for osversion in platform.osversion.os.get_versions():
                    osversion = osversion.name
                    if osversion == u'ALL':
                        continue
                    osversions.append(osversion)
            else:
                osversions = [osversion]
            ret['plugins'].setdefault(extension, {}).setdefault(path, [])
            for osversion in osversions:
                tvariable = {'key': key, 'type': typ, 'value': value}
                if info != None:
                    tvariable['info'] = info
                ret['plugins'][extension][path].append(tvariable)
    if conflevel == confcomputer and group.installpkg:
        ret.setdefault('packages', [])
        for package in platform.packages:
            lang = group.lang
            if package.get('lang', lang) == lang:
                if package['name'] not in \
                        ret['packages']:
                    ret['packages'].append(package['name'])
    return ret

@trace
def gen_choice_level(gasp, group, platform, level, conflevel):
    osversion = platform.get_osversion()
    osversionname = osversion.name
    osname = osversion.os.name
    name = "%s_%s" % (osname, osversionname)
    base_dir = build_choice_dir(group, platform)
    #all
    all_dir = join(base_dir, level)
    filename = join(all_dir, 'ALL.gasp')
    path = {'all': {name: filename}}
    plugins_apply = get_plugins_apply(group, None, platform, conflevel)
    for extension, datas in plugins_apply['plugins'].items():
        for tpath, tvariables in datas.items():
            gasp.setdefault(filename, {'plugins': {}})['plugins'].setdefault(
                                extension, {}).setdefault(tpath, []
                                ).extend(tvariables)
    #user
    for user in group.users:
        user_dir = join(base_dir, user.type, level)
        filename = join(user_dir, '%s.gasp' % user.name)
        path.setdefault(user.type, {})[user.name] = \
                       {name: filename}
        plugins_apply = get_plugins_apply(group, user, platform, conflevel)
        for extension, datas in plugins_apply['plugins'].items():
            for tpath, tvariables in datas.items():
                gasp.setdefault(filename, {'plugins': {}})['plugins'].setdefault(
                                        extension, {}).setdefault(tpath, []
                                        ).extend(tvariables)
    return path

@trace
def build_choice_dir(group, platform):
    osversion = platform.get_osversion()
    osversionname = osversion.name
    osname = osversion.os.name
    return join(store_dir_apply, group.name, osname,
                             osversionname)

def gen_choice_computer(gasp, group, platform):
    #computer
    confcomputer = get_confcomputer()
    base_dir = build_choice_dir(group, platform)
    filename = join(base_dir, 'computer.gasp')
    osversion = platform.get_osversion()
    osversionname = osversion.name
    osname = osversion.os.name
    name = "%s_%s" % (osname, osversionname)
    path = {name: filename}
    plugins_apply = get_plugins_apply(group, None, platform, confcomputer)
    for extension, datas in plugins_apply['plugins'].items():
        for tpath, tvariables in datas.items():
            gasp.setdefault(filename, {'plugins': {}})['plugins'].setdefault(
                                extension, {}).setdefault(tpath, []
                                ).extend(tvariables)
    if 'packages' in plugins_apply:
        gasp[filename].setdefault('packages', []).append(
                                    plugins_apply['packages'])
    return path

@trace
def apply_choices(create_file=True):
    ret = []
    if create_file and isdir(store_dir_apply):
        rmtree(store_dir_apply)
    for group in list(get_all_groups()):
        computer = [(name, _type) for name, _type in group.get_computers()]
        #if no computer, don't generate group
        if computer == []:
            continue
        tgroup = {'group': computer, 'path': {'user': {},
                        'context': {}, 'computer': {}}}

        #calculate platforms
        platforms = get_platforms()
        tplatforms = None
        if group.oses != [] or group.softs != []:
            if group.oses != []:
                for platform in platforms:
                    if platform.osversion.os not in group.oses:
                        platforms.remove(platform)
            if group.softs != []:
                for platform in platforms:
                    if platform.softwareversion.software not in group.softs:
                        platforms.remove(platform)
            tplatforms = platforms
        gasp = {}
        for platform in platforms:
            for typ, users in gen_choice_level(gasp, group, platform,
                                            'user', get_confuser()).items():
                if typ != u'all':
                    for user, datas in users.items():
                        tgroup['path']['user'].setdefault(typ, {}).setdefault(user, {}).update(datas)
                else:
                    tgroup['path']['user'].setdefault(typ, {}).update(users)
            for typ, contexts in gen_choice_level(gasp, group, platform,
                                        'context', get_confcontext()).items():
                if typ != u'all':
                    for context, datas in contexts.items():
                        tgroup['path']['context'].setdefault(typ, {}).setdefault(context, {}).update(datas)
                else:
                    tgroup['path']['context'].setdefault(typ, {}).update(
                                                         contexts)
            computer = gen_choice_computer(gasp, group, platform)
            tgroup['path']['computer'].update(computer)
        if create_file:
            for filename, content in gasp.items():
                _dir = dirname(filename)
                if not isdir(_dir):
                    makedirs(_dir)
                fh = file(filename, 'w')
                json.dump(content, fh, indent=1)
                fh.close()
        ret.insert(0, tgroup)
    if create_file:
        filename = join(store_dir_apply, 'gaspacho.json')
        fh = file(filename, 'w')
        json.dump(ret, fh, indent=1)
        fh.close()
    else:
        return ret

# vim: ts=4 sw=4 expandtab
