# Copyright (C) 2010-2011 Team Gaspacho (see README for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"""
Create and manage Category, Tag and ConfLevel

Category: subdivision of rules
Tag: subcategories
ConfLevel: if rules are only for computers or for computers/users

ConfLevel    Category
    |            |
    |           Tag
    '--- Rule ---'

"""

from elixir import (Entity, Field, UnicodeText, ManyToOne, OneToMany,
                    ManyToMany, PickleType)
from copy import copy

from gaspacho.valid import valid
from gaspacho.rule import Rule
from gaspacho.config import ordered_lang
from gaspacho.log import trace #, logger

class Category(Entity):
    """
    Elixir class for Categories

    comment: description of the category
    tags: each category have one or many tags
    """
    name = Field(PickleType)
    comment = Field(UnicodeText)
    tags = OneToMany('Tag')

#tag
    @trace
    def get_tag(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            #search all object set
            objs = Tag.query.filter_by(category=self).all()
            if objs != None:
                for obj in objs:
                    #name is an object with lang as key
                    lname = obj.name.get(lang, None)
                    #if label name is equal to name
                    if lname == name:
                        return obj
            return None
        except TypeError, e:
            raise Exception("Error in get_tag: " + str(e))

    @trace
    def get_tags(self):
        return Tag.query.filter_by(category=self).all()

    @trace
    def add_tag(self, comment=u''):
        """use this function to add Tag object in database
        May have a comment.
        """
        try:
            if self.name == {}:
                raise Exception('Need translation before add tag')
            comment = valid(comment, 'unicode')
            return Tag(comment=comment, category=self, name={})
        except Exception, e:
            raise Exception("Error in add_tag: " + str(e))

    def del_tag(self, tag):
        try:
            valid(tag, Tag)
            if tag.get_rules() != []:
                raise Exception('Remove associated rules before')
            tag.delete()
        except Exception, e:
            raise Exception("Error in del_tag: " + str(e))

#translation/name
    @trace
    def add_translation(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            if self.name.has_key(lang):
                raise Exception('This category has a translation in %s' % lang)
            if get_category(name=name, lang=lang) != None:
                raise Exception('The category %s in %s already exists' %
                                    (name, lang))
            tname = copy(self.name)
            tname[lang] = name
            self.name = tname
        except Exception, e:
            raise Exception("Error in add_translation: " + str(e))

    @trace
    def mod_translation(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            if not self.name.has_key(lang):
                raise Exception('This category has not a translation for %s' % lang)
            if get_category(name=name, lang=lang) != None:
                raise Exception('The category %s in %s already exists' % (name, lang))
            self.name[lang] = name
        except Exception, e:
            raise Exception("Error in mod_translation: " + str(e))

    @trace
    def get_name(self, lang):
        try:
            if self.name.has_key(lang):
                return self.name[lang]
            for tlang in ordered_lang:
                if self.name.has_key(tlang):
                    return self.name[tlang]
            return None
        except Exception, e:
            raise Exception("Error in get_name: " + str(e))

#comment
    @trace
    def get_comment(self):
        return self.comment

    @trace
    def mod_comment(self, comment):
        try:
            comment = valid(comment, 'unicode')
            self.comment = comment
        except Exception, e:
            raise Exception("Error in mod_comment: " + str(e))

@trace
def add_category(comment=u''):
    """use this function to add Category object in database
    Need, optionaly, a comment.
    """
    try:
        comment = valid(comment, 'unicode')
        return Category(comment=comment, name={})
    except Exception, e:
        raise Exception("Error in add_category: " + str(e))

@trace
def get_category(name, lang):
    try:
        name = valid(name, 'unicode')
        lang = valid(lang, 'unicode')
        #search all object set
        objs = Category.query.all()
        if objs != None:
            for obj in objs:
                #name is an object with lang as key
                lname = obj.name.get(lang, None)
                #if label name is equal to name
                if lname == name:
                    return obj
        return None
    except TypeError, e:
        raise Exception("Error in get_category: " + str(e))

@trace
def get_categories(grp=None, user_level=False, display=False):
    """
    return all categories with (optionaly) grp ou conflevel
    """
    from gaspacho import get_confuser, get_confcontext
    confuser = get_confuser()
    confcontext = get_confcontext()

    #if no group specified or group without software and os
    if grp == None or (grp.softs == [] and grp.oses == []):
        #get all categories
        categories = Category.query.all()
        if user_level == False:
            #confcomputer level
            if not display:
                return categories
            else:
                ret = []
                for category in categories:
                    for tag in category.tags:
                        if Rule.query.filter_by(tag=tag, display=True).all():
                            ret.append(category)
                            break
                return ret
        else:
            #else search if at least one Rule in confuser or confcontext
            ret = []
            for category in categories:
                is_break = False
                for tag in category.tags:
                    if display:
                        rules = Rule.query.filter_by(tag=tag,
                                display=True).all()
                    else:
                        rules = Rule.query.filter_by(tag=tag).all()
                    for rule in rules:
                        if confuser in rule.conflevels or \
                                confcontext in rule.conflevels:
                            ret.append(category)
                            is_break = True
                            break
                    if is_break:
                        is_break = False
                        break
            return ret
    else:
        #if group is specified and has softwares/oses
        retsoftwares = []
        retoses = []
        #search categories for those softwares/oses
        softwares = grp.softs
        oses = grp.oses
        #search categories for softwares
        if softwares != []:
            for software in softwares:
                for rule in software.rules:
                    category = rule.tag.category
                    if user_level == False or confuser in rule.conflevels or \
                                confcontext in rule.conflevels:
                        retsoftwares.append(category)
            #remove duplicated categories
            ret = set(retsoftwares)
        if oses != []:
            for os in oses:
                for rule in os.rules:
                    category = rule.tag.category
                    if user_level == False or confuser in rule.conflevels or \
                                confcontext in rule.conflevels:
                        retoses.append(category)
            #remove duplicated oses 
            ret = set(retoses)
        if softwares != [] and oses != []:
            #if softwares and oses specified, only return same categories
            ret = set(retsoftwares) & set(retoses)
        #return a list, not a set
        return list(ret)

@trace
def del_category(category):
    try:
        valid(category, Category)
        if category.get_tags() != []:
            raise Exception('Remove associated tag before')
        category.delete()
    except Exception, e:
        raise Exception("Error in del_category: " + str(e))

class Tag(Entity):
    """
    Elixir class for Tags

    name: name of the tag
    comment: description of the tag
    category: each tags have in one and only one category
    rules: each rules are dispached in tags
    """
    name = Field(PickleType)
    comment = Field(UnicodeText)
    category = ManyToOne('Category')
    rules = OneToMany('Rule')

#translation/name
    @trace
    def add_translation(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            tname = copy(self.name)
            tname[lang] = name
            self.name = tname
        except Exception, e:
            raise Exception("Error in add_translation: " + str(e))

    @trace
    def mod_translation(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            if not self.name.has_key(lang):
                raise Exception('This tag has not a translation for %s' % lang)
            if self.category.get_tag(name=name, lang=lang) != None:
                raise Exception('The tag %s in %s already exists' % (name, lang))
            self.name[lang] = name
        except Exception, e:
            raise Exception("Error in mod_translation: " + str(e))

    @trace
    def get_name(self, lang):
        try:
            if self.name.has_key(lang):
                return self.name[lang]
            for tlang in ordered_lang:
                if self.name.has_key(tlang):
                    return self.name[tlang]
            return None
        except Exception, e:
            raise Exception("Error in get_name: " + str(e))

#rule
    @trace
    def add_rule(self, type, defaultvalue=None, display=True,
                defaultstate=u'free', options={}):
        """use this function to add Rule object in database
        Need a type and optionaly, defaultvalue, defaultstate
        and comment.
        """
        try:
            type = valid(type, 'type')
            display = valid(display, 'boolean')

            if type == u'enum':
                if options == {}:
                    raise Exception('if type is enum, options must not be None')
                options = valid(options, 'enum')
                is_ok=False
            else:
                #defaultvalue must be None if boolean (value_on is apply)
                #defaultvalue should validate with the type but must be
                #unicode in database
                if type != u'boolean':
                    if type == u'integer':
                        ret = set(options.keys()) - set(['min', 'max'])
                        if list(ret) != []:
                            raise Exception('Options must be min or max for integer')
                        for value in options.values():
                            valid(value, 'integer')
                    elif type == u'list':
                        ret = set(options.keys()) - set(['separator'])
                        if list(ret) != []:
                            raise Exception('Options must be only separator for list')
                    elif type == u'unicode':
                        if options != {}:
                            raise Exception('if type is unicode, options must be None')
                    elif type == u'multi':
                        keys_needed = ['values']
                        keys_optional = ['separator']
                        keys_optional.extend(keys_needed)
                        if list(set(keys_needed) - \
                                    set(options.keys())) != []:
                            raise Exception(
                                    'Multi must have attribute %s (%s)' %\
                                    (', '.join(keys_needed),
                                    ', '.join(options.keys())))
                        if list(set(options.keys()) - \
                                    set(keys_optional)) != []:
                            raise Exception(
                                    'Multi can only have attributes %s (%s)' %\
                                    (', '.join(keys_optional),
                                    ', '.join(options.keys())))
                else:
                    if options != {}:
                        raise Exception('if type is boolean, options must be None')
                defaultstate = valid(defaultstate, 'unicode')
                if defaultstate not in [u'on', u'off', u'free']:
                    raise ValueError("defaultstate must be 'on', 'off' or 'free'")
            rule = Rule(type=type, display=display, name={}, \
                    defaultstate=defaultstate, options=options, tag=self)
            rule.set_defaultvalue(defaultvalue)
        except Exception, e:
            #import traceback
            #print traceback.print_exc()
            raise Exception("Error in add_rule: " + str(e))
        return rule

    @trace
    def get_rules(self, softwares=[], oses=[], user_level=False, display=False):
        """get all rules for a tag and optionnaly for a conflevel
        display: if True filter by attribut display"""

        from gaspacho import get_confuser, get_confcontext
        confuser = get_confuser()
        confcontext = get_confcontext()

        if display:
            rules = Rule.query.filter_by(tag=self, display=True).all()
        else:
            rules = Rule.query.filter_by(tag=self).all()
        if user_level:
            trules = []
            for rule in rules:
                if confuser in rule.conflevels or \
                        confcontext in rule.conflevels:
                    trules.append(rule)
            rules = trules
        #if softwares or oses specified
        if softwares != [] or oses != []:
            trules = []
            for rule in rules:
                added = False
                if softwares != []:
                    for software in rule.softs:
                        if software in softwares:
                            added = True
                            trules.append(rule)
                            break
                if added == False and oses != []:
                    for os in rule.oses:
                        if os in oses:
                            trules.append(rule)
                            break
            return trules
        else:
            return rules

    @trace
    def get_rule(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            #search all label with rule set
            rules = Rule.query.filter_by(tag=self).all()
            if rules != None:
                for rule in rules:
                    #name is an object with lang as key
                    lname = rule.name.get(lang, None)
                    #if label name is equal to name
                    if lname == name:
                        return rule
            return None
        except Exception, e:
            raise Exception("Error in get_rule: " + str(e))

    @trace
    def del_rule(self, rule):
        try:
            valid(rule, Rule) 
            if rule.get_variables() != []:
                raise Exception('Remove associated variables before')
            rule.delete()
        except Exception, e:
            raise Exception("Error in del_rule: " + str(e))

#comment
    @trace
    def get_comment(self):
        return self.comment

    @trace
    def mod_comment(self, comment):
        try:
            comment = valid(comment, 'unicode')
            self.comment = comment
        except Exception, e:
            raise Exception("Error in mod_comment: " + str(e))

class ConfLevel(Entity):
    """
    Elixir class for ConfLevel
    
    name: name of the conflevel, only confcomputer, confusers and confcontext
          are allowed
    comment: description of the conflevel
    rules: each rules are dispached in conflevel
    """
    name = Field(UnicodeText)
    comment = Field(UnicodeText)
    rules = ManyToMany('Rule')
    variables = OneToMany('Variable')

    @trace
    def get_name(self):
        return self.name

    def __repr__(self):
        return '{"name": "%s", "comment": "%s"}' % (self.name, self.comment)

@trace
def add_conflevel(name, comment=u''):
    """use this function to add ConfLevel object in database
    Need a name and, optionaly, a comment.
    """
    try:
        name = valid(name,'unicode')
        comment = valid(comment, 'unicode')
        return ConfLevel(name=name, comment=comment)
    except Exception, e:
        raise Exception("Error in add_conflevel: " + str(e))

# vim: ts=4 sw=4 expandtab
