# Copyright (C) 2010 Team Gaspacho (see README for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
Create and manage Choices

Choice: response for a rule with specific group and user

                       ,-- Group
Rule ------- Choice ---'-- Template  User
                '----------------------'
"""

from elixir import Entity, Field, UnicodeText, ManyToOne
from gaspacho.valid import valid
from gaspacho.config import default_separator
from gaspacho.log import trace #, logger

class Choice(Entity):
    """
    Elixir class for Choice

    state: should be "on", "off" or "free". For herited choice, just
           remove it
    value: response for the choice
    rule, group/template, user: related object for the decision
    platform: if choice is only for a specific platform
    """
    state = Field(UnicodeText)
    value = Field(UnicodeText)
    rule = ManyToOne('Rule')
    group = ManyToOne('Group')
    template = ManyToOne('Template')
    user = ManyToOne('User')
    platform = ManyToOne('Platform')

#state
    @trace
    def set_state(self, state):
        try:
            state = valid(state, 'state')
        except TypeError, e:
            raise Exception("Error in set_state: ", str(e))
        try:
            self.state = state
        except Exception, e:
            raise Exception("Error in set_state: ", str(e))

    @trace
    def get_state(self):
        return self.state

    @trace
    def set_value(self, value):
        try:
            #valid value not enum and multi
            if self.rule.type != 'enum' and self.rule.type != u'multi':
                value = valid(value, self.rule.type)
            #for list or multi, check if separator in values and join
            if self.rule.type == u'list' or self.rule.type == u'multi':
                options = self.rule.options
                separator = options.get('separator',
                                                  default_separator)
                tvalue = []
                for val in value:
                    if self.rule.type == u'list' and type(val) not in [str, unicode]:
                        raise Exception('Value in list must be a string')
                    if separator in val:
                        raise Exception('Separator must not be in value')
                    tvalue.append(str(val))
                if self.rule.type == u'multi' and \
                        len(options['values']) != len(value):
                    raise Exception('Value for multi must have same len has option')
                value = separator.join(tvalue)
            #if enum, check if values are in option
            elif self.rule.type == u'enum':
                value = valid(value, 'unicode')
                is_in_option = False
                for option in self.rule.options:
                    if value == option[0]:
                        is_in_option = True
                        break
                if not is_in_option:
                    raise Exception('Value enum must be list in option enum')
            #valid unicode
            if value != None:
                value = valid(value, 'unicode')
        except TypeError, e:
            raise Exception("Error in set_value: ", str(e))
        try:
            self.value = value
        except Exception, e:
            raise Exception("Error in set_value: ", str(e))

    @trace
    def get_value(self):
        if self.rule.type in [u'list', u'multi']:
            return self.value.split(self.rule.options.get('separator',
                                                          default_separator))

        if self.rule.type == u'integer':
            return valid(self.value, self.rule.type)
        return self.value


# vim: ts=4 sw=4 expandtab
