# Copyright (C) 2010-2011 Team Gaspacho (see README for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
Create and manage Group, Computer, Template, User

Group: relation between users and computers. Group could herited from an other
       Group
Computer: computers define by name or by ip
Template: same as Group by without herired and computers
User: user name or group of users

 ,--- Group ------- Computer
 |     | '------------,
 | ,- Template ------ User
 | |
Software
"""

from elixir import (Entity, Field, UnicodeText, Boolean, ManyToMany, ManyToOne,
                    OneToMany, Integer, PickleType)
from elixir.options import using_options
from copy import copy

from gaspacho.platform import Software, OS
from gaspacho.valid import valid
from gaspacho.database import commit_database
from gaspacho.config import multi_site, default_serv_lang
from gaspacho.log import trace #, logger

class Site(Entity):
    """Elixir class for Site

    * Site
    name: name of Site
    comment: description of the Site
    cn: common name (or id) of Site
    * Relation
    groups: list of groups in this Site
    templates: list of templates in this Site
    """
    name = Field(UnicodeText)
    comment = Field(UnicodeText)
    cn = Field(UnicodeText)
    groups = OneToMany('Group')
    templates = OneToMany('Template')

@trace
def add_site(name, cn, comment=u''):
    try:
        name = valid(name, 'unicode')
        cn = valid(cn, 'unicode')
        comment = valid(comment, 'unicode')
        if Site.query.filter_by(name=name).first() != None:
            raise Exception('site %s already exist' % name)
        if Site.query.filter_by(cn=cn).first() != None:
            raise Exception('site %s already exist' % name)
        return Site(name=name, cn=cn, comment=comment)
    except Exception, e:
        raise Exception("Error in add_site: " + str(e))

class Group(Entity):
    """Elixir class for Group

    * Group
    name: name of Group
    comment: description of the Group
    computers: list of computer in relation with this Group
    level: to order Group, default is 999
    installpkg: need to install packages for this group
    lang: langage in this group (now just for pkg installation)
    * Manager
    managers: people allow to manage Group (set Choices)
    * Relation
    users: list of users in relation with this Group
    tmpls: use for herited Template
    softs: Group only display Choice for this list of softwares
    oses: Group only display Choice for this list of OS
    site: Group in particular site
    * Inherite
    parent/children: use for herited Group
    * Choice
    choices: related choice
    Note: choice is a relation with group and user. There is only one choice
          available by relation group/user. There is several choices because
          there could have different relation with users.
    """
    name = Field(UnicodeText)
    comment = Field(UnicodeText)
    level = Field(Integer, default=999)
    installpkg = Field(Boolean)
    lang = Field(UnicodeText)
    computers = Field(PickleType)
    users = ManyToMany('User', inverse='groups')
    managers = ManyToMany('User', inverse='gmanage')
    parent = ManyToOne('Group', inverse='children')
    children = OneToMany('Group')
    tmpls = ManyToMany('Template')
    softs = ManyToMany('Software')
    oses = ManyToMany('OS')
    site = ManyToOne('Site')
    choices = OneToMany('Choice')
    using_options(order_by='level')

    def __repr__(self):
        return "{'name': '%s', 'comment': '%s'}"  % (self.name, self.comment)

#name
    @trace
    def get_name(self):
        return self.name

    @trace
    def mod_name(self, name):
        try:
            name = valid(name, 'unicode')
            if get_group(name=name, parent=self.parent):
                raise Exception('Group with name %s already exists' % name)
            self.name = name
        except Exception, e:
            raise Exception("Error in mod_name: " + str(e))

#comment
    @trace
    def get_comment(self):
        return self.comment

    @trace
    def mod_comment(self, comment):
        try:
            comment = valid(comment, 'unicode')
            self.comment = comment
        except Exception, e:
            raise Exception("Error in mod_comment: " + str(e))

#computer
    @trace
    def is_computer(self, name, type):
        for tname, ttype in self.computers:
            if name == tname and ttype == type:
                return True
        return False

    @trace
    def add_computer(self, name, type=u'ip'):
        try:
            type = valid(type, 'computer')
            name = valid(name, 'unicode')
        except Exception, e:
            raise Exception("Error in add_computer: " + str(e))
        if self.is_computer(name, type):
            raise Exception('Computer already exists: %s' % name)
        tcomputers = copy(self.computers)
        tcomputers.append((name, type))
        self.computers = tcomputers

    @trace
    def del_computer(self, name, type=u'ip'):
        try:
            name = valid(name, 'unicode')
            #if name not in self.computers:
            #    raise Exception('Computer %s not already exists' % name)
            tcomputers = copy(self.computers)
            try:
                tcomputers.remove((name, type))
            except ValueError:
                raise Exception('Unable to remove computer (%s, %s)' % \
                            (name, type))
            self.computers = tcomputers
        except Exception, e:
            raise Exception("Error in del_computer: " + str(e))

    @trace
    def get_computers(self):
        return self.computers

#user
    @trace
    def add_user(self, user):
        try:
            valid(user, User)
            if user in self.users:
                raise Exception('Relation between this group and this user already exist')
            self.users.append(user)
        except Exception, e:
            raise Exception("Error in add_user: " + str(e))

    @trace
    def del_user(self, user, delete=True):
        """
        delete relation between group and user with all related choices
        """
        try:
            valid(user, User)
            if user not in self.users:
                raise Exception('No relation between this group and this user')
            if delete:
                #remove all choices for this group
                for choice in self.choices:
                    if choice.user == user:
                        choice.delete()
            self.users.remove(user)
        except Exception, e:
            raise Exception("Error in del_user: " + str(e))

    @trace
    def get_users(self):
        return self.users

#manager
    @trace
    def add_manager(self, user):
        try:
            valid(user, User)
            self.managers.append(user)
        except Exception, e:
            raise Exception("Error in add_manager: " + str(e))

    @trace
    def del_manager(self, user):
        try:
            valid(user, User)
            if user not in self.managers:
                raise Exception('No relation between this group and this manager')
            self.managers.remove(user)
        except Exception, e:
            raise Exception('Error in del_manager: ' + str(e))

    @trace
    def get_managers(self):
        return self.managers
#software
    @trace
    def add_software(self, software):
        try:
            valid(software, Software)
            self.softs.append(software)
        except Exception, e:
            raise Exception("Error in add_software: " + str(e))

    @trace
    def del_software(self, software):
        try:
            valid(software, Software)
            if software not in self.softs:
                raise Exception('No relation between this group and this software')
            self.softs.remove(software)
        except Exception, e:
            raise Exception('Error in del_software: ' + str(e))

    @trace
    def del_softwares(self):
        self.softs = []

    @trace
    def get_softwares(self):
        return self.softs

#os
    @trace
    def add_os(self, os):
        try:
            valid(os, OS)
            self.oses.append(os)
        except Exception, e:
            raise Exception("Error in add_os: " + str(e))

    @trace
    def del_os(self, os):
        try:
            valid(os, OS)
            if os not in self.oses:
                raise Exception('No relation between this group and this OS')
            self.oses.remove(os)
        except Exception, e:
            raise Exception('Error in del_os: ' + str(e))

    @trace
    def del_oses(self):
        self.oses = []

    @trace
    def get_oses(self):
        return self.oses

#template
    @trace
    def add_template(self, template):
        try:
            valid(template, Template)
            self.tmpls.append(template)
        except Exception, e:
            raise Exception("Error in add_template: " + str(e))

    @trace
    def del_template(self, template):
        try:
            valid(template, Template)
            if template not in self.tmpls:
                raise Exception('No relation between this group and this template')
            self.tmpls.remove(template)
        except Exception, e:
            raise Exception("Error in del_template: " + str(e))

    @trace
    def get_templates(self):
        return self.tmpls

#installpkg
    @trace
    def get_installpkg(self):
        return self.installpkg

    @trace
    def mod_installpkg(self, installpkg):
        try:
            installpkg = valid(installpkg, 'boolean')
            self.installpkg = installpkg
        except Exception, e:
            raise Exception("Error in mod_installpkg: " + str(e))

@trace
def add_group(name, parent=None, installpkg=False, comment=u'', site=None):
    try:
        name = valid(name, 'unicode')
        comment = valid(comment, 'unicode')
        installpkg = valid(installpkg, 'boolean')
        if multi_site:
            if site == None:
                raise Exception('Need site in multi_site configuration')
        else:
            if site != None:
                raise Exception('Site must be None if multi_site is desactived')
        if get_group(name=name, parent=parent):
            raise Exception('group %s already exist' % name)
        #FIXME: lang only default_serv_lang
        if parent == None:
            ret = Group(name=name, comment=comment, computers=[],
                        installpkg=installpkg, lang=unicode(default_serv_lang))
        else:
            ret = Group(name=name, parent=parent, comment=comment,
                        computers=[], installpkg=installpkg,
                        lang=unicode(default_serv_lang))
    except Exception, e:
        raise Exception("Error in add_group: " + str(e))
    return ret

@trace
def del_group(group):
    """
    delete a group with all related choices, groups and users
    """
    try:
        valid(group, Group)
        #delete user in group but don't delete it's choices
        for user in group.users:
            group.del_user(user, delete=False)
        for child in group.children:
            del_group(child)
        #delete all choices
        for choice in group.choices:
            choice.delete()
        group.delete()

        #Need it.
        #Without it, some errors append with large heritage
        commit_database()
    except Exception, e:
        raise Exception('Error in del_group: ' + str(e))

@trace
def get_groups(parent=None):
    """
    get_groups
    parent: return children of this group
            None to return root groups
    """
    if parent == None:
        return Group.query.filter_by(parent=None).all()
    else:
        parent = valid(parent, Group)
        return parent.children

@trace
def get_group(name, parent=None):
    """
    get_group
    parent: this group is a child of this group
              None for a root group
    """
    name = valid(name, 'unicode')
    if parent:
        parent = valid(parent, Group)
    return Group.query.filter_by(name=name, parent=parent).first()

@trace
def get_all_groups(grp=None):
    ret = []
    for group in get_groups(parent=grp):
        ret.append(group)
        get = get_all_groups(grp=group)
        if get != None:
            ret.extend(get)
    return ret

class Template(Entity):
    """
    Elixir class for Template

    * Template
    name: name of Template
    comment: description of the Template
    * Manager
    managers: people allow to manage Template (set Choices)
    * Relation
    users: list of users in relation with this Template
    softs: list of software in relation with this Template
    oses: list of OS in relation with this Template
    site: Template in particular site
    * Inherited
    depends: some Groups could depend to Template
    * Choice
    choices: related choice
    Note: choice is a relation with template and user. There is only one choice
          available by relation template/user. There is several choices because
          there could have different relation with users.
    """
    name = Field(UnicodeText)
    comment = Field(UnicodeText)
    managers = ManyToMany('User', inverse='tmanage')
    users = ManyToMany('User', inverse='tmpls')
    softs = ManyToMany('Software')
    oses = ManyToMany('OS')
    site = ManyToOne('Site')
    depends = ManyToMany('Group')
    choices = OneToMany('Choice')

#name
    @trace
    def get_name(self):
        return self.name

    @trace
    def mod_name(self, name):
        try:
            name = valid(name, 'unicode')
            if get_template(name=name):
                raise Exception('Template with name %s already exists' % name)
            self.name = name
        except Exception, e:
            raise Exception("Error in mod_name: " + str(e))

#comment
    @trace
    def get_comment(self):
        return self.comment

    @trace
    def mod_comment(self, comment):
        try:
            comment = valid(comment, 'unicode')
            self.comment = comment
        except Exception, e:
            raise Exception("Error in mod_comment: " + str(e))

#user
    @trace
    def add_user(self, user):
        try:
            valid(user, User)
            self.users.append(user)
        except Exception, e:
            raise Exception("Error in add_user: " + str(e))

    @trace
    def del_user(self, user, delete=True):
        try:
            valid(user, User)
            if user not in self.users:
                raise Exception('No relation between this group and this user')
            #remove all choices for this group
            if delete:
                for choice in self.choices:
                    if choice.user == user:
                        choice.delete()
            self.users.remove(user)
        except Exception, e:
            raise Exception('Error in del_user: ' + str(e))

    @trace
    def get_users(self):
        self.users()

#software
    @trace
    def add_software(self, software):
        try:
            valid(software, Software)
            self.softs.append(software)
        except Exception, e:
            raise Exception("Error in add_software: " + str(e))

    @trace
    def del_software(self, software):
        try:
            valid(software, Software)
            if software not in self.softs:
                raise Exception('No relation between this group and this software')
            self.softs.remove(software)
        except Exception, e:
            raise Exception('Error in del_software: ' + str(e))

    @trace
    def del_softwares(self):
        self.softs = []

    @trace
    def get_softwares(self):
        return self.softs
#os
    @trace
    def add_os(self, os):
        try:
            valid(os, OS)
            self.oses.append(os)
        except Exception, e:
            raise Exception("Error in add_os: " + str(e))

    @trace
    def del_os(self, os):
        try:
            valid(os, OS)
            if os not in self.oses:
                raise Exception('No relation between this group and this OS')
            self.oses.remove(os)
        except Exception, e:
            raise Exception('Error in del_os: ' + str(e))

    @trace
    def del_oses(self):
        self.oses = []

    @trace
    def get_oses(self):
        return self.oses
#manager
    @trace
    def add_manager(self, user):
        try:
            valid(user, User)
            self.managers.append(user)
        except Exception, e:
            raise Exception("Error in add_manager: " + str(e))

    @trace
    def del_manager(self, user):
        try:
            valid(user, User)
            if user not in self.managers:
                raise Exception('No relation between this template and this manager')
            self.managers.remove(user)
        except Exception, e:
            raise Exception('Error in del_manager: ' + str(e))


@trace
def add_template(name, comment=u''):
    try:
        name = valid(name, 'unicode')
        comment = valid(comment, 'unicode')
        if get_template(name):
            raise Exception('Template %s already exists' % name)
        ret = Template(name=name, comment=comment)
    except Exception, e:
        raise Exception("Error in add_template: " + str(e))
    return ret

@trace
def del_template(template):
    """
    delete a template with all related choices
    """
    try:
        valid(template, Template)
        if template.depends != []:
            raise Exception('template already link to a group')
        #delete user in template but don't delete it's choices
        for user in template.users:
            template.del_user(user, delete=False)
        for user in template.managers:
            template.del_manager(user)
        #delete all choices
        for choice in template.choices:
            choice.delete()
        template.delete()

    except Exception, e:
        raise Exception('Error in del_template: ' + str(e))

@trace
def get_templates():
    """
    Return all templates
    """
    return Template.query.all()

@trace
def get_template(name):
    """
    get_template
    """
    name = valid(name, 'unicode')
    return Template.query.filter_by(name=name).first()

class User(Entity):
    """
    Elixir class for User

    name: name of the User
    type: type of the User (user or usergroup)
    comment: description of User
    * Relation
    groups: related Groups
    tmpls: related Templates
    * Manager
    gmanage: User manage those Groups
    tmanage: User manage those Templates
    * Choice
    choices: related Choices
    """
    name = Field(UnicodeText)
    type = Field(UnicodeText)
    comment = Field(UnicodeText)
    groups = ManyToMany('Group')
    tmpls = ManyToMany('Template')
    gmanage = ManyToMany('Group')
    tmanage = ManyToMany('Template')
    choices = ManyToMany('Choice')

#name
    @trace
    def get_name(self):
        return self.name

    @trace
    def mod_name(self, name):
        try:
            name = valid(name, 'unicode')
            if get_user(name=name, type=self.type):
                raise Exception('User with name %s already exists' % name)
            self.name = name
        except Exception, e:
            raise Exception("Error in mod_name: " + str(e))
#comment
    @trace
    def get_comment(self):
        return self.comment

    @trace
    def mod_comment(self, comment):
        try:
            comment = valid(comment, 'unicode')
            self.comment = comment
        except Exception, e:
            raise Exception("Error in mod_comment: " + str(e))

    def __repr__(self):
        return '{name: "%s", type: "%s", comment: "%s"}' % (self.name, self.type,
                self.comment)

@trace
def add_user(name, type=u'user', comment=u''):
    try:
        name = valid(name, 'unicode')
        type = valid(type, 'user')
        comment = valid(comment, 'unicode')
    except Exception, e:
        raise Exception("Error in add_user: " + str(e))
    if User.query.filter_by(name=name, type=type).first() != None:
        raise Exception('User already exists')
    try:
        return User(name=name, type=type, comment=comment)
    except Exception, e:
        raise Exception("Error in add_user: " + str(e))

@trace
def del_user(user):
    if not valid(user, User):
        raise Exception("Not a user in del_user")
    if user.groups != []:
        raise Exception('user already used in a group')
    if user.gmanage != [] or user.tmanage != []:
        raise Exception('user is manager for a group or template')
    try:
        user.delete()
    except Exception, e:
        raise Exception("Error in del_user: " + str(e))

@trace
def get_users():
    try:
        return User.query.all()
    except Exception, e:
        raise Exception('Error in get_users: ' + str(e))

@trace
def get_user(name, type=u'user'):
    try:
        name = valid(name, 'unicode')
        type = valid(type, 'user')
        return User.query.filter_by(name=name, type=type).first()
    except Exception, e:
        raise Exception('Error in get_user: ' + str(e))

# vim: ts=4 sw=4 expandtab
