# Copyright (C) 2010-2012 Team Gaspacho (see README for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from elixir import Entity, Field, UnicodeText, ManyToMany

from gaspacho.category import get_category, add_category
from gaspacho.platform import (add_platform, add_path, add_os, add_software,
        get_os, get_software, get_platform, get_path)
from gaspacho.config import database_version
from gaspacho import get_confuser, get_confcomputer, get_confcontext
from gaspacho.log import trace #, logger

class Import(Entity):
    """
    Elixir class for Import
    This table all information about importation file (name, version, ...)
    """
    name = Field(UnicodeText)
    version = Field(UnicodeText)
    variables = ManyToMany('Variable')
    platforms = ManyToMany('Platform')

    @trace
    def add_rule(self, variable, platform):
        self.variables.append(variable)
        self.platforms.append(platform)

@trace
def load(datas):
    desc = GaspDescr(datas)
    desc.process()

class GaspDescr:
    @trace
    def __init__(self, datas):
        """{"name"      : "fedora-15/libreoffice-3.3",
         "version"   : "0.1",
         "database"  : "0.91",
         "os"        : {
          "name"      : "fedora",
          "version"   : "15"
         },
         "software"  : {
           "name"     : "libreoffice",
           "version"  : "3.3"
         },
         "packages"  : [{
            "name"    : "libreoffice"
         }, {
            "name"    : "libreoffice-i18n-fr",
            "lang"    :"fr"
         }],
         "categories" : ...
        """
        self.datas = datas
        self._validate()
        # setting the datas available by attributes
        for key, value in datas.items():
            if key == u'categories':
                continue
            if type(key) == unicode:
                setattr(self, key, value)
            elif type(key) == dict:
                # defining a sub object that stores the os and software datas
                # self.software = object()
                obj = setattr(self, key, object())
                # self.software.name = "fedora"
                for k, v in value:
                    setattr(getattr(self, obj), k, v)
        self.categories = []
        for category in self.datas['categories']:
            self.categories.append(GaspCategory(category))
        
    @trace
    def _validate(self):
        #check gaspfile version
        database = self.datas.get('database', None)
        if database == None:
            raise Exception('no database version, this not a Gasp file')
        if self.datas['database'] != database_version:
            raise Exception('database version need to be %s not %s' % \
                            (database_version, self.datas['database']))

        keys_descr = set(['name', 'version', 'database', 'os', 'software',
                                    'categories', 'packages'])
        keys = set(['name', 'version'])
        try:
            type(self.datas) == dict
            assert 'categories' in self.datas
            set(self.datas.keys()) == keys
            type(self.datas['os']) == dict
            type(self.datas['software']) == dict
            set(self.datas['os']) == keys
            set(self.datas['software']) == keys
            type(self.datas['packages']) == dict
        except Exception, e:
             raise TypeError("incorrect data values in datas: %s" % e)

    @trace
    def _get_os_version(self):
        os = get_os(self.os['name'])
        if os == None:
            os = add_os(name=self.os['name'])

        osv = os.get_version(self.os['version'])
        if osv == None:
            osv = os.add_version(self.os['version'])

        return osv

    @trace
    def _get_software_version(self):
        software = get_software(self.software['name'])
        if software == None:
            software = add_software(name=self.software['name'])

        swv = software.get_version(self.software['version'])
        if swv == None:
            swv = software.add_version(self.software['version'])

        return swv

    @trace
    def process(self):
        #check if gaspfile already imported
        import_obj = Import.query.filter_by(name=self.name).first()
        if import_obj != None:
            if import_obj.version >= self.version:
                raise Exception("This gasp's file is already imported")
            else:
                #remove_rules(timport_obj)
                #FIXME:
                print "not yet implemented"
                sys.exit(1)
        import_obj = Import(name=self.name, version=self.version)

        osv = self._get_os_version()
        swv = self._get_software_version()
        packages = self.datas['packages']
        for category in self.categories:
            category.process(osv, swv, packages, import_obj)


class GaspCategory:
    @trace
    def __init__(self, datas):
        """"categories": [{
              "names"      : [{
                   "lang"      : "fr",
                   "label"     : "Reseau"
              }],
              "tags"      : ...
        """
        self.datas = datas
        self._validate()
        self.names = self.datas['names']
        #for name in names:
        #    for k, v in name.items():
        #        self.names.append((k, v))
        self.tags = []
        for tag in self.datas['tags']:
            self.tags.append(GaspTag(tag))
    
    @trace
    def _validate(self):
        keys_name = ['lang', 'label']
        keys_needed = ["names", "tags"]
        try:
            type(self.datas) == dict, "datas not a dict"
            assert set(keys_needed) == set(self.datas.keys()), \
                    'must have attributes %s (%s)' % (', '.join(keys_needed),
                    ', '.join(self.datas.keys()))
            for name in self.datas['names']:
                type(name) == dict, 'name must be a dict'
                assert set(keys_name) == set(name.keys()), \
                        'names must have attributes %s (%s)' % \
                        (', '.join(keys_name),
                    ', '.join(name.keys()))

        except Exception, e:
            raise TypeError("incorrect datas structure: %s" % e)

    @trace
    def process(self, osv, swv, packages, import_obj):
        #FIXME: lang par defaut!
        category = get_category(name=self.names[0][u'label'], lang=self.names[0][u'lang'])
        if category == None:
            category = add_category()
            for catname in self.names:
                category.add_translation(lang=catname[u'lang'], name=catname[u'label'])
        for tag in self.tags:
            tag.process(category, osv, swv, packages, import_obj)

class GaspTag:
    @trace
    def __init__(self, datas):
        """ "tags"      : [{
               "name"      : [{
                "lang"      : "fr",
                "label"     : "Configuration du proxy"
             }],              
             "rules" : ...
        """
        self.datas = datas
        self._validate()
        self.names = self.datas['names']
        self.rules = []
        for rule in self.datas['rules']:
            self.rules.append(GaspRule(rule))
    
    @trace
    def _validate(self):
        keys = ['lang', 'label']
        keys_needed = ["names", "rules"]
        try:
            assert type(self.datas) == dict, 'datas must be a dict'
            assert set(keys_needed) == set(self.datas.keys()), \
                        'must have attributes %s (%s)'% (', '.join(keys_needed),
                        ', '.join(self.datas.keys()))
            assert type(self.datas['names']) == list, 'names must be a list'
            for name in self.datas['names']:
                assert type(name) == dict, 'names must be a list of dict'
                assert set(keys) == set(name.keys()), 'names must have attributes %s' % ', '.join(keys)
        except Exception, e:
            raise TypeError("GaspTag: %s" % e)

    @trace
    def process(self, category, osv, swv, packages, import_obj):
        #FIXME: lang par defaut!
        tag = category.get_tag(name=self.names[0][u'label'], lang=self.names[0][u'lang'])
        if tag == None:
            tag = category.add_tag()
            for tagname in self.names:
                tag.add_translation(lang=tagname[u'lang'], name=tagname[u'label'])

        for rule in self.rules:
            rule.process(tag, osv, swv, packages, import_obj)

class GaspRule:
    @trace
    def __init__(self, datas):
        """ "rules"     : [{
              "names"      : [{
               "lang"      : "fr",
               "label"     : "Activer le proxy"
              }],
              "type":     : 'boolean',
              "variables" : ...
             }]
        """
        self.datas = datas
        self._validate()
        self.names = []
        self.defaultvalue = None
        self.defaultstate = u'free'
        self.options = {}
        self.display = True
        self.comments = {}
        for key, value in self.datas.items():
            if key == u'variables':
                continue
            if type(key) == unicode:
                setattr(self, key, value)
            elif type(key) == dict:
                # defining a sub object that stores the os and software datas
                # self.software = object()
                obj = setattr(self, key, object())
                # self.software.name = "fedora"
                for k, v in value:
                    setattr(getattr(self, obj), k, v)

        self.variables = []
        for variable in self.datas['variables']:
            self.variables.append(GaspVariable(variable))
    
    @trace
    def _validate(self):
        keys_name = ['lang', 'label']
        keys_needed = ["names", "type", "variables"]
        keys_optional = ['defaultstate', 'display', 'defaultvalue', 'options',
                         'comments']
        keys_optional.extend(keys_needed)
        try:
            assert type(self.datas) == dict, 'datas must be a dict (%s)' % \
                        type(self.datas)
            assert set(keys_needed) - set(self.datas.keys()) == set([]), \
                        'must have attributes %s (%s)' % (
                        ', '.join(keys_needed), ', '.join(self.datas.keys()))
            assert set(self.datas.keys()) - set(keys_optional) == set([]), \
                        'allowed attributes %s (%s)' % (
                        ', '.join(keys_optional), ', '.join(self.datas.keys()))
            for name in self.datas['names']:
                type(name) == dict
                assert set(keys_name) == set(name.keys())
        except Exception, e:
            raise TypeError("GaspRule: %s" % e)

    @trace
    def process(self, tag, osv, swv, packages, import_obj):
        #FIXME: should be default
        rule = tag.get_rule(name=self.names[0][u'label'],
                            lang=self.names[0][u'lang'])
        if rule == None:
            rule = tag.add_rule(type=self.type, defaultvalue=self.defaultvalue,
                    defaultstate=self.defaultstate, display=self.display,
                    options=self.options)
            for rulename in self.names:
                rule.add_translation(name=rulename[u'label'],
                            lang=rulename[u'lang'])
            for rulecomment in self.comments:
                rule.add_comment(name=rulecomment[u'label'],
                            lang=rulecomment[u'lang'])
        else:
            if rule.options != self.options or \
                    rule.get_defaultvalue() != self.defaultvalue or \
                    rule.type != self.type:
                raise Exception('Found an old rule but options, type or defaultvalue are different\nfor rule %s'
                                    % rule.name)

        for variable in self.variables:
            variable.process(rule, osv, swv, packages, import_obj)

class GaspVariable:
    @trace
    def __init__(self, datas):
        """ "variables" : [{
               "value_on"  : "2",
               "value_off" : "SUPPR",
               "type"      : "integer",
               "name"      : "ooInetProxyType",
               "extension" : "xcu3",
               "info"      : "Inet/Settings",
               "path"      : "$HOME/.libreoffice/3/user/registrymodifications.xcu"
            }]
        """
        self.datas = datas
        self._validate()
        self.value_on = None
        self.info = None
        self.comment = ''
        for key, value in self.datas.items():
            if key != 'level':
                setattr(self, key, value)
        if self.datas['level'] == 'user':
            self.conflevel = get_confuser()
        elif self.datas['level'] == 'context':
            self.conflevel = get_confcontext()
        else:
            self.conflevel = get_confcomputer()

    @trace
    def _validate(self):
        keys_needed = ["name", "level", "type", "extension", "value_off",
                       "path"]
        keys_optional = ["value_on", "info", "comment"]
        keys_level = ['user', 'computer', 'context']
        keys_optional.extend(keys_needed)
        try:
            assert type(self.datas) == dict, 'datas must be a dict'
            assert set(keys_needed) - set(self.datas.keys()) == set([]), \
                        'must have attributes %s (%s)' % (
                        ', '.join(keys_needed), ', '.join(self.datas.keys()))
            assert set(self.datas.keys()) - set(keys_optional) == set([]), \
                        'allowed attributes %s (%s)' % (
                        ', '.join(keys_optional), ', '.join(self.datas.keys()))
            assert self.datas['level'] in keys_level, \
                        'level must be %s (%s)' % (' or '.join(keys_level),
                        ', '.join(self.datas['level']))
        except Exception, e:
            name = self.datas.get('name', '')
            raise TypeError("GaspVariable: %s (%s)" % (e, name))

    @trace
    def process(self, rule, osv, swv, packages, import_obj):
        variable = rule.get_variable(name=self.name,
                conflevel=self.conflevel)
        if variable == None:
            variable = rule.add_variable(name=self.name,
                    conflevel=self.conflevel, type=self.type, 
                    value_on=self.value_on, 
                    value_off=self.value_off)
        else:
            if variable.type != self.type or \
                    variable.value_on != self.value_on or \
                    variable.value_off != self.value_off:
                raise Exception('Found an old variable but type, value_on or value_off are different\nfor variable %s'% self.name)

        path = get_path(name=self.path, info=self.info, extension=self.extension)
        if path == None:
            path = add_path(name=self.path, info=self.info, extension=self.extension,
                            comment=self.comment)
        platform = get_platform(osversion=osv, softwareversion=swv, path=path)
        if platform == None:
            platform = add_platform(osversion=osv, softwareversion=swv, path=path, packages=packages)
        else:
            if platform.packages != packages:
                raise Exception('Different package name: %s - %s' % (
                        platform.packages, packages))
        variable.add_platform(platform)
        import_obj.add_rule(variable, platform)

# vim: ts=4 sw=4 expandtab
