# Copyright (C) 2010-2012 Team Gaspacho (see README for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
Rule: Information display in user interface with default value
Variable: variable for speficied file, register, ...

Rule
  |
Variable
"""
from elixir import *
from copy import copy

from gaspacho.valid import valid, normalize_type
from gaspacho.platform import Platform
from gaspacho.config import ordered_lang, default_separator
from gaspacho.choice import Choice
from gaspacho.group import User, Group, Template
from gaspacho.log import trace #, logger

class Rule(Entity):
    """
    Elixir class for Rule

    name: label of the Rule
    type: type of the name (see below)
    defaultvalue: default value
    options: options for this rule (see below)
    comment: description of this Rule
    vars: related Variable
    tag: related Tag
    choices: related Choices
    display: if display is False, this rule is not display in interface
    defaultstate: default state must be "on", "off" or "free" (by default)

    Type:
    unicode: string
    boolean: boolean
    integer: integer
    ip: IP
    list: list of string
    enum: list of predeterminate choice
    multi: several field for a rule

    Options: 
    enum: list of available choice list (ie: [['1', 'first], '2', 'second']])
    integer: specify min and max value(ie: {'min': 0, 'max': 23})
    list: specify the separator character (ie: {'separator': ';'}), default: -
    multi: description of all values, ie:
           {'values': [{'name': 'PROXY_HOST', 'type': 'unicode'},
                       {'name': 'PROXY_PORT', 'type': 'integer',
                        'options': {'min': 0, 'max': 65535}}],
           'separator': ':'}
    """
    name = Field(PickleType)
    type = Field(UnicodeText)
    defaultvalue = Field(UnicodeText)
    defaultstate = Field(UnicodeText)
    display = Field(Boolean)
    options = Field(PickleType)
    comment = Field(PickleType)
    vars = OneToMany('Variable')
    tag = ManyToOne('Tag')
    choices = ManyToMany('Choice')
    #just for performance, don't touch this
    conflevels = ManyToMany('ConfLevel')
    oses = ManyToMany('OS')
    softs = ManyToMany('Software')

#defaultvalue
    @trace
    def get_defaultvalue(self):
        defaultvalue = self.defaultvalue
        if (self.type == u'list' or self.type == u'multi') \
                                and defaultvalue != None:
            separator = self.options.get('separator',
                                         default_separator)
            defaultvalue = defaultvalue.split(separator)
        return defaultvalue

    @trace
    def set_defaultvalue(self, defaultvalue):
        if self.type == u'enum':
            if defaultvalue == None:
                raise Exception('if type enum, defaultvalue must not be None')
            defaultvalue = valid(defaultvalue, 'unicode')

            #valid if defaultvalue is in options
            for option in self.options:
                if defaultvalue == option[0]:
                    is_ok=True
                    break
            if not is_ok:
                raise Exception('Defaultvalue must be in options')

        elif self.type == u'list' or self.type == u'multi':
            if defaultvalue == None:
                raise Exception('if type is %s, defaultvalue must not be None'\
                            % self.type)
            valid(defaultvalue, self.type)
            if self.type == u'multi' and len(defaultvalue) != \
                            len(self.options['values']):
                raise Exception('defaultvalue in multi must have same len')
            separator = self.options.get('separator', default_separator)
            for val in defaultvalue:
                if separator in val:
                    raise Exception('Separator must not be in defaultvalue')
            defaultvalue = valid(separator.join(defaultvalue), 'unicode')

        elif self.type == u'boolean':
            if defaultvalue != None:
                raise Exception('if type is boolean, defaultvalue must be None')
        else:
            if defaultvalue == None:
                raise Exception('if type is not boolean, defaultvalue must not be None')
            valid(defaultvalue, self.type)
            defaultvalue = valid(defaultvalue, 'unicode')

        self.defaultvalue = defaultvalue

#variable
    @trace
    def add_variable(self, name, type, value_off, conflevel, value_on=None,
                            comment=u''):
        try:
            name = valid(name, 'unicode')
            type = valid(type, 'type')
            if self.get_variable(name=name, conflevel=conflevel):
                raise Exception('Variable with name %s and conflevel %s already exists' % (name, conflevel))
            #if type of the rule is boolean, value_on must not be None
            #if type of the rule is not boolean, value_on could be None or not
            #type of variable is not important
            if self.type == u'boolean':
                if value_on == None:
                    raise Exception('value_on must not be None for %s'%name)
            if value_on != None:
                value_on = valid(value_on, 'unicode')
            #value_off should be in a normalize_type or in type "type"
            #but must be unicode in the database
            if value_off not in normalize_type:
                valid(value_off, type)
            value_off = valid(value_off, 'unicode')
            comment = valid(comment, 'unicode')

        except TypeError, e:
            raise Exception("Error in add_variable: " + str(e))
        try:
            variable = Variable(name=name, type=type, rule=self, 
                    value_on=value_on, value_off=value_off, conflevel=conflevel,
                    comment=comment)
            if conflevel not in self.conflevels:
                self.conflevels.append(conflevel)
        except Exception, e:
            raise Exception("Error in add_variable: " + str(e))

        return variable

    @trace
    def get_variable(self, name, conflevel):
        try:
            name = valid(name, 'unicode')
            return Variable.query.filter_by(name=name, conflevel=conflevel,
                            rule=self).first()
        except Exception, e:
            raise Exception("Error in get_variable: "+ str(e))

    @trace
    def get_variables(self):
        try:
            return self.vars
        except Exception, e:
            raise Exception("Error in get_variables: "+ str(e))

#translation/name
    @trace
    def add_translation(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            if self.tag.get_rule(name, lang):
                raise Exception('Rule translation %s in %s already exists' %
                            (name, lang))
            tname = copy(self.name)
            tname[lang] = name
            self.name = tname
        except Exception, e:
            raise Exception("Error in add_translation: " + str(e))

    @trace
    def get_name(self, lang):
        try:
            lang = valid(lang, 'unicode')
            if self.name.has_key(lang):
                return self.name[lang]
            for tlang in ordered_lang:
                if self.name.has_key(tlang):
                    return self.name[tlang]
            return u'No translation in %s'%lang
        except Exception, e:
            raise Exception("Error in get_name: " + str(e))

#comment
    @trace
    def add_comment(self, name, lang):
        try:
            name = valid(name, 'unicode')
            lang = valid(lang, 'unicode')
            if self.comment == None:
                self.comment = {}
            tcomment = copy(self.comment)
            tcomment[lang] = name
            self.comment = tcomment
        except Exception, e:
            raise Exception("Error in add_comment: " + str(e))

    @trace
    def get_comment(self, lang):
        try:
            lang = valid(lang, 'unicode')
            if self.comment == None:
                return u''
            if self.comment.has_key(lang):
                return self.comment[lang]
            for tlang in ordered_lang:
                if self.comment.has_key(tlang):
                    return self.comment[tlang]
            return u'No translation in %s'%lang
        except Exception, e:
            raise Exception("Error in get_comment: " + str(e))

#choice
    @trace
    def get_choice(self, group, user=None):
        tree_group = get_tree_group(group, user)
        non_herited = True
        if not isinstance(group, Template) and not isinstance(group, Group):
            raise Exception('Group in get_choice must be Template or Group')
        if user != None:
            valid(user, User)
        choices = _get_choices([self], Choice.query.filter_by(rule=self).all(),
                                group, user)
        return choices.get(self, (None, None))

    @trace
    def set_choice(self, group, state, value=None, user=None, platform=None):
        """
        set_choice add a choice to the database.
        Need group or template and, optionaly, user, platform, state and value.
        """
        try:
            state = valid(state, 'state')
            if user != None:
                if not isinstance(user, User):
                    raise TypeError('Error in set_choice: not a valid User')
                from gaspacho.conflevel import get_confuser, get_confcontext
                confuser = get_confuser()
                confcontext = get_confcontext()
                if confuser not in self.conflevels and confcontext not in \
                            self.conflevels:
                    raise Exception("Rule don't have conflevel variable")
            if platform != None:
                platform = valid(platform, Platform)
            if isinstance(group, Template):
                choice = Choice.query.filter_by(rule=self, template=group,
                        user=user, platform=platform).first()
            elif isinstance(group, Group):
                choice = Choice.query.filter_by(rule=self, group=group, user=user, \
                        platform=platform).first()
            else:
                raise TypeError('Error in set_choice: group need to be a Group or a Template for rule %s' % self.name)
            if group.softs != [] and self.softs != []:
                if list(set(group.softs) & set(self.softs)) == []:
                    raise Exception('Group link to software not listed in this rule')
        except TypeError, e:
            raise Exception("Error in set_choice: ", str(e))


        if choice == None:
            #if no choice is already set to this combination
            if isinstance(group, Template):
                choice = Choice(rule=self, template=group, user=user,
                                platform=platform)
            else:
                choice = Choice(rule=self, group=group, user=user, 
                                platform=platform)

        if value == None and self.type != 'boolean':
            raise Exception("Need value for this rule's type")
        elif value != None and self.type == 'boolean':
            raise Exception("Must no value for this rule's type")
        choice.set_state(state)
        choice.set_value(value)
        return choice

    @trace
    def del_choice(self, group, user=None, platform=None):
        """
        delete a specified choice by rule/group/user/platform
        """
        if user != None:
            if not isinstance(user, User):
                raise TypeError('Error in del_choice: not a valid User')
        if platform != None:
           valid(platform, Platform)

        if isinstance(group, Template):
            choice = Choice.query.filter_by(rule=self, template=group, \
                    user=user, platform=platform).first()
        elif isinstance(group, Group):
            choice = Choice.query.filter_by(rule=self, group=group, user=user, \
                    platform=platform).first()
        else:
            raise TypeError('Error in del_choice: group need to be a Group or a Template for rule %s' % self)
        if choice != None:
            choice.delete()

#platform
    @trace
    def get_platforms(self):
        try:
            ret = []
            for variable in self.vars:
                ret.extend(variable.plats)
            #return list of uniq platforms
            return list(set(ret))
        except Exception, e:
            raise Exception('Error in get_platforms: ' + str(e))

class Variable(Entity):
    """
    Elixir class for Variable

    name: name of the Variable for key
    type: type of the Variable
    value_on: value of the key if Rule state is "on"
    value_off: value of the key if Rule state is "off"
    comment: description of this variable
    rule: related Rule
    plats: related Platforms
    """
    name = Field(UnicodeText)
    type = Field(UnicodeText)
    value_on = Field(UnicodeText)
    value_off = Field(UnicodeText)
    comment = Field(UnicodeText)
    rule = ManyToOne('Rule')
    plats = ManyToMany('Platform')
    conflevel = ManyToOne('ConfLevel')
    imports = ManyToMany('Import')

#comment
    @trace
    def get_comment(self):
        return self.comment

    @trace
    def mod_comment(self, comment):
        try:
            comment = valid(comment, 'unicode')
            self.comment = comment
        except Exception, e:
            raise Exception("Error in mod_comment: " + str(e))


#platform
    @trace
    def add_platform(self, platform):
        try:
            platform = valid(platform, Platform)
            self.plats.append(platform)
            #for performance
            os = platform.osversion.os
            software = platform.softwareversion.software
            if not os in self.rule.oses:
                self.rule.oses.append(os)
            if not software in self.rule.softs:
                self.rule.softs.append(software)
        except Exception, e:
            raise Exception("Error in add_platform: " + str(e))

    @trace
    def get_platforms(self):
        return self.plats

    #def del_platform(self, platform):
    #    #FIXME
    #    #remove os and software in rules but only if not in an other platform
    #    pass

    def __repr__(self):
        return '["%s", {"type": "%s", "value_on": "%s", "value_off": "%s", \
                    "comment": "%s"}]' % (self.name, self.type, self.value_on,
                    self.value_off, self.comment)

@trace
def _load_choices(choices, user_level, tree_group, force_level=None):
    from gaspacho.conflevel import get_confuser, get_confcontext, \
                                   get_confcomputer
    confuser = get_confuser()
    confcontext = get_confcontext()
    confcomputer = get_confcomputer()
    tchoices = {}
    for choice in choices:
        #if user level : only confuser et confcontext
        if (force_level != None and force_level in choice.rule.conflevels) or \
                (force_level == None and (user_level == True and 
                    (confuser in choice.rule.conflevels or \
                    confcontext in choice.rule.conflevels) or \
                    user_level == False)):
            #test if choice is group or template
            if choice.group:
                grp = choice.group
            else:
                grp = choice.template
            #if grp is in tree_group
            if (grp, choice.user) in tree_group:
                if not tchoices.has_key((grp, choice.user)):
                    tchoices[(grp, choice.user)] = {}
                tchoices[(grp, choice.user)][choice.rule] = choice
    return tchoices

class DefaultChoice:
    @trace
    def __init__(self, type, state, value):
        self.type = type
        self.state = state
        self.value = value

    @trace
    def get_value(self):
        return valid(self.value, self.type)

    @trace
    def get_state(self):
        return self.state

@trace
def _get_choices(rules, choices, group, user, platforms=None, force_level=None):
    from gaspacho.conflevel import get_confuser, get_confcontext, \
                                   get_confcomputer
    confuser = get_confuser()
    confcontext = get_confcontext()
    confcomputer = get_confcomputer()

    user_level = not user == None

    #load all choices for better performance
    tree_group = get_tree_group(group, user)
    tchoices = _load_choices(choices, user_level, tree_group,
                             force_level=force_level)
    #parse loaded choices to get choice
    choices = {}
    non_herited = True
    for group_user in tree_group:
        if tchoices.has_key(group_user):
            for rule, choice in tchoices[group_user].items():
                if platforms:
                    if choice.platform and choice.platform not in platforms:
                        continue
                    is_platform = False
                    set_platforms = set(platforms)
                    for variable in rule.get_variables():
                        if set_platforms & set(variable.get_platforms()):
                            is_platform = True
                            break
                    if not is_platform:
                        continue
                if non_herited:
                    choices[rule] = (choice, None)
                else:
                    if choices.has_key(rule):
                        #only if no herited choice
                        if not choices[rule][1]:
                            choices[rule] = (choices[rule][0], choice)
                    else:
                        #if no rule, choice is not set
                        choices[rule] = (None, choice)
        non_herited = False

    #get default value
    for rule in rules:
        if rule.defaultstate != u'free':
            choice = DefaultChoice(rule.type, rule.defaultstate,
                                   rule.get_defaultvalue())
            if not choices.has_key(rule):
                choices[rule] = (None, choice)
            elif not choices[rule][1]:
                choices[rule] = (choices[rule][0], choice)

    return choices

@trace
def get_choices(group, user=None, platforms=None, force_level=None):
    """
    Return the choice and herited choice for a rule/group|template/user
    """
    if user and user not in group.users:
        raise Exception("User %s not in group %s" % (user.name, group.name))
            
    return _get_choices(Rule.query.all(), Choice.query.all(), group, user,
            platforms=platforms, force_level=force_level)

@trace
def get_tree_group(grp, user=None):
    """
    Build tree dependencies for group (Group + User)
    Examples of inherit:
    default (None, user1, user2)
     '--group1 (None, user1, user2)
         |--group2 (None, user1, user2)
         |
        template1 (None, user1, user2)
    - example 1: grp=group2, user=user1:
      build [(group2, user1), (group2, None), (group1, user1), (group1, None) 
      (template1, user1), (tempate1, None), (default, user1), (default, None)]
    - example 2: grp=group2:
      [(group2, None), (group1, None), (template1, None), (default, None)]
    - example 3: grp=default:
      [(default, None)]
    """
    ret = []
    if user != None:
        ret.append((grp, user))
    ret.append((grp, None))
    #if not template
    if isinstance(grp, Group):
        for template in grp.tmpls:
            if user != None:
                ret.append((template, user))
            ret.append((template, None))
        grp2=grp.parent
        if grp2 != None:
            ret.extend(get_tree_group(grp2, user))
    return ret


# vim: ts=4 sw=4 expandtab
