# Copyright (C) 2010-2011 Team Gaspacho (see README for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from os import makedirs, unlink
from os.path import abspath, dirname, isdir, isfile, join
from py.test import raises
from sys import exit

here = dirname(abspath(__file__))

dir_name = join(here, 'datas')
file_name = join(dir_name, 'group.sqlite')
database = 'sqlite:///%s' % file_name

if not isdir(dir_name):
    makedirs(dir_name)

if isfile(file_name):
    unlink(file_name)

from gaspacho import *
from gaspacho.category import add_category, del_category
initialize_database(create=True, database=database)
from gaspacho.conflevel import get_confcomputer, get_confuser

##############################################################################
# conflevel                                                                  #
##############################################################################

def test_confcomputer():
    confcomputer = get_confcomputer()
    assert confcomputer.get_name() == u'confcomputer'

def test_confuser():
    confuser = get_confuser()
    assert confuser.get_name() == u'confuser'

##############################################################################
# category                                                                   #
##############################################################################

def test_add_category():
    category = add_category()
    category.add_translation(lang='en', name='example')

def test_get_category():
    category = get_category('example', 'en')
    assert category != None

def test_get_unknown_category():
    category = get_category('unknown', 'en')
    assert category == None

def test_get_categories():
    category = get_category('example', 'en')
    assert [category] == get_categories()

def test_translation_category():
    category = get_category('example', 'en')
    category.add_translation(lang='fr', name='exemple')
    category.add_translation(lang='it', name='esempio')
    category2 = get_category('exemple', 'fr')
    assert category == category2

def test_get_name_category():
    category = get_category('example', 'en')
    assert category.get_name('en') == u'example'
    assert category.get_name('fr') == u'exemple'
    assert category.get_name('it') == u'esempio'

def test_delete_category():
    category = get_category('example', 'en')
    assert [category] == get_categories()
    category1 = add_category()
    category1.add_translation('del_category', 'fr')
    assert set([category, category1]) == set(get_categories())
    del_category(category1)
    assert [category] == get_categories()

def test_category_without_translation():
    category = get_category('example', 'en')
    category1 = add_category()
    assert category1.get_name('en') == None
    assert set([category, category1]) == set(get_categories())

def test_exists_category():
    category = add_category()
    raises(Exception, "category.add_translation(lang='en', name='example')")

def test_rewrite_translation():
    category = get_category('example', 'en')
    assert category.get_name('fr') == u'exemple'
    raises(Exception, "category.add_translation(lang='fr', name='example')")

def test_mod_translation_category():
    category = get_category('example', 'en')
    category.mod_translation(lang='fr', name='example')
    assert category.get_name('fr') == u'example'
    raises(Exception, "category.mod_translation(lang='de', 'beispiel')")
    raises(Exception, "category.mod_translation(lang='fr', 'example')")

#comment
def test_create_comment_category():
    category = add_category(comment='add comment')
    category.add_translation('comment_fr', 'fr')
    assert category.get_comment() == 'add comment'

def test_mod_comment_category():
    category1 = get_category('example', 'en')
    assert category1.get_comment() == u''
    category1.mod_comment('new comment')
    assert category1.get_comment() == 'new comment'
    category2 = get_category('comment_fr', 'fr')
    assert category2.get_comment() == 'add comment'
    category2.mod_comment('new comment')
    assert category2.get_comment() == 'new comment'


##############################################################################
# tag                                                                        #
##############################################################################

def test_add_tag():
    category = get_category('example', 'en')
    tag = category.add_tag()
    tag.add_translation(lang='en', name='tag1')

def test_add_tag_category_no_translation():
    category = get_category('category1', 'en')
    raises(Exception, "category.add_tag()")

def test_get_tag():
    category = get_category('example', 'en')
    assert category.get_tag('tag1', 'en') != None

def test_get_unknown_tag():
    category = get_category('example', 'en')
    assert category.get_tag('unknowntag', 'en') == None

def test_get_tags():
    category = get_category('example', 'en')
    tag1 = category.get_tag('tag1', 'en')
    assert [tag1] == category.get_tags()

def test_translation_tag():
    category = get_category('example', 'en')
    tag1 = category.get_tag('tag1', 'en')
    tag1.add_translation(lang='fr', name='tag1_fr')
    tag1.add_translation(lang='it', name='tag1_it')
    tag2 = category.get_tag('tag1_fr', 'fr')
    tag1 = category.get_tag('tag1_fr', 'fr')
    assert tag1 == tag2

def test_get_name_tag():
    category = get_category('example', 'en')
    tag1 = category.get_tag('tag1', 'en')
    assert tag1.get_name('fr') == u'tag1_fr'
    assert tag1.get_name('it') == u'tag1_it'

def test_tag_without_translation():
    category = get_category('example', 'en')
    tag1 = category.get_tag('tag1', 'en')
    tag2 = category.add_tag()
    assert tag2.get_name('en') == None
    assert set([tag1, tag2]) == set(category.get_tags())

def test_exists_tag():
    category = add_category()
    tag1 = category.get_tag('tag1', 'en')
    raises(Exception, "tag1.add_translation(lang='en', name='tag1_en')")

def test_exists_tag_other_category():
    category1 = add_category()
    category1.add_translation('example1', 'en')
    tag1 = category1.add_tag()
    tag1.add_translation('tag1', 'en')
    category = get_category('example', 'en')
    tag2 = category.get_tag('tag1', 'en')
    assert tag1.get_name('en') == tag2.get_name('en')

def test_rewrite_translation_tag():
    category = get_category('example', 'en')
    tag1 = category.get_tag('tag1', 'en')
    assert tag1.get_name('fr') == u'tag1_fr'
    raises(Exception, "category.add_translation(lang='fr', name='tag1_fr_new')")

def test_mod_translation_tag():
    category = get_category('example', 'en')
    tag1 = category.get_tag('tag1', 'en')
    tag1.mod_translation(lang='fr', name='tag1_fr_new')
    assert tag1.get_name('fr') == u'tag1_fr_new'
    raises(Exception, "tag1.mod_translation(lang='de', 'tag1_de')")
    raises(Exception, "tag1.mod_translation(lang='fr', 'tag1_it')")

def test_delete_category_with_tags():
    """
    try to delete a category with associated tag
    """
    category = get_category('example', 'en')
    raises(Exception, "del_category(category)")

#comment
def test_create_comment_tag():
    category = get_category('comment_fr', 'fr')
    tag1 = category.add_tag(comment='add comment')
    tag1.add_translation('tag comment', 'en')
    assert tag1.get_comment() == 'add comment'

def test_mod_comment_tag():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    assert tag1.get_comment() == u''
    tag1.mod_comment('new comment')
    assert tag1.get_comment() == 'new comment'
    tag2 = category1.add_tag('new comment')
    tag2.add_translation('tag2', 'en')
    assert tag2.get_comment() == 'new comment'
    tag2.mod_comment('new comment')
    assert tag2.get_comment() == 'new comment'

def test_delete_tag():
    category1 = get_category('example1', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    assert [tag1] == category1.get_tags()
    category1.del_tag(tag1)
    assert [] == category1.get_tags()

##############################################################################
# rule                                                                       #
##############################################################################

def test_add_rule():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule1 = tag1.add_rule(type='boolean')
    rule1.add_translation(lang='en', name='example 1')
    rule2 = tag1.add_rule(type='boolean')
    rule2.add_translation(lang='en', name='example 2')

def test_get_rule():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule1 = tag1.get_rule("example 1", 'en')
    assert rule1 != None
    rule2 = tag1.get_rule("example 2", 'en')
    assert rule2 != None

def test_get_rules():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule1 = tag1.get_rule("example 1", 'en')
    rule2 = tag1.get_rule("example 2", 'en')
    assert set([rule1, rule2]) == set(tag1.get_rules())

def test_get_unknown_rule():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule1 = tag1.get_rule("unknown 1", 'en')
    assert rule1 == None

def test_get_rules_confuser():
    confcomputer = get_confcomputer()
    confuser = get_confuser()
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule1 = tag1.get_rule("example 1", 'en')
    rule1.add_variable(name='variable1', type='boolean', value_on=True,
                        value_off=False, conflevel=confuser)
    assert set([rule1]) == set(tag1.get_rules(conflevel=confuser))
    assert set([]) == set(tag1.get_rules(conflevel=confcomputer))
    rule2 = tag1.get_rule("example 2", 'en')
    rule2.add_variable(name='variable2', type='boolean', value_on=True,
                        value_off=False, conflevel=confcomputer)
    assert set([rule1]) == set(tag1.get_rules(conflevel=confuser))
    assert set([rule2]) == set(tag1.get_rules(conflevel=confcomputer))
    rule2.add_variable(name='variable3', type='boolean', value_on=True,
                        value_off=False, conflevel=confuser)
    assert set([rule1, rule2]) == set(tag1.get_rules(conflevel=confuser))
    assert set([rule2]) == set(tag1.get_rules(conflevel=confcomputer))

def test_add_rule_display():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule3 = tag1.add_rule(type='boolean', display=False)
    rule3.add_translation(lang='en', name='example 3')

def test_get_rules_display():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule1 = tag1.get_rule("example 1", 'en')
    rule2 = tag1.get_rule("example 2", 'en')
    rule3 = tag1.get_rule("example 3", 'en')
    assert set([rule1, rule2, rule3]) == set(tag1.get_rules())
    assert set([rule1, rule2]) == set(tag1.get_rules(display=True))
    assert set([rule1, rule2, rule3]) == set(tag1.get_rules(display=False))

def test_add_same_translation_rule_same_tag():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    rule1 = tag1.add_rule(type='boolean', display=False)
    rule1.add_translation(lang='en', name='same name rule')
    rule2 = tag1.add_rule(type='boolean', display=False)
    raises(Exception, "rule2.add_translation(lang='en', name='same name rule')")

def test_add_same_translation_rule_diff_tags():
    category1 = get_category('example', 'en')
    tag1 = category1.get_tag('tag1', 'en')
    tag2 = category1.get_tag('tag2', 'en')
    rule1 = tag1.add_rule(type='boolean', display=False)
    rule1.add_translation(lang='en', name='same name rule 2')
    rule2 = tag2.add_rule(type='boolean', display=False)
    rule2.add_translation(lang='en', name='same name rule 2')

def test_get_categories_conflevel():
    confcomputer = get_confcomputer()
    confuser = get_confuser()
    category1 = get_category('example', 'en')
    category2 = get_category('comment_fr', 'fr')
    tag1 = category2.get_tag('tag comment', 'en')
    assert [category1] == get_categories(conflevel=confuser)
    assert [category1] == get_categories(conflevel=confcomputer)
    rule1 = tag1.add_rule(type='boolean')
    rule1.add_translation(lang='en', name='name rule 3')
    rule1.add_variable(name='variable2', type='boolean', value_on=True,
                        value_off=False, conflevel=confuser)
    assert [category1, category2] == get_categories(conflevel=confuser)
    assert [category1] == get_categories(conflevel=confcomputer)
    rule1.add_variable(name='variable3', type='boolean', value_on=True,
                        value_off=False, conflevel=confcomputer)
    assert [category1, category2] == get_categories(conflevel=confuser)
    assert [category1, category2] == get_categories(conflevel=confcomputer)

def test_del_tag_with_rule():
    category2 = get_category('comment_fr', 'fr')
    tag1 = category2.get_tag('tag comment', 'en')
    raises(Exception, "category2.del_tag(tag1)")


##FIXME category par groupe


def test_close_database():
    commit_database()
    close_database()
