# -*- coding: utf-8 -*-

#########################################################################
# pyeole.service._network - manage network service
# Copyright © 2013 Pôle de Compétence EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
#########################################################################

"""EOLE network service management

This internal module is only for use by :mod:`pyeole.service`.

"""

import io
from os.path import join
from os.path import isfile

from os import unlink

import logging

# Base exception
from pyeole.service.error import ServiceError

from pyeole.service.module.upstart import Service as Upstart

from pyeole.process import system_out

log = logging.getLogger(__name__)
log.addHandler(logging.NullHandler())

_ENCODING = 'utf-8'

_RUN_DIR = u'/run/network'

_STATE_FILE = join(_RUN_DIR, u'ifstate')

_INTERFACE_FILE = u'/etc/network/interfaces'

_UPSTART_SERVICE = {u'name': u'networking',
                    u'container': u'root',
                    u'method': u'upstart',
                    u'pty': True}

#####
##### Per action workers
#####


class Service(Upstart):
    module_name = u'network'
    def cmd_check_service(self, service, ctx):
        super(Service, self).cmd_check_service(_UPSTART_SERVICE, ctx)

    def _enable_service(self, service, ctx):
        super(Service, self).cmd_enable_service(_UPSTART_SERVICE, ctx)

    def _disable_service(self, service, ctx):
        super(Service, self).cmd_disable_service(_UPSTART_SERVICE, ctx)


    def cmd_status_service(self, service, ctx):
        missing_interfaces = []
        enabled_interfaces = self._parse_interfaces()
        active_interfaces = self._list_ifstate()
        active_bridges = self._list_bridges()

        remaining_interfaces = active_interfaces[:]

        for interface, bridge in enabled_interfaces.items():
            if bridge and interface in active_bridges.get(bridge, []):
                if interface in remaining_interfaces:
                    remaining_interfaces.remove(interface)
                continue
            elif interface in active_interfaces:
                if interface in remaining_interfaces:
                    remaining_interfaces.remove(interface)
                continue
            else:
                missing_interfaces.append(interface)

        if missing_interfaces:
            plural = u's' * (len(missing_interfaces) > 1)
            msg = u'Unconfigured interface{0}: {1}'
            log.warn(msg.format(plural, u', '.join(missing_interfaces)))

        if remaining_interfaces:
            plural = u's' * (len(remaining_interfaces) > 1)
            msg = u'Unknown interface{0} up: {1}'
            log.warn(msg.format(plural, u', '.join(remaining_interfaces)))

    def cmd_stop_service(self, service, ctx):
        self._down_interfaces()

    def cmd_restart_service(self, service, ctx):
        self._down_interfaces()
        self._if_up(u'-a')

    def cmd_start_service(self, service, ctx):
        self.cmd_restart_service(service, ctx)

    def _reload_service(self, service, ctx):
        raise NotImplementedError('Reloading network is not supported')


    #####
    ##### Utilities for network
    #####


    def _run_iproute(self, command):
        """Wrapper to run iproute commands

        :param command: ip command to run
        :type command: `list`
        :return: return values of :func:`system_out`
        :rtype: `tuple`

        """
        cmd = [u'ip', u'-o'] + command
        return system_out(cmd)


    def _down_interfaces(self):
        """Unconfigure an interface

        """
        interfaces = self._parse_interfaces()
        ifstate = self._list_ifstate()
        bridges = self._list_bridges()

        for interface in ifstate:
            if interface == u'lo' or bridges.get(interface, None):
                # Do not touch to loopback
                # and skip bridges, they could have other interfaces
                continue
            elif interface.find('.') != -1:
                log.debug(u'Deleting vlan interface {0}'.format(interface))
                self._ip_delete(interface)
            elif interface.find(':') != -1:
                log.debug(u'Disable alias interface {0}'.format(interface))
                self._ip_down(interface)
            else:
                log.debug(u'Disable interface {0}'.format(interface))
                self._ip_down(interface)

        for bridge, interfaces in bridges.items():
            if bridge == 'br0':
                continue
            for port in interfaces:
                not_managed_interfaces = []
                if not port.startswith(u'eth'):
                    not_managed_interfaces.append(port)
                else:
                    msg = u'Disable port {0} in bridge {1}'
                    log.debug(msg.format(port, bridge))
                    self._ip_down(port)

            if not_managed_interfaces:
                log.debug(u'Disable bridge {0}'.format(bridge))
                self._ip_down(bridge)
            else:
                log.debug(u'Delete bridge {0}'.format(bridge))
                self._ip_delete(bridge)


    def _ifconfig_down(self, name):
        """Disable an interface with ifconfig

        :param name: interface name
        :type name: `str`

        """
        if name == u'lo':
            return
        system_out([u'ifconfig', name, 'down'])
        self._delete_if_from_state(name)


    def _if_up(self, interface):
        """Enable an interface with ifup

        :param interface: interface name or “-a” for all
        :type interface: `str`

        """
        if interface == u'lo':
            return
        system_out([u'ifup', interface], pty=True)


    def _if_down(self, name):
        """Disable an interface with ifdown

        :param name: interface name
        :type name: `str`

        """
        if name == u'lo':
            return
        system_out([u'ifdown', name])


    def _ip_down(self, name):
        """Disable an interface with iproute

        :param name: interface name
        :type name: `str`

        """
        if name == u'lo':
            return
        self._run_iproute([u'link', u'set', name, u'nomaster'])
        self._run_iproute([u'link', u'set', name, u'down'])
        self._run_iproute([u'addr', u'flush', name])
        self._delete_if_from_state(name)


    def _ip_delete(self, name):
        """Delete an interface with iproute

        :param name: interface name
        :type name: `str`

        """
        if name == u'lo':
            return
        self._run_iproute([u'link', 'delete', name])
        self._delete_if_from_state(name)


    def _list_ifstate(self):
        """List all interfaces seen by networking script.

        All running interfaces are listed in ``/run/network/ifstate``.

        :return: interfaces
        :rtype: `list`

        """
        interfaces = []
        try:
            with io.open(_STATE_FILE, 'r', encoding=_ENCODING) as state:
                for line in state.readlines():
                    elts = line.split(u'=')
                    if len(elts) != 2:
                        continue
                    interfaces.append(elts[0])
        except IOError:
            pass
        return interfaces


    def _delete_if_from_state(self, name):
        """Remove an interface from state file

        :param name: interface name
        :type name: `str`

        """
        statelines = []
        ifstate = join(_RUN_DIR, 'ifstate.{0}'.format(name))
        try:
            with io.open(_STATE_FILE, 'r', encoding=_ENCODING) as state:
                for line in state.readlines():
                    if line.startswith(name):
                        continue
                    statelines.append(line)
            with io.open(_STATE_FILE, 'w', encoding=_ENCODING) as state:
                state.writelines(statelines)
        except IOError:
            pass

        if isfile(ifstate):
            unlink(ifstate)


    def _list_interfaces(self):
        """List all active interfaces seen by iproute.

        Strip loopback interface as we never want to stop it.

        :return: network interfaces
        :rtype: `list`

        """
        interfaces = []
        code, stdout, stderr = self._run_iproute([u'link', u'show'])
        for line in stdout.split(u'\n'):
            elts = line.split()
            if len(elts) > 2:
                interface = elts[1].strip(u':')
                if interface != u'lo':
                    interfaces.append(interface)
        return interfaces


    def _list_bridges(self):
        """List all active bridges and attached interfaces.

        :return: bridges interfaces with their associated interfaces
        :rtype: `dict`

        """
        bridges = {}
        code, stdout, stderr = self._run_iproute([u'link', u'show'])
        for line in stdout.split(u'\n'):
            elts = line.split()
            if u'master' in elts:
                master = elts[elts.index(u'master')+1]
                if not bridges.get(master, None):
                    bridges[master] = []
                bridges[master].append(elts[1].strip(u':').split('@')[0])
        return bridges


    def _parse_interfaces(self):
        """Parse network configuration file

        The returned value consist of a dictionary :

            ``interface``:``master bridge name``

        The bridge name is ``None`` if the interface is not part of a
        bridge.

        :return: configured interface
        :rtype: `dict`

        """
        interfaces = {}
        conf_file = io.open(_INTERFACE_FILE, 'r', encoding=_ENCODING)
        interface = {}

        all_eth = [name for name in self._list_interfaces() if name.startswith(u'eth')]

        # Always add an empty line
        for line in conf_file.readlines() + [u'']:
            clean_line = line.strip()
            if clean_line.startswith(u'#'):
                continue

            if len(clean_line) == 0 and interface:
                for port in interface[u'ports']:
                    if interfaces.get(port, None) is not None:
                        msg = u'Interface can not be in multiple bridges'
                        raise ValueError(msg)

                    interfaces[port] = interface[u'name']
                interfaces[interface[u'name']] = None
                interface = {}
            elif len(clean_line) == 0:
                continue
            if clean_line.startswith(u'iface'):
                name = clean_line.split()[1]
                interface[u'name'] = name
                interface[u'ports'] = []
                continue
            elif interface:
                param = clean_line.split()
                if param[0] == u'bridge_ports':
                    ports = param[1:]
                    if u'all' in param[1:]:
                        ports = list(set(param[1:] + all_eth))
                        ports.remove(u'all')
                    interface[u'ports'] = ports

        conf_file.close()
        return interfaces
