# -*- coding: utf-8 -*-

#########################################################################
# pyeole.service._service - manage System V services
# Copyright © 2012-2013 Pôle de Compétence EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
#########################################################################

"""EOLE System V service management

This internal module is only for use by :mod:`pyeole.service`.

"""

from glob import glob

from os import unlink, symlink
from os.path import dirname
from os.path import isfile
from os.path import join
from os.path import relpath

import re

import logging

# Base exception
from pyeole.service.error import ServiceError
# Configuration
from pyeole.service.error import ConfigureError
from pyeole.service.error import DisabledError
from pyeole.service.error import UnknownServiceError
# Action
from pyeole.service.error import StartError
from pyeole.service.error import StopError
from pyeole.service.launcher import Command

_DISTRIB_RUNLEVEL = 2
_DISTRIB_STOP_RUNLEVEL = 0
_DISTRIB_DEFAULT_START_LEVEL = 20
_DISTRIB_DEFAULT_STOP_LEVEL = 20
log = logging.getLogger(__name__)
log.addHandler(logging.NullHandler())


_SERVICE_CMD = u'/usr/sbin/service'

#####
##### Per action workers
#####

class Service(Command):
    module_name = u'System V'
    fake_restart = True

    def cmd_check_service(self, service, ctx):
        name = service[u'name']
        cont = ctx[u'name']
        init, start_links, stop_links = self._build_sysv_filenames(service,
                                                              ctx,
                                                              globbing=True)
        #start = glob(start_links[_DISTRIB_RUNLEVEL])
        #stop = glob(stop_links[_DISTRIB_STOP_RUNLEVEL])

        if not isfile(init):
            msg = u'Missing System V init file for service {0} in {1}: {2}'
            raise UnknownServiceError(msg.format(name, cont, init))
        #elif not start:
        #    raise DisabledError(u'Service {0} disabled'.format(name))
        #elif len(start) > 1:
        #    links = u', '.join(start)
        #    msg = u'Multiple System V start links for service {0} in {1}'
        #    msg += u' for runlevel {2}: {3}'
        #    raise ConfigureError(msg.format(name, cont, _DISTRIB_RUNLEVEL, links))
        #elif not stop:
        #    msg = u'Missing System V stop link for service {0} in {1}'
        #    msg += u' for runlevel {2}'
        #    raise ConfigureError(msg.format(name, cont, _DISTRIB_STOP_RUNLEVEL))
        #elif len(stop) > 1:
        #    links = u', '.join(stop)
        #    msg = u'Multiple System V stop links for service {0} in {1}'
        #    msg += u' for runlevel {2}: {3}'
        #    raise ConfigureError(msg.format(name, cont,
        #                                    _DISTRIB_STOP_RUNLEVEL, links))


    def cmd_enable_service(self, service, ctx):
        name = service[u'name']
        cont = ctx[u'name']
        init, start_links, stop_links = self._build_sysv_filenames(service, ctx)
        try:
            log.debug(u'Enabling System V service requires a cleaned place')
            self.cmd_disable_service_worker(service, ctx)
            for runlevel, link in start_links.items():
                relative_init = relpath(init, dirname(link))
                debug_msg = u'Creating runlevel {0} start link {1} => {2}'
                log.debug(debug_msg.format(runlevel, link, relative_init))
                symlink(relative_init, link)

            for runlevel, link in stop_links.items():
                relative_init = relpath(init, dirname(link))
                debug_msg = u'Creating runlevel {0} stop link {1} => {2}'
                log.debug(debug_msg.format(runlevel, link, relative_init))
                symlink(relative_init, link)
        except (IOError, OSError), err:
            msg = u'Can not enable System V service {0} in {1}: {2}'
            raise ConfigureError(msg.format(name, cont, err))


    def cmd_disable_service(self, service, ctx):
        self.cmd_disable_service_worker(service, ctx)


    def cmd_disable_service_worker(self, service, ctx):
        name = service[u'name']
        cont = ctx[u'name']
        init, start_globs, stop_globs = self._build_sysv_filenames(service, ctx,
                                                              globbing=True)
        try:
            for runlevel, link_glob in start_globs.items():
                for link in glob(link_glob):
                    debug_msg = u'Removing runlevel {0} start link {1}'
                    log.debug(debug_msg.format(link, runlevel))
                    unlink(link)

            for runlevel, link_glob in stop_globs.items():
                for link in glob(link_glob):
                    debug_msg = u'Removing runlevel {0} stop link {1}'
                    log.debug(debug_msg.format(link, runlevel))
                    unlink(link)
        except (IOError, OSError), err:
            msg = u'Can not disable System V service {0} in {1}: {2}'
            raise ConfigureError(msg.format(name, cont, err))


    def cmd_status_service(self, service, ctx):
        self._do_sysv(u'status', service, ctx)


    def cmd_start_service(self, service, ctx):
        self._do_sysv(u'start', service, ctx)


    def cmd_stop_service(self, service, ctx):
        self._do_sysv(u'stop', service, ctx)


    def cmd_restart_service(self, service, ctx):
        # workaround missing restart
        self.stop_service(service, ctx)
        self.start_service(service, ctx)


    def cmd_reload_service(self, service, ctx):
        self._do_sysv(u'reload', service, ctx)


    #####
    ##### System V catch action
    #####


    def _do_sysv(self, action, service, ctx):
        """Perform action for System V service.

        Check for return code of the check command or output.
        Some init scripts status functions have some non-standard output.

        :param action: action to perform
        :type action: `str` in [``status``, ``start``, ``restart``,
                      ``stop``, ``reload``]
        :param service: service informations
        :type service: `dict`
        :param ctx: container context
        :type ctx: `dict`
        :raise StartError: if start, restart or reload is not
                                  confirmed
        :raise StopError: if stop is not confirmed
        :raise ServiceError: if another :data:`action` is not confirmed

        """
        name = service[u'name']
        cont = ctx[u'name']
        cmd = [_SERVICE_CMD, name, action]
        check = [_SERVICE_CMD, name, u'status']
        # As defined in /lib/lsb/init-functions pidofproc
        status_code = [0]
        if action in [u'start', u'restart', u'reload']:
            code_ok = [0]
            match_ok = r'{0} is running$'
            error_string = u'Service {0} in {1} not started: {2}'
            service_exc = StartError
        elif action == u'stop':
            code_ok = [0, 3, 7]
            match_ok = r'{0} is not running$'
            error_string = u'Service {0} in {1} not stopped: {2}'
            service_exc = StopError
        elif action == u'status':
            code_ok = status_code
            match_ok = r'heavily depending of service script'
            error_string = u'Unable to get status of'
            error_string += u' System V service {0} in {1}: {2}'
            service_exc = ServiceError
        else:
            code_ok = [0]
            match_ok = r'.'
            error_string = u'Error: {2}'
            service_exc = ServiceError
        action_code, action_stdout, action_stderr = self._exec_cmd(cmd, service, ctx)
        code, stdout, stderr = self._exec_cmd(check, service, ctx)

        # Check return code of action or result of “status”
        if not (action_code in code_ok
                or code in status_code or re.match(match_ok.format(name), stdout)):
            output = action_stderr or stderr or stdout
            raise service_exc(error_string.format(name, cont, output))


    #####
    ##### Utilities for System V services
    #####


    def _build_sysv_filenames(self, service, ctx, globbing=False):
        """Return the init filenames and the start/stop links.

        The enabled and disabled links are `dict` per supported runlevels.

        :param service: service informations
        :type service: `dict`
        :param ctx: container context
        :type ctx: `dict`
        :return: init filename, start and stop links
        :rtype: `tuple`

        """
        name = service[u'name']
        init = join(u'/', ctx[u'path'], u'etc/init.d/{0}'.format(name))
        start_links, stop_links = self._build_sysv_links(service, ctx,
                                                    globbing=globbing)

        return init, start_links, stop_links


    def _build_sysv_links(self, service, ctx, globbing=False):
        """Build System V start/stop links per supported runlevels.

        :param service: service informations
        :type service: `dict`
        :param ctx: container context
        :type ctx: `dict`
        :param globbing: use globbing instead of service level
        :type globbing: `bool`
        :return: start and stop links per runlevels
        :rtype: `tuple`

        """
        name = service[u'name']
        # Compute start/stop runlevels
        runlevels = set(range(0, 7))
        start_runlevels = set(range(_DISTRIB_RUNLEVEL, 6))
        # Stop service in all unstarted levels
        stop_runlevels = runlevels - start_runlevels

        if globbing:
            startlevel = u'??'
            stoplevel = u'??'
        else:
            startlevel = service.get(u'startlevel', _DISTRIB_DEFAULT_START_LEVEL)
            stoplevel = service.get(u'stoplevel', _DISTRIB_DEFAULT_STOP_LEVEL)

        start_links = {}
        stop_links = {}
        for stop in stop_runlevels:
            link = join(u'/', ctx[u'path'],
                        u'etc/rc{0}.d/K{1}{2}'.format(stop, stoplevel, name))
            stop_links[stop] = link

        for start in start_runlevels:
            link = join(u'/', ctx[u'path'],
                        u'etc/rc{0}.d/S{1}{2}'.format(start, startlevel, name))
            start_links[start] = link

        return start_links, stop_links
