# -*- coding: utf-8 -*-

#########################################################################
# pyeole.service._upstart - manage upstart services
# Copyright © 2013 Pôle de Compétence EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
#########################################################################

"""EOLE upstart service management

This internal module is only for use by :mod:`pyeole.service`.

"""

from os import unlink
from os.path import join, isfile

import re

import logging

# Base exception
from pyeole.service.error import ServiceError
# Configuration
from pyeole.service.error import ConfigureError
from pyeole.service.error import DisabledError
from pyeole.service.error import UnknownServiceError
# Action
from pyeole.service.error import StartError
from pyeole.service.error import StopError

from pyeole.service.launcher import Command

log = logging.getLogger(__name__)
log.addHandler(logging.NullHandler())

_UPSTART_OVERRIDE = u'manual'
"""Token to disable an upstart service.

This token must be present in ``/etc/init/<service>.override`` file.

"""

_SERVICE_CMD = u'/usr/sbin/service'

#####
##### Per action workers
#####


class Service(Command):
    module_name = u'Upstart'
    def cmd_check_service(self, service, ctx):
        conf, override = self._build_upstart_filenames(service, ctx)

        #try:
        #    overrode = file(override, 'r').read().startswith(_UPSTART_OVERRIDE)
        #except (IOError, OSError):
        #    overrode = False

        if not isfile(conf):
            msg = u'Missing upstart configuration file for service {0} in {1}: {2}'
            raise UnknownServiceError(msg.format(service[u'name'], ctx[u'name'],
                                                 conf))
        #elif isfile(override) and overrode:
        #    msg = u'Service {0} disabled'
        #    raise DisabledError(msg.format(service[u'name']))


    def cmd_enable_service(self, service, ctx):
        conf, override = self._build_upstart_filenames(service, ctx)
        if not isfile(override):
            msg = u'Already enabled upstart service {0} in {1}'
            log.debug(msg.format(service[u'name'], ctx[u'name']))
        else:
            log.debug(u'Deleting upstart override file {0}'.format(override))
            try:
                unlink(override)
            except OSError, err:
                msg = u'Can not enable upstart service {0} in {1}: {2}'
                raise ConfigureError(msg.format(service[u'name'], ctx[u'name'],
                                                err))


    def cmd_disable_service(self, service, ctx):
        conf, override = self._build_upstart_filenames(service, ctx)
        try:
            overrode = file(override, 'r').read().startswith(_UPSTART_OVERRIDE)
        except (IOError, OSError):
            overrode = False

        if isfile(override) and overrode:
            msg = u'Already disabled upstart service {0} in {1}'
            log.debug(msg.format(service[u'name'], ctx[u'name']))
        else:
            log.debug(u'Creating upstart override file {0}'.format(override))
            try:
                file(override, 'w').write(u'{0}\n'.format(_UPSTART_OVERRIDE))
            except (IOError, OSError), err:
                msg = u'Can not disable upstart service {0} in {1}: {2}'
                raise ConfigureError(msg.format(service[u'name'], ctx[u'name'],
                                                err))


    def cmd_status_service(self, service, ctx):
        self._do_upstart(u'status', service, ctx)


    def cmd_start_service(self, service, ctx):
        self._do_upstart(u'start', service, ctx)


    def cmd_stop_service(self, service, ctx):
        self._do_upstart(u'stop', service, ctx)


    def cmd_restart_service(self, service, ctx):
        self._do_upstart(u'restart', service, ctx)


    def cmd_reload_service(self, service, ctx):
        self._do_upstart(u'reload', service, ctx)


    #####
    ##### Upstart catch action
    #####


    def _do_upstart(self, action, service, ctx):
        """Perform action for upstart service.

        Upstart never use exit code to report status.

        :param action: action to perform
        :type action: `str` in [``status``, ``start``, ``restart``,
                      ``stop``, ``reload``]
        :param service: service informations
        :type service: `dict`
        :param ctx: container context
        :type ctx: `dict`
        :raise StartError: if start, restart or reload is not confirmed
        :raise StopError: if stop is not confirmed
        :raise ServiceError: if another :data:`action` is not confirmed

        """
        name = service[u'name']
        cmd = [_SERVICE_CMD, name, action]
        check = [_SERVICE_CMD, name, u'status']
        if action in [u'start', u'restart', u'reload']:
            match_ok = r'^{0} start/running'
            error_string = u'Service {0} in {1} not started: {2}'
            service_exc = StartError
        elif action == u'stop':
            match_ok = r'^{0} stop/waiting'
            error_string = u'Service {0} in {1} not stopped: {2}'
            service_exc = StopError
        elif action == u'status':
            match_ok = r'^{0} start/running'
            error_string = u'Unable to get status of upstart service {0} in {1}:'
            error_string += u' {2}'
            service_exc = ServiceError
        else:
            match_ok = r'.'
            error_string = u'Error: {2}'
            service_exc = ServiceError

        action_code, action_stdout, action_stderr = self._exec_cmd(cmd, service, ctx, out=True)
        code, stdout, stderr = self._exec_cmd(check, service, ctx, out=True, pty=False)

        if not re.match(match_ok.format(name), stdout):
            output = action_stderr or stderr or stdout
            raise service_exc(error_string.format(name, ctx[u'name'], output))


    #####
    ##### Utilities for upstart
    #####


    def _build_upstart_filenames(self, service, ctx):
        """Return the configuration and override filenames of an upstart service.

        :param service: service informations
        :type service: `dict`
        :param ctx: container context
        :type ctx: `dict`
        :return: configuration and override filenames
        :rtype: `tuple`

        """
        name = service[u'name']
        conf = join(u'/', ctx[u'path'], u'etc/init/{0}.conf'.format(name))
        override = join(u'/', ctx[u'path'], u'etc/init/{0}.override'.format(name))
        return conf, override
