# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
    Gestion des préférences d'importation
"""
from os.path import isfile
from scribe.ldapconf import PROFILS, SUPPORT_ETAB, dico
from scribe.importation.config import PREF_FILES
if SUPPORT_ETAB:
    import sys
    sys.path.append('/usr/share/')
    from ead2.backend.actions.scribe.tool.etablissements import get_etabs

class Preference:
    """
    Objet préférence
    """

    def __init__(self, name, _type, question, options, default=''):
        if _type not in ['liste', 'texte', 'entier', 'select']:
            raise Exception("Type de préférence invalide : %s" % _type)
        self._type = _type
        self.name = name
        self.question = question
        self.options = options
        if default == '' and _type == 'liste':
            self.default = options[0][0]
        else:
            self.default = default

    def set_default(self, value):
        """
        mise à jour de la valeur par défaut
        """
        if self._type == 'liste' or self._type == 'select':
            if value in [item[0] for item in self.options]:
                self.default = value
        elif self._type == 'entier':
            try:
                self.default = str(int(value))
            except:
                pass
        else:
            self.default = value

class Preferences:
    """
    Gestionnaire de préférences
    """
    def __init__(self, preferences, filepath):
        self.preferences = preferences
        self.filepath = filepath
        self._update_default()

    def _find_pref(self, name):
        """
        renvoit l'objet Preference recherché
        """
        for pref in self.preferences:
            if pref.name == name:
                return pref

    def _update_default(self):
        """
        récupération des valeurs par défaut depuis le fichier
        """
        if isfile(self.filepath):
            lines = file(self.filepath).read().splitlines()
            for line in lines:
                if len(line.split('##')) == 2:
                    pref, value = line.split('##')
                    self.set_default(pref, value)

    def get_default(self, name):
        """
        renvoie la valeur par défaut courante d'une préférence
        """
        return self._find_pref(name).default

    def set_default(self, name, value):
        """
        enregistrement de la valeur par defaut
        """
        self._find_pref(name).set_default(value)

    def save(self):
        """
        enregistrement des valeurs par défaut dans un fichier
        """
        lines = []
        for pref in self.preferences:
            lines.append("%s##%s" % (pref.name, pref.default))
        pref_file = file(self.filepath, 'w')
        pref_file.write("\n".join(lines))
        pref_file.close()

# -- préférences d'importation -- #

PREF_DOMAINE = ['domaine', 'liste',
        "Domaine de messagerie élève par défaut",
        (('restreint', 'restreint'), ('internet', 'Internet'))]
PREF_QUOTA = ['quota', 'entier',
            "Quota disque en Mo (0 pour inactif)",
            ""]
PREF_GEN_PWD = ['gen_pwd', 'liste',
            "Génération des mots de passe",
            (('alea', 'aléatoire'), ('naiss', 'date de naissance'))]
PREF_CHANGE_PWD = ['change_pwd', 'liste',
            "Changement du mot de passe à la première connexion",
            (('oui', 'oui'), ('non', 'non'))]
PREF_LOGIN = ['login', 'liste',
            "Génération des identifiants",
            (('standard', 'prenom.nom'), ('pnom', 'pnom'),
             ('nomp', 'nomp'), ('p.nnn', 'p.nnn (format Gibii)'),
             ('prenom.n', 'prenom.n'),
            )]
PREF_SHELL = ['shell', 'liste',
            "Activer le shell (clients Linux)",
            (('oui', 'oui'), ('non', 'non'))]
PREF_PROFIL = ['profil', 'liste',
            "Profil Windows",
            PROFILS]

PREF_MAIL = ['mail', 'liste',
             "Adresse mail",
             ( ('perso_restreint', 'adresse fournie ou domaine restreint'),
               ('perso_internet', 'adresse fournie ou domaine Internet'),
               ('perso_aucune', 'adresse fournie ou aucune'),
               ('restreint', 'adresse locale, domaine restreint'),
               ('internet', 'adresse locale, domaine Internet'),
               ('aucune', 'aucune adresse'),
            )]

PREF_PREFIX_ETAB = ['etab_prefix', 'texte',
        "Prefix des groupes de cet établissement", ""]

if dico.get('activer_nfs', 'non') == 'oui' or dico.has_key('interface_client_ltsp'):
    DEFAULT_CHANGE_PWD = 'non'
    DEFAULT_SHELL = 'oui'
else:
    DEFAULT_CHANGE_PWD = 'oui'
    DEFAULT_SHELL = 'non'

def get_etab():
    """
    charge les établissements à chaque fois qu'on le demande
    """
    tetabs = get_etabs()
    etabs = [(tetabs[0], tetabs[0])]
    for etab in tetabs[1]:
        etabs.append((etab, etab))
    return ['etab', 'select', "Etablissement", etabs]

def get_eleves_prefs():
    """ préférences pour les élèves """
    prefs_eleve = [ Preference(*PREF_DOMAINE),
                    Preference(default='50', *PREF_QUOTA),
                    Preference(*PREF_LOGIN),
                    Preference(*PREF_GEN_PWD),
                    Preference(default=DEFAULT_CHANGE_PWD, *PREF_CHANGE_PWD),
                    Preference(default=DEFAULT_SHELL, *PREF_SHELL),
                    Preference(default='2', *PREF_PROFIL),
    ]
    if SUPPORT_ETAB:
        prefs_eleve.insert(0, Preference(*get_etab()))
        prefs_eleve.insert(1, Preference(*PREF_PREFIX_ETAB))
    return Preferences(prefs_eleve, PREF_FILES['eleve'])

def get_responsables_prefs():
    """ préférences pour les responsables """
    prefs_resp = (Preference(*PREF_LOGIN),
                  Preference(*PREF_GEN_PWD),
                  Preference(*PREF_MAIL),
    )
    return Preferences(prefs_resp, PREF_FILES['responsable'])


def get_enseignants_prefs():
    """ préférences pour les enseignants """
    prefs_ens = [   Preference(default='0', *PREF_QUOTA),
                    Preference(*PREF_LOGIN),
                    Preference(*PREF_GEN_PWD),
                    Preference(default=DEFAULT_CHANGE_PWD, *PREF_CHANGE_PWD),
                    Preference(default=DEFAULT_SHELL, *PREF_SHELL),
                    Preference(default='2', *PREF_PROFIL),
                    Preference(default='perso_aucune', *PREF_MAIL),
    ]
    if SUPPORT_ETAB:
        prefs_ens.insert(0, Preference(*get_etab()))
        prefs_ens.insert(1, Preference(*PREF_PREFIX_ETAB))
    return Preferences(prefs_ens, PREF_FILES['enseignant'])

def get_administratifs_prefs():
    """ préférences pour les administratifs """
    prefs_admin = ( Preference(default='0', *PREF_QUOTA),
                    Preference(*PREF_LOGIN),
                    Preference(*PREF_GEN_PWD),
                    Preference(default=DEFAULT_CHANGE_PWD, *PREF_CHANGE_PWD),
                    Preference(default=DEFAULT_SHELL, *PREF_SHELL),
                    Preference(default='1', *PREF_PROFIL),
                    Preference(default='perso_aucune', *PREF_MAIL),
    )
    return Preferences(prefs_admin, PREF_FILES['administratif'])

def get_invites_prefs():
    """ préférences pour les comptes invités """
    prefs_autre = (Preference(*PREF_LOGIN),
                   Preference(*PREF_GEN_PWD),
                   Preference(*PREF_MAIL),
    )
    return Preferences(prefs_autre, PREF_FILES['invite'])
