# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG - 2011
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################
"""
 librairie pour le parsing des fichiers de données Sconet/STS-Web

 - parse_sco_eleves : parsing des fichiers XML issus de Sconet
 - parse_sts_profs  : parsing des fichiers XML issus de STS-Web

"""

from scribe.eoletools import replace_cars, formate_civilite, \
not_empty, is_empty, formate_date, replace_more_cars
from scribe.importation import log
from scribe.importation.config import DEBUG_NUM
from scribe.storage import Eleve, \
Responsable, Adresse, JointureResponsableEleve, \
Classe, Niveau, Enseignant, Administratif, \
EnsClasse, JointureClasseEnseignant, \
Matiere, JointureMatiereEnseignant, \
Groupe, JointureGroupeUser, Service
from scribe.parsing.tools import parse_xml
from scribe.parsing.nomenclature import REGIMES

##########################################
# Extraction Sconet Elèves & Responsables
# Fichiers :
# - ElevesSansAdresses.xml
# - ResponsablesAvecAdresses.xml
##########################################

ERRMEF = "Le niveau portant le code MEF %s n'est pas défini dans la nomenclature"

def parse_sco_divisions(storage, structure_file, nomenclature_file):
    """
    Liaison Division (classe) - MEF (niveau)
    d'après : Structures.xml et Nomenclatures.xml
    """
    num = 0
    log.infolog("Lecture des classes et des niveaux...", title=True)
    niveaux = {}
    context, _ = parse_xml(nomenclature_file, 'MEF')
    for _, tnom in context:
        mef = tnom.attrib['CODE_MEF']
        niveaux[mef] = {}
        clean_formation = replace_cars(tnom.find('FORMATION').text)
        niveaux[mef]['label'] = unicode(clean_formation)
        if tnom.find('MEF_RATTACHEMENT') is not None:
            niveaux[mef]['real'] = tnom.find('MEF_RATTACHEMENT').text
    divisions = {}
    context, _ = parse_xml(structure_file, 'DIVISION')
    for _, tstr in context:
        div = tstr.attrib['CODE_STRUCTURE']
        if div.lower() == 'inactifs':
            # élèves inactifs (cf. #913)
            continue
        tmef = tstr.findall('MEFS_APPARTENANCE/MEF_APPARTENANCE/CODE_MEF')
        if len(tmef) == 1:
            try:
                divisions[div] = niveaux[tmef[0].text]['label']
            except KeyError:
                msg = ERRMEF % tmef[0].text
                log.errorlog(msg)
                raise Exception(msg)
        else:
            # si plusieurs mef, on prend celle de rattachement
            errmef = set()
            for txtmef in tmef:
                try:
                    mef = niveaux[txtmef.text]['real']
                except KeyError:
                    log.errorlog(ERRMEF % tmef[0].text)
                    errmef.add(tmef[0].text)
                    continue
                try:
                    divisions[div] = niveaux[mef]['label']
                    break
                except KeyError:
                    log.errorlog(ERRMEF % mef)
                    errmef.add(mef)
                    continue
            if not divisions.has_key(div):
                if errmef:
                    raise Exception(ERRMEF % list(errmef))
                else:
                    raise Exception("La division %s n'est associée à aucune MEF" % div)
    for classe, niveau in divisions.items():
        my_niveau = storage.findOrCreate(Niveau, nom=niveau)
        clean_classe = unicode(replace_cars(classe))
        storage.findOrCreate(Classe, nom=clean_classe,
                             niveau=my_niveau)
        num += 1
    log.infolog("TOTAL : %d classes" % num)

def parse_sco_groupes(store, structure_file):
    """
    Recherche des Groupes (Options) et de leurs libellés
    """
    num = 0
    log.infolog("Lecture des groupes (options)...", title=True)
    context, _ = parse_xml(structure_file, 'GROUPE')
    for _, tstr in context:
        grp = unicode(replace_cars(tstr.attrib['CODE_STRUCTURE']))
        desc = unicode(replace_cars(tstr.find('LIBELLE_LONG').text))
        Groupe(store=store, nom=grp, description=desc)
        num += 1
    log.infolog("TOTAL : %d groupes" % num)

def parse_sco_eleves(store, eleve_file):
    """
    parsing des élèves depuis Sconet
    """
    num = 0
    log.infolog("Lecture des élèves...", title=True)
    context, eleve_file = parse_xml(eleve_file, 'ELEVE')
    # nb : la date de naissance est au format jj/mm/aaaa
    mapping = {'nom':'NOM',
              'prenom':'PRENOM',
              'date':'DATE_NAISS',
              'civilite':'CODE_SEXE',
              'numero':'ELENOET',
              'ine':'ID_NATIONAL',
              'prenom2':'PRENOM2',
              'regime':'CODE_REGIME',
    }

    # parcours des élèves
    for _, televe in context:
        eleid = televe.attrib['ELEVE_ID']
        sortie = televe.find('DATE_SORTIE')
        if sortie != None and sortie.text != '':
            continue
        # attributs non initialisés
        eleve = {'int_id':unicode(eleid)}
        for cle, balise in mapping.items():
            try:
                clean_text = replace_more_cars(televe.find(balise).text)
                eleve[cle] = unicode(clean_text)
            except:
                pass
        if not eleve.get('civilite', ''):
            log.infolog("Attribution arbitraire d'une civilité à l'élève : %s %s" % (
                         str(eleve['prenom']), str(eleve['nom'])))
            eleve['civilite'] = u'1'
        if eleve.get('regime', ''):
            # mapping du régime de l'élève selon la nomenclature Sconet
            if eleve['regime'].rstrip() in REGIMES:
                eleve['regime'] = unicode(REGIMES[eleve['regime'].rstrip()])
        try:
            Eleve(store=store, **eleve)
            num += 1
            if num % DEBUG_NUM == 0:
                log.debuglog("%d élèves lus..." % num)
        except TypeError, msg:
            log.infolog("Erreur sur l'élève %s : %s" % (eleid, msg))
            continue
    log.infolog("TOTAL : %d élèves" % num)

    # affectation des élèves
    context, _ = parse_xml(eleve_file, 'STRUCTURES_ELEVE', check=False)
    num = 0
    for _, tstruct in context:
        eleid = tstruct.attrib['ELEVE_ID']
        eleve = store.findFirst(Eleve, Eleve.int_id==unicode(eleid))
        if eleve is None:
            continue
        for tstr in tstruct.getiterator('STRUCTURE'):
            type_struct = str(tstr.find('TYPE_STRUCTURE').text)
            if type_struct == 'D':
                # c'est une classe
                nom_classe = replace_cars(tstr.find('CODE_STRUCTURE').text)
                if nom_classe.lower() == 'inactifs':
                    # élèves inactifs (cf. #913)
                    continue
                classe = store.findFirst(Classe, Classe.nom==unicode(nom_classe))
                eleve.classe = classe
                eleve.niveau = classe.niveau
                num += 1
            elif type_struct == 'G':
                # c'est un groupe (-> option)
                nom_groupe = replace_cars(tstr.find('CODE_STRUCTURE').text)
                groupe = store.findFirst(Groupe, Groupe.nom==unicode(nom_groupe))
                JointureGroupeUser(store=store, groupe=groupe, user=eleve)
                num += 1
            else:
                log.infolog("Type de structure inconnu :", type_struct)
            if num % DEBUG_NUM == 0:
                log.debuglog("%d affectations lues..." % num)
    log.infolog("TOTAL : %d affectations d'élèves" % num)


def parse_sco_responsables(store, responsable_file):
    """
    parsing des responsables depuis Sconet
    """
    num = 0
    log.infolog("Lecture des responsables...", title=True)
    context, responsable_file = parse_xml(responsable_file, 'PERSONNE')
    # liste des responsables
    for _, tresp in context:
        rid = tresp.attrib['PERSONNE_ID']
        responsable = {'int_id':unicode(rid)}
        mapping = {'nom':'NOM',
              'prenom':'PRENOM',
              'mail':'MEL',
              'telephone':'TEL_PERSONNEL',
              'civilite':'LC_CIVILITE',
              'id_adresse':'ADRESSE_ID',
              'tel_portable':'TEL_PORTABLE',
              'tel_pro':'TEL_PROFESSIONNEL',
        }
        for cle, balise in mapping.items():
            try:
                clean_text = replace_more_cars(tresp.find(balise).text)
                responsable[cle] = unicode(clean_text)
            except:
                pass
        if responsable.has_key('civilite'):
            responsable['civilite'] = unicode(formate_civilite(str(responsable['civilite'])))
        try:
            Responsable(store=store, **responsable)
            num += 1
            if num % DEBUG_NUM == 0:
                log.debuglog("%d responsables lus..." % num)
        except TypeError, msg:
            log.infolog("Erreur sur le responsable %s : %s" % (rid, msg))
    log.infolog("TOTAL : %d responsables élèves" % num)

    # adresses des responsables
    num = 0
    context, _ = parse_xml(responsable_file, 'ADRESSE', check=False)
    for _, taddr in context:
        aid = taddr.attrib['ADRESSE_ID']
        adresse = {'int_id':unicode(aid)}
        mapping = {'code_postal':'CODE_POSTAL',
                   'ville':'LIBELLE_POSTAL',
                   'pays':'LL_PAYS'}
        for cle, balise in mapping.items():
            try:
                clean_text = replace_cars(taddr.find(balise).text)
                adresse[cle] = unicode(clean_text)
            except:
                pass
        adr = Adresse(store=store, **adresse)
        txt_addr = []
        for balise in ['LIGNE1_ADRESSE', 'LIGNE2_ADRESSE',
                'LIGNE3_ADRESSE', 'LIGNE4_ADRESSE']:
            try:
                clean_text = replace_cars(taddr.find(balise).text)
                if clean_text != '':
                    txt_addr.append(unicode(clean_text))
            except:
                pass
            if txt_addr != []:
                # FIXME "\n" ?
                adr.adresse = unicode("\n".join(txt_addr))
        resp = store.findFirst(Responsable,
                Responsable.id_adresse==unicode(aid))
        if resp is None:
            # l'adresse n'est à personne
            continue
        resp.adresse = adr
        num += 1
        if num % DEBUG_NUM == 0:
            log.debuglog("%d adresse lues..." % num)
    log.infolog("TOTAL : %d adresses de responsables" % num)

    # jointures responsable/élève
    num = 0
    context, _ = parse_xml(responsable_file, 'RESPONSABLE_ELEVE', check=False)
    for _, tjoint in context:
        resp_legal = tjoint.find('RESP_LEGAL').text
        if resp_legal == '0':
            # on ne garde que les responsables 1 et 2 (ref #1163)
            continue
        eleve_id = tjoint.find('ELEVE_ID').text
        responsable_id = tjoint.find('PERSONNE_ID').text
        eleve = store.findFirst(Eleve,
                Eleve.int_id==unicode(eleve_id))
        if eleve is None:
            continue
        responsable = store.findFirst(Responsable,
                Responsable.int_id==unicode(responsable_id))
        if responsable is None:
            log.infolog("responsable n°%s non trouvé" % responsable_id)
            continue
        jointure = dict(eleve=eleve, responsable=responsable)
        mapping = {'resp_legal':'RESP_LEGAL',
                   'resp_financier':'RESP_FINANCIER',
                   'resp_paiement':'PERS_PAIEMENT',
                   'pers_contact':'PERS_CONTACT',
                   'code_parente':'CODE_PARENTE',}
        for cle, balise in mapping.items():
            try:
                clean_text = replace_cars(tjoint.find(balise).text)
                jointure[cle] = unicode(clean_text)
            except:
                pass
        JointureResponsableEleve(store=store, **jointure)
        num += 1
        if num % DEBUG_NUM == 0:
            log.debuglog("%d liens responsable/élève lus..." % num)
    log.infolog("TOTAL : %d liens responsable/élève" % num)


##########################################
# Extraction STS Professeurs
# Fixhier :
# - sts_emp_$RNE$_$ANNEE$.xml
##########################################
def _parse_service(tree):
    """
    lecture des services d'une classe ou d'une option
    """
    tservs = tree.find('SERVICES')
    if tservs is None:
        return []
    machin = []
    for tserv in tservs.getiterator('SERVICE'):
        type_cours = tserv.attrib['CODE_MOD_COURS']
        #matiere = t.ok_groupe(code_matieres[tserv.attrib['CODE_MATIERE']], 'm')
        matiere = tserv.attrib['CODE_MATIERE']
        tens = tserv.find('ENSEIGNANTS')
        for ten in tens.getiterator('ENSEIGNANT'):
            # id enseignant
            #print "le prof", ten.attrib['ID'], "enseigne", matiere, "c'est", type_cours
            machin.append([ten.attrib['ID'], matiere, type_cours])
    return machin

def _parse_division_appartenance(tree):
    """
    lecture des divisions affectées à une option
    """
    tdivs = tree.find('DIVISIONS_APPARTENANCE')
    if tdivs is None:
        return []
    div = []
    for tdiv in tdivs.getiterator('DIVISION_APPARTENANCE'):
        div.append(tdiv.attrib['CODE'])
    return div

def parse_sts_profs(storage, sts_file):
    """
    parsing des professeurs depuis sts
    """
    num = 0
    log.infolog("Lecture des personnels", title=True)
    context, sts_file = parse_xml(sts_file, 'MATIERE')
    code_matieres = {}
    # lien code/libellé pour les matières
    for _, tcmat in context:
        code = tcmat.attrib['CODE']
        mat = tcmat.find('CODE_GESTION').text
        lib = tcmat.find('LIBELLE_EDITION').text
        code_matieres[code] = (mat, lib)

    # liens prof-classes, prof-matieres, prof-options
    classes = {}
    matieres = {}
    options = {}

    # -- parcours des classes -- #
    context, _ = parse_xml(sts_file, 'DIVISION', check=False)
    for _, tmat in context:
        code_classe = tmat.attrib['CODE']
        nom = unicode(replace_cars(code_classe))
        my_classe = storage.findOrCreate(EnsClasse, nom=nom)
        for prof, mat, _ in _parse_service(tmat):
            classes.setdefault(prof, []).append(my_classe)
            if code_matieres.has_key(mat):
                matiere = code_matieres[mat]
                nom = unicode(replace_cars(matiere[0]))
                desc = unicode(replace_cars(matiere[1]))
                my_mat = storage.findOrCreate(Matiere, nom=nom, description=desc)
                matieres.setdefault(prof, []).append(my_mat)
            else:
                log.infolog("matière %s inconnue" % mat)

    # -- parcours des groupes (options) -- #
    context, _ = parse_xml(sts_file, 'GROUPE', check=False)
    for _, tgrp in context:
        code_groupe = tgrp.attrib['CODE']
        nom = unicode(replace_cars(code_groupe))
        desc = unicode(replace_cars(tgrp.find('LIBELLE_LONG').text))
        my_groupe = storage.findOrCreate(Groupe, nom=nom, description=desc)
        my_classes = []
        for div in _parse_division_appartenance(tgrp):
            nom = unicode(replace_cars(div))
            my_classes.append(storage.findOrCreate(EnsClasse, nom=nom))
        for prof, mat, _ in _parse_service(tgrp):
            options.setdefault(prof, []).append(my_groupe)
            for my_classe in my_classes:
                # on ajoute les classes touchées par l'option au prof
                classes.setdefault(prof, []).append(my_classe)
            if code_matieres.has_key(mat):
                matiere = code_matieres[mat]
                nom = unicode(replace_cars(matiere[0]))
                desc = unicode(replace_cars(matiere[1]))
                my_mat = storage.findOrCreate(Matiere, nom=nom, description=desc)
                matieres.setdefault(prof, []).append(my_mat)
            else:
                log.infolog("matière %s inconnue" % mat)

    mapping = {'NOM_USAGE':'nom',
               'PRENOM':'prenom',
               'DATE_NAISSANCE':'date',
               'CIVILITE':'civilite',
               'NOM_PATRONYMIQUE':'nom_patronymique',
    }
    context, _ = parse_xml(sts_file, 'INDIVIDU', check=False)
    for _, ind in context:
        #<INDIVIDU ID='2453' TYPE='epp'>
        profid = ind.attrib['ID']
        professeur = {'int_id':unicode(profid)}
        fonction = ind.find('FONCTION')
        if fonction is None:
            log.infolog("(pas de fonction pour le personnel %s)" % profid)
            continue
        for balise, cle in mapping.items():
            value = ind.find(balise)
            if value is None:
                professeur[cle] = u''
            else:
                clean_text = replace_more_cars(value.text)
                professeur[cle] = unicode(clean_text)
        if not_empty(professeur, 'date'):
            my_date = formate_date(str(professeur['date']).replace('-', ''))
            professeur['date'] = unicode(my_date)
        else:
            # date de naissance arbitraire #1730
            professeur['date'] = u'01/01/0001'
        if is_empty(professeur, 'civilite'):
            # civilité arbitraire #2599
            log.infolog("Attribution arbitraire d'une civilité au personnel : %s %s" % (
                         str(professeur['prenom']), str(professeur['nom'])))
            professeur['civilite'] = u'1'
        if fonction.text == 'ENS':
            # c'est un enseignant !
            # on regarde si il est prof principal
            principal = []
            tprincs = ind.findall('PROFS_PRINC/PROF_PRINC')
            for tprinc in tprincs:
                nom = unicode(replace_cars(tprinc.find('CODE_STRUCTURE').text))
                my_classe = storage.findOrCreate(EnsClasse, nom=nom)
                principal.append(my_classe)
            try:
                prof = Enseignant(store=storage, **professeur)
                num += 1
            except Exception, msg:
                log.infolog("Erreur sur l'enseignant %s : %s" % (profid, msg))
            # affectation des classes
            if classes.has_key(profid):
                for classe in classes[profid]:
                    if classe in principal:
                        is_principal = True
                    else:
                        is_principal = False
                    JointureClasseEnseignant(store=storage, classe=classe,
                            enseignant=prof, profprincipal=is_principal)

            # affectation des options
            if options.has_key(profid):
                for groupe in options[profid]:
                    JointureGroupeUser(store=storage, groupe=groupe,
                            user=prof)

            # affectation des matieres
            if matieres.has_key(profid):
                for mat in matieres[profid]:
                    storage.findOrCreate(JointureMatiereEnseignant,
                            matiere=mat, enseignant=prof)
        else:
            # c'est un administratif
            try:
                admin = Administratif(store=storage, **professeur)
                nom = unicode(replace_cars(fonction.text))
                groupe = storage.findOrCreate(Service, nom=nom)
                admin.groupe = groupe
                num += 1
            except Exception, msg:
                log.infolog("Erreur sur le personnel %s : %s" % (profid, msg))
        if num % DEBUG_NUM == 0:
            log.debuglog("%d personnels lus..." % num)
    log.infolog("TOTAL : %d personnels" % num)

