#!/bin/bash

DRYRUN=0
ENCODE=0
LOGFILE="/tmp/$(basename ${0})-run-$(date +'%Y-%m-%d-%Hh%Mmn%Ss').log"
LogUsersNoExist="/tmp/UsersNoExist-run-$(date +'%Y-%m-%d-%Hh%Mmn%Ss').log"

function msg()
{
	echo ${@} | tee -a ${LOGFILE}
}

function usage()
{
	echo "usage: $(basename $0) -i inputfile.txt inputfile.txt -b /var/samba/dbfile.ldb"
	echo "     options : -s define SAM DB file (default: /var/lib/samba/private/sam.ldb)"
	echo "     options : -o default OU for created groups (without the domainDN part)"
	echo "     options : -l log file (default $(basename ${0})-run-<DATE>-<TIME>.log)"
	echo "     options : -S Modification or not SID => -S OUI|O|YES|Y (default: NO)"
	#echo "     options : -e enable base64 encoding of the objectSID"
	#echo "     options : -d dry run don't do anything and don't cleanup temporary files"
	return 0
}

function getGroupDN()
{
	group=$1
    dbfile=$2
	ldapreq="(&(objectClass=group)(sAMAccountName=${group}))"
	cmd="ldbsearch -H ${dbfile} ${ldapreq} samaccountname"
	res=($(${cmd}))
	index=0
	for entry in ${res[@]}
	do
		if [[ ${entry} =~ ^dn: ]]
		then
			dn=${res[$((index+1))]}
			break
		fi
		((index++))
	done
	echo ${dn}
}

function addMembers()
{
	local grp=${1}
	local mbrs=${2}
	local cmd="samba-tool group addmembers"
	local opts=""

	user_exist=""
	if [[ -n ${mbrs} ]]
	then
		msg "INFO: Members of the group ${grp} => [${mbrs}]"
		#debut agu
		#on supprime dans les membres du groupe les comptes n'existant pas
		mbr=(${mbrs//,/ })
		for ((i=0 ; ${#mbr[@]} - $i ; i++))
		do
			#les comptes n'existant pas vont dans le fichier ${LogUsersNoExist}
			res=$(id ${mbr[i]}>/dev/null 2>> ${LogUsersNoExist})
			ret=${?}
			if [[ ${ret} -eq 0 ]]	# test existence du compte (=0 le compte existe)
			then
				#on separe les comptes dans la liste par des virgules
				if [[ -z ${user_exist} ]]
				then
					user_exist=${mbr[i]}	#Pour le 1er on ne met pas de virgule avant
				else
					user_exist="$user_exist,${mbr[i]}"
				fi
			fi
		done
	fi
	if [[ -n ${user_exist} ]]
	then
		msg "INFO: Adding effective members to group ${grp} => ${user_exist}"
		res=$(${cmd} ${opts} ${grp} ${user_exist})	#ajout des comptes existant au groupe
		return ${?}
	else
		#le groupe a aucun membre
		msg "INFO: Group ${grp} empty"
		return 0
	fi
	#fin agu

}

#
# Update Group SID and group members
# $1: group name
# $2: group SID
# $3: group members
#
function updateGroup()
{
	msg "INFO: Updating group ${1}"
	local dn=$(getGroupDN $1 ${SAMFILE})

	# Building LDAP request for SID update
	if [[ -n ${dn} ]]
	then
		case $SID in
			OUI|O|YES|Y)
				echo "INFO: ******** modification SID et ajout mail dans attribut extensionName ********"
				local ldif=$(mktemp)
				local cmd="ldbmodify -v -H ${DBFILE}"
				local request="dn: ${dn}\n"
				
				request="${request}changetype: modify\n"
				request="${request}replace: objectSid\n"
				request="${request}objectSid: ${2}\n"
				[[ -n ${4} ]] && request="${request}extensionName: AMANDE.Liste: mail=${4}"
				echo -e ${request} >> ${ldif}

				res=$(${cmd} ${ldif})
				ret=${?}
				if [[ ${ret} -eq 0 ]]
				then
					msg "INFO: New SID ${2}"
					msg "INFO: Ajout extensionName: AMANDE.Liste: mail=${4}"
					rm ${ldif}
					addMembers "${1}" "${3}"
					local aret=${?}
					if [[ ${aret} -ne 0 ]]
					then
						msg "ERROR: Adding members for group ${1} [${3}]"
					fi
					return ${aret}
				else
					return ${ret}
				fi
				;;
			*)
				echo "INFO: ************************* PAS de modification SID *********************"
				# On n execute pas la modification de SID :
				#	 request="${request}replace: objectSid\n"
				#	 request="${request}objectSid: ${2}\n"
				# et on modifie seulement l'attribut extensionName
				#    en lui précisant l'adresse mail du groupe si elle existe 
				echo "INFO: ******** Ajout mail dans attribut extensionName ********"
				local ldif=$(mktemp)
				local cmd="ldbmodify -v -H ${DBFILE}" 
				local request="dn: ${dn}\n"
				
				request="${request}changetype: modify\n"
				request="${request}replace: extensionName\n"
				[[ -n ${4} ]] && request="${request}extensionName: AMANDE.Liste: mail=${4}"
				echo -e ${request} >> ${ldif}

				res=$(${cmd} ${ldif})
				ret=${?}
				if [[ ${ret} -eq 0 ]]
				then
					msg "INFO: Pas de modification de SID"
					msg "INFO: Ajout extensionName: AMANDE.Liste: mail=${4}"
					rm ${ldif}
					addMembers "${1}" "${3}"
					local aret=${?}
					if [[ ${aret} -ne 0 ]]
					then
						msg "ERROR: Adding members for group ${1} [${3}]"
					fi
					return ${aret}
				else
					return ${ret}
				fi
				;;
		esac
    else
		return 123
	fi
}

function createGroup()
{
	msg "INFO: Creating group ${1}"
	local ret=0
	local res=""
	local cmd="samba-tool group add"
	local opts=""

	[[ -n ${DEFAULTOU} ]] && opts="${opts} --groupou=${DEFAULTOU}"
	
	res=$(${cmd} ${opts} ${name})
	ret=${?}

	return ${ret}
}

while getopts :i:b:s:o:l:S:hde FLAG; do
	case $FLAG in
		i) INPUT=$OPTARG ;;
		b) DBFILE=$OPTARG ;;
		s) SAMFILE=${OPTARG} ;;
		e) ENCODE=1 ;;
		d) DRYRUN=1 ;;
		o) DEFAULTOU=${OPTARG} ;;
		l) LOGFILE=${OPTARG} ;;
		S) SID=$OPTARG ;;
		h) usage
		   exit 1
		   ;;
		\?) #unrecognized option - show help
			echo -e \\n"Option -${BOLD}$OPTARG${NORM} not allowed."
			usage
			exit 2
			;;
	esac
done

SID=${SID^^}

if [[ -z ${INPUT} ]]
then
	usage
	echo "Input files are mandatory"
	exit 1
fi

if [[ -z ${DBFILE} ]]
then
	usage
	echo "Database file is mandatory"
	exit 1
fi

[[ -z ${SAMFILE} ]] && SAMFILE="/var/lib/samba/private/sam.ldb"

SMBGRPS=($(samba-tool group list))
WORKED=()

#debut agu
# on efface (si il existe) le fichier ou sont stockés les comptes (de tous les groupes) n existant pas
# On le fait ici car la boucle ci-dessous passe en revue tous les groupes
# on aura surement des doublons (compte n existant pas mais present dans differents groupes)
rm -f ${LogUsersNoExist}
#fin agu

for file in ${INPUT}
do
	msg "INFO: Working on file : ${file} "

	while read -r line
	do
		data=(${line//:/ })
		name=${data[0]}
		sid=${data[1]}
		members=${data[2]}
		mail=${data[3]}

		[[ ${members} == "mail" ]] && members=""

		if [[ " ${WORKED[*]} " == *" ${name} "*  ]]
		then
			msg "WARNING: DOUBLE ENTRY [${name}][${file}]"
		else
			if [[ ! " ${SMBGRPS[*]} " == *" ${name} "*  ]]
			then
				createGroup "${name}" "${sid}" "${members}" "${mail}"
				[[ ${?} -ne 0 ]] && msg "ERROR: ${name} group creation failed"
			fi

			updateGroup "${name}" "${sid}" "${members}" "${mail}"
			if [[ ${?} -eq 0 ]]
			then
				WORKED+=(${name})
			else
				msg "ERROR: Update on group ${name} failed"
			fi
		fi
	done < "${file}"
	echo
done
