##./#REQUIRES -Version 2.0

<#
.SYNOPSIS
    Script de jonction au domaine des machines windows
.DESCRIPTION
    ...
.NOTES
    File Name      : joinDom.ps1
    Author         : Philippe Caseiro (pcaseiro@cadoles.com)
    Requires       : PowerShell V2 CPT3
.EXAMPLE
    Example 1
.EXAMPLE
    Example 2
#>

function itemPropertyExists($key, $itemName)
{
    <#
        .DESCRIPTION
        Test if an registry item exists

        .PARAMETER key
        The registry key name.

        .PARAMETER itemName
        The item name.

        .EXAMPLE
        itemPropertyExists "HKLM:\SYSTEM\CurrentControlSet\Services\TCPIP\Parameter" "SearchList"
    #>
    try
    {
        $exists = Get-ItemProperty $key $itemName -ErrorAction SilentlyContinue
        if (($exists -eq $null) -or ($exists.Length -eq 0))
        {
            return $false
        }
        else
        {
            return $true
        }
    }
    catch
    {
        write-host "DEBUG RETURN FALSE in the catch"
        return $false
    }
}

function cleanWINS($WMI)
{
    <#
        .DESCRIPTION
        Clean and disable WINS Configuration

        .PARAMETER WMI
        The Win32_NetworkAdapterConfiguration WMI object

        .EXAMPLE
        $wmi = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "IPEnabled=TRUE"
        cleanWINS $wmi
    #>
   $WMI | Foreach-Object{ $_.SetWINSServer("$null","$null") }
}

function setDNS($WMI, $DNS)
{
    <#
        .DESCRIPTION
        Define the DNS main server IP address

        .PARAMETER WMI
        The Win32_NetworkAdapterConfiguration WMI object

        .PARAMETER DNS
        The server IP

        .EXAMPLE
        $wmi = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "IPEnabled=TRUE"
        setDNS $wmi "10.0.0.1"
    #>
    $WMI.SetDNSServerSearchOrder($DNS)
}

function setDomain($WMI, $domain)
{
    <#
        .DESCRIPTION
        Define the DNS domain name

        .PARAMETER WMI
        The Win32_NetworkAdapterConfiguration WMI object

        .PARAMETER domain
        The domain name

        .EXAMPLE
        $wmi = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "IPEnabled=TRUE"
        setDomain $wmi "10.0.0.1"
    #>
    $WMI.SetDNSDomain($domain)
}

function setSearchDomainSuffix($suffix)
{
    <#
        .DESCRIPTION
        Define more search domain suffix

        .PARAMETER suffix
        The domain suffix

        .EXAMPLE
        setSearchDomainSuffix "in.ac-test.fr"
    #>
    $regkey = "HKLM:\SYSTEM\CurrentControlSet\Services\TCPIP\Parameters"
    $item = "SearchList"

    if ( -Not (Test-Path $regkey) )
    {
        New-Item -Path ${regkey}
    }

    if ( itemPropertyExists $regkey $item )
    {
        Remove-ItemProperty -Path "$regkey" -Name $item
    }

    # Ajoute un suffixe de recherche dns par défaut
    #
    New-ItemProperty -Path $regkey -Name $item -Value $suffix
}


function setDHCP($WMI)
{
    <#
        .DESCRIPTION
        Enable DHCP for IP and DNS configuration

        .PARAMETER WMI
        The Win32_NetworkAdapterConfiguration WMI object

        .EXAMPLE
        $wmi = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "IPEnabled=TRUE"
        setDHCP $wmi
    #>
    $WMI.EnableDHCP()
    $WMI.SetDNSServerSearchOrder()
}

function setDynamicDNS($WMI, $FDRE, $DDRE )
{
    <#
        .DESCRIPTION
        Enable of disable "FullDNSRegistration" and "domainDnsRegistration"
        Coche "enregistrer dans le système DNS" et coche "utiliser le suffixe DNS pour cette connexion"

        .PARAMETER WMI
        The Win32_NetworkAdapterConfiguration WMI object

        .PARAMETER FDRE
        The FullDNSRegistrationEnabled ($true or $false)

        .PARAMETER DDRE
        The domainDnsRegistrationEnabled ($true or $false)

        .EXAMPLE
        $wmi = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "IPEnabled=TRUE"
        setDynamicDNS $wmi $true $true
    #>
    $WMI.SetDynamicDNSRegistration($FDRE, $DDRE)
}

function disableNetBios($WMI)
{
    <#
        .DESCRIPTION
        Disable NetBios

        .PARAMETER WMI
        The Win32_NetworkAdapterConfiguration WMI object

        .EXAMPLE
        $wmi = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "IPEnabled=TRUE"
        disableNetBios $wmi
    #>
    # Disable Netbios (2)
    $WMI.SetTcpipNetbios(2)
}

function cleanRegistry($toRemove)
{
    <#
        .DESCRIPTION
        Clean registry items

        .PARAMETER toRemove
        Hash @{ "key" = "value"; } of items to remove (Key = itemName, Value = Reg key url)

        .EXAMPLE
        $keystoremove = @{
            "lmcompatibilitylevel" = "hklm:\system\currentcontrolset\control\lsa";`
            "dnsnameresolutionrequired" = "hklm:\system\currentcontrolset\services\lanmanworkstation\parameters" ;`
            "domaincompatibilitymode" = "hklm:\system\currentcontrolset\services\lanmanworkstation\parameters" }
        cleanRegistry $keystoremove
    #>
    # Disable Netbios (2)
    foreach ($item in $toRemove.Keys)
    {
        $regkey = $toRemove.Get_Item($item)
        if ( itemPropertyExists "$regkey" $item )
        {
            write-host "Removing item property $item for key $regkey"
            Remove-ItemProperty -Path "$regkey" -Name $item
        }
        else
        {
            write-host "Item $regkey\$item does not exists ... skipped"
        }
    }
}

function changeDomain($ntDomain, $sethDomain, $sethUsername, $sethPassword)
{
    <#
        .DESCRIPTION
        change computer from old domain to new one

        .PARAMETER ntDomain
        Old domain name

        .PARAMETER sethDomain
        New Active Directory domain (Seth Eole AD module)

        .PARAMETER sethUsername
        Active Directory domain admin

        .PARAMETER sethPassword
        Active Directory domain admin password

        .EXAMPLE
        changeDomain 'scribe' 'ac-test.fr' 'admin' 'XXXXXXX'
    #>

#debut agu
# Suppression de cette partie car plus nécessaire de récupérer le compte logué
#    $currentuser_key = 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Authentication\LogonUI'
#    $newuser_key = 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon'
#    $currentuser = (Get-ItemProperty -Path $currentuser_key -Name 'LastLoggedOnUser').LastLoggedOnUser.split("\")[-1]
#    write-host "Utilisateur courant $currentuser"
#fin agu

    write-host "Change domain NT to AD"
    $computer.UnjoinDomainOrWorkgroup($null, $null, 0)
#    $computer.JoinDomainOrWorkgroup($sethDomain, $sethPassword, $sethUsername, $null, 3)
    $secpasswd = ConvertTo-SecureString "$sethPassword" -AsPlainText -Force
    $mycreds = New-Object System.Management.Automation.PSCredential ("$sethdomain\$sethUsername", $secpasswd)
    Add-Computer -DomainName "$sethDomain" -Credential $mycreds -PassThru -ErrorAction stop

#debut agu
# Suppression de cette partie car plus nécessaire de récupérer le compte logué
    # Remet le nom d'utilisateur
#    if ( itemPropertyExists "$newuser_key" 'DefaultDomainName' )
#    {
#        Remove-ItemProperty -Path "$newuser_key" -Name 'DefaultDomainName'
#    }
#    if ( itemPropertyExists "$newuser_key" 'DefaultUserName' )
#    {
#        Remove-ItemProperty -Path "$newuser_key" -Name 'DefaultUserName'
#    }
#    New-ItemProperty -Path "$newuser_key" -Name 'DefaultDomainName' -Value "$sethDomain"
#    New-ItemProperty -Path "$newuser_key" -Name 'DefaultUserName' -Value "$currentuser"
#fin agu

}

function reportMigrationStatus
{
    <#
        .DESCRIPTION
        Append message to file, creating it if non existent.

        .PARAMETER statusfile
        File to write message in.

        .PARAMETER message
        Message to write.

        .EXAMPLE
        reportMigrationStatus '\\FILER\rapport\migration_OK' 'my_computer`t192.168.0.10'
    #>
        param([string]$statusfile, [string]$message)
        if (Test-Path $statusfile)
        {
                $message | Add-Content $statusfile
        }
        else
        {
                $message | Set-Content $statusfile
        }
}

function getHypotheticalIP
{
    <#
        .DESCRIPTION
        Retrieve one IP from configured wired networks, excluding VirtualBox networks.

        .PARAMETER wmi
        Get-WmiObject -Class Win32_NetworkAdapter -Filter "NetConnectionStatus=2"

        .EXAMPLE
        $wmi = Get-WmiObject -Class Win32_NetworkAdapter -Filter "NetConnectionStatus=2"
        getHypotheticalIP $wmi
    #>
    param($wmi)
    $filteredNet = $wmi | Where-Object {($_.AdapterTypeId -eq 0) -and ($_.ServiceName -notlike "VBox*")}
    if ( $($filterdNet | measure).Count -gt 1 )
    {
        $netIndex = $filteredNet[0].InterfaceIndex
    }
    else
    {
        $netIndex = $filteredNet.InterfaceIndex
    }
    $candidateNet = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "InterfaceIndex=$netIndex"
    return $candidateNet.IPAddress[0]
}

function rebootHost()
{
    Restart-Computer -force
}

#debut agu
# Rajout de cette fonction pour les tests
function pause ($Message="Taper une touche pour continuer...")
{
    Write-Host -NoNewLine $Message
    $null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
    Write-Host ""
}
#fin agu

#
# Main function
#
function Main($args)
{
    # Loading configuration file
    if ($args[1])
    {
        $configFile = $args[1]
    }
    else
    {
        $configFile = "joinDom.psd1"
    }

    Import-LocalizedData -BindingVariable "conf" -FileName $configFile

    #debut agu
    # Suppression des fichiers sources pour effacer les traces
    Remove-Item joinDom.* -Force
    #fin agu

    $statusKO = Join-Path $conf.statusfolder "migration_KO"
    $statusOK = Join-Path $conf.statusfolder "migration_OK"

    # Initialize windows configuration components
    # WMI : Network object
    $wmi = Get-WmiObject -Class Win32_NetworkAdapterConfiguration -Filter "IPEnabled=TRUE"
    $netAdapters = Get-WmiObject -Class Win32_NetworkAdapter -Filter "NetConnectionStatus=2"
    # Computer : ComputerSystem object
    $hypotheticalIP = getHypotheticalIP $netAdapters
    $computer =  Get-WmiObject Win32_ComputerSystem
    $Date = Get-Date -Format g
    $message = $computer.Name + "`t" + $hypotheticalIP + "`t" + $Date + "`n"
    trap
    {
    $lastError = $Error[0]
    $lineError = $lastError.InvocationInfo.Line
        $msg = $message + $lastError + " (" + $lineError.trim() + ")"
        reportMigrationStatus $statusKO $msg
        exit 2
    }

        # If more than one active connection, do not try to be smart.
    $m = $wmi | measure
    $nbWMI = $m.Count
    if ($nbWMI -ne 1)
    {
        $msg = $message + "$nbWMI interface(s) disponible(s)."
        reportMigrationStatus $statusKO $msg
        exit 2
    }
    # Validating configuration
    if (!($computer.domain).toLower().Equals(($conf.ntDomain).ToLower()))
    {
        # The original domain setted in configuration is not the current domain
        write-host $computer.domain
        write-host "Current domain not" $conf.ntDomain
        $msg = $message + "Le domaine courant (" + $computer.domain + ") est différent du domaine cible (" + $conf.ntDomain + ")"
        reportMigrationStatus $statusKO $msg
    exit 2
    }
    else
    {
        # Clean and diable WINS configuration
            cleanWINS $wmi

        # Checking if the network mode of configuration is "static"
        if ( "static".Equals($conf.networkMode))
        {
            # Define new DNS
            setDNS $wmi $conf.dnsIP
            # Define new DNS Domain
            setDomain $wmi $conf.dnsDomain
            # Define new DNS Suffix and full Dynamic DNS registration
            setSearchDomainSuffix $conf.dnsSearchSuffix `
                                  $conf.FullDNSRegistrationEnabled `
                                  $conf.DomainDNSRegistrationEnabled
            # Enable Dynamic DNS
            setDynamicDNS $wmi $conf.FullDNSRegistrationEnabled $conf.DomainDNSRegistrationEnabled
        }
        elseif ( "dhcp".Equals($conf.networkMode) )
        {
            # The network mode is "DHCP" Enable DHCP for IP and DNS
            setDHCP $wmi
        }
        else
        {
            # The network mode is not supported
            write-host "Configuration error unknown network mode [" $conf.networkMode "]"
                        $msg = $message + "Mode réseau inconnu"
            reportMigrationStatus $statusKO $msg
            exit 2
        }
        # Diable NetBios
        disableNetBios $wmi

        # Clean registry keys and items defined in configuration
        cleanRegistry($conf.keysToRemove)
	
        # Moving computer to the new domain
        changeDomain $conf.ntDomain $conf.sethDomain $conf.sethUsername $conf.sethPassword

        reportMigrationStatus $statusOK $message

        # Final reboot
        if ( $conf.autoReboot )
        {
            rebootHost
        }
        else
        {
            write-host "Reboot automatique désactivé dans la configuration !"
            write-host "Pour finaliser la procédure il faut rebooter la machine !"
        }
    }
	
}

Main $args
exit 0
