#!/bin/bash

#
# Build Samba
#

EOLERELEASE="2.6.1"

BUILDDEPDS="acl attr autoconf bison build-essential \
debhelper dnsutils docbook-xml docbook-xsl flex gdb krb5-user \
libacl1-dev libaio-dev libattr1-dev libblkid-dev libbsd-dev \
libcap-dev libcups2-dev libgnutls-dev libjson-perl \
libldap2-dev libncurses5-dev libpam0g-dev libparse-yapp-perl \
libpopt-dev libreadline-dev perl perl-modules-5.22 pkg-config \
python-all-dev python-dev python-dnspython python-crypto \
xsltproc zlib1g-dev libgpgme11-dev python-gpgme python-m2crypto"

SMBDEPENDS="eole-dhcp"
SCRIPTDEPENDS="git"

SMBPKG="samba samba-common samba-common-bin libwbclient0 samba-dsbd-modules samba-libs samba-vfs-modules python-samba"

VERSION="4.5.1"
SOURCENAME="samba-${VERSION}"
SOURCEFILE="${SOURCENAME}.tar.gz"
SOURCEURL="https://download.samba.org/pub/samba/stable/${SOURCEFILE}"
GITURL="https://dev-eole.ac-dijon.fr/git/eole-ad-dc.git"
#GITBRANCH="release/${EOLERELEASE}"
GITBRANCH="feat-pass-sync"
CHECKSUM="9f77c562e9578ef68ea9b1f80bd2adebd93dc013"
WORKDIR="/tmp/samba-build"
ARCH=$(uname -i)

#BUILDOPT="${BUILDOPT} --prefix=/usr"
#BUILDOPT="${BUILDOPT} --mandir=/usr/share/man"
#BUILDOPT="${BUILDOPT} --sysconfdir=/etc"
#BUILDOPT="${BUILDOPT} --localstatedir=/var"
#BUILDOPT="${BUILDOPT} --with-piddir=/var/run"
#BUILDOPT="${BUILDOPT} --disable-rpath"
#BUILDOPT="${BUILDOPT} --disable-rpath-install"
#BUILDOPT="${BUILDOPT} --with-modulesdir=/usr/lib/${ARCH}-linux-gnu/samba/"
#BUILDOPT="${BUILDOPT} --libdir=/usr/lib/${ARCH}-linux-gnu/"
#BUILDOPT="${BUILDOPT} --bundled-libraries=NONE,pytevent,iniparser"
#BUILDOPT="${BUILDOPT} --builtin-libraries=ctdb,replace,ccan"
BUILDOPT="${BUILDOPT} --enable-fhs"
BUILDOPT="${BUILDOPT} --with-gpgme"

# Unsupported options
#BUILDOPT="${BUILDOPT} --disable-ntdb"

LOGFILE="${WORKDIR}/build-smb-$(date +'%d%m%Y-%H%M%S').log"

function buildDepends()
{
	run "apt-get update"
    run "apt-eole install ${BUILDDEPDS}"
    run "apt-eole install ${SMBDEPENDS}"
    return ${?}
}

function validateSum()
{
    local cfile=${1}
    local vsum=${2}
    local sum=$(sha1sum ${cfile} -b | cut -d ' ' -f 1)

    if [[ "${sum}" == "${vsum}" ]]
    then
        return 0
    else
        return 27
    fi
}

function getSources()
{
    if [[ -f "${WORKDIR}/${SOURCEFILE}" ]]
    then
        validateSum ${WORKDIR}/${SOURCEFILE} ${CHECKSUM}
        return ${?}
    else
        run "wget --directory-prefix ${WORKDIR} ${SOURCEURL}"
        if [[ ${?} -eq 0 ]]
        then
            validateSum ${WORKDIR}/${SOURCEFILE} ${CHECKSUM}
            return ${?}
        else
            return 1
        fi
    fi
}

function unpackSources()
{
    [[ ! -d ${WORKDIR}/sources ]] && mkdir -p ${WORKDIR}/sources/
	if [[ ! -d ${WORKDIR}/sources/${SOURCENAME} ]]
	then
		run "tar xzvf ${WORKDIR}/${SOURCEFILE} -C ${WORKDIR}/sources/"
		return ${?}
	fi
}

function removeSamba()
{
	for pkg in ${SMBPKG}
	do
		run "dpkg -r --force-all ${pkg}"
	done
}

function build()
{
    local cpu=$(nproc)
    local opts=""

    [[ ${cpu} -ge 2 ]] && opts="-j ${cpu}"

    cd ${WORKDIR}/sources/${SOURCENAME}
    run "./configure ${BUILDOPT}"
    run "make ${opts}"
	removeSamba
    run "make install"
}

function run()
{
    echo "Running : $@" | tee -a ${LOGFILE}
    $@ >> ${LOGFILE} 2>&1
    return ${?}
}

function error()
{
    echo "[ERROR][${1}"
    exit ${2}
}

function makeSeth()
{
    run "apt-eole install git"
	run "git clone ${GITURL} ${WORKDIR}/sources/eole-ad-dc"
	run "cd ${WORKDIR}/sources/eole-ad-dc"
	run "git checkout ${GITBRANCH}"

	DESTDIR='/'
	run "make install"
	run "cd ${WORKDIR}"
}

function addService()
{
	INIT="/etc/systemd/system/samba-ad-dc.service"
	WINBIND="/etc/systemd/system/winbind.service"
	cat <<EOF >${INIT}
[Unit]
Description=Samba4 AD DC
After=network.target remote-fs.target nss-lookup.target

[Service]
Type=forking
ExecStart=/usr/local/samba/sbin/samba -D -s /etc/samba/smb.conf
PIDFile=/usr/local/samba/var/run/samba/samba.pid

[Install]
WantedBy=multi-user.target
EOF

	run "systemctl daemon-reload"
	run "systemctl enable samba-ad-dc"
	run "systemctl start samba-ad-dc"
}

function fixInit()
{
	local initdir="/etc/init.d"
	local smb="smbd"
	local nmb="nmbd"
	local ad="samba-ad-dc"

	cat <<__EOF__ >${initdir}/${nmb}
#!/bin/sh
### BEGIN INIT INFO
# Provides:          nmbd
# Required-Start:    \$network \$local_fs \$remote_fs
# Required-Stop:     \$network \$local_fs \$remote_fs
# X-Start-Before:    smbd
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: start Samba NetBIOS nameserver (nmbd)
### END INIT INFO
exit 0
__EOF__

	chmod +x ${initdir}/${nmb}

	cat <<_EOF_ >${initdir}/${smb}
#!/bin/sh
### BEGIN INIT INFO
# Provides:          smbd
# Required-Start:    \$network \$local_fs \$remote_fs
# Required-Stop:     \$network \$local_fs \$remote_fs
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Should-Start:      slapd cups
# Should-Stop:       slapd cups
# Short-Description: start Samba SMB/CIFS daemon (smbd)
### END INIT INFO
exit 0
_EOF_

	chmod +x ${initdir}/${smb}

	cat <<EOF >${initdir}/${ad}
#! /bin/sh
### BEGIN INIT INFO
# Provides:          samba-ad-dc
# Required-Start:    $network $local_fs $remote_fs
# Required-Stop:     $network $local_fs $remote_fs
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: start Samba daemons for the AD DC
### END INIT INFO
exit 0
EOF

	chmod +x ${initdir}/${ad}
}

function fixPath()
{
	smbsbin="/usr/local/samba/sbin"
	smbbin="/usr/local/samba/bin"
	smbetc="/usr/local/samba/etc/samba"
	for bin in $(ls ${smbsbin}/*)
	do
		ln -s ${bin} /usr/sbin
	done

	for bin in $(ls ${smbbin}/*)
	do
		ln -s ${bin} /usr/bin
	done

	ln -s ${smbetc} /etc
}

#
# Fixing eole script bug in 2.6.1 release
#
function fixLibSamba()
{
	local libfile="/usr/lib/eole/samba4.sh"

	sed -i '177s/"$ADMIN_PASSWORD"/"${ADMIN_PASSWORD}" \\/' ${libfile}
	return ${?}
}

#
# MAIN
#

[[ ! -d ${WORKDIR} ]] && mkdir -p ${WORKDIR}
cd ${WORKDIR}

# Install build depends
buildDepends
[[ ${?} -ne 0 ]] && error "Problem during deps installation" 1

# Download sources and validate
getSources
if [[ ${?} -eq 1 ]]
then
    error "Problem downloading the file from ${SOURCEURL}" 2
elif [[ ${?} -eq 27 ]]
then
    error "Downloaded file sum is not ${CHECKSUM}" 3
else
    unpackSources
    [[ ${?} -ne 0 ]] && error "Unpacking problem for file ${WORKDIR}/${SOURCEFILE}" 4
fi

build
[[ ${?} -ne 0  ]] && error "Error during build" 5

makeSeth
addService
fixInit
fixPath
fixLibSamba
[[ ${?} -ne 0  ]] && error "Error fixing samba4.sh lib" 10
