#!/usr/bin/env python3
"""Hangman example
"""


from random import choice
import unicodedata
import re
from json import dumps, loads
from http.server import HTTPServer, SimpleHTTPRequestHandler
from urllib.parse import unquote
from tiramisu import RegexpOption, OptionDescription, Config, getapi, IntOption, UnicodeOption, BoolOption, ParamOption, Params
from tiramisu_web import TiramisuWeb


LANG = 'fr_FR'


DICT_FILE = '/usr/share/myspell/{}.dic'.format(LANG)
WORD_REGEXP = re.compile(r'^[a-z]{7,12}$')
PROPOSALS_LEN = 27
NB_PROPOSALS = 6


def remove_accent(word):
    """remove all accent"""
    word = unicodedata.normalize('NFD', word)
    return word.encode('ascii', 'ignore').decode()


def get_random_word():
    """get line randomly in myspell file
    """
    with open(DICT_FILE, 'r') as file_content:
        word = choice(file_content.readlines()).strip()
        if word.endswith('/S.'):
            word = word[:-3]
        if word.endswith('/X.'):
            word = word[:-3]
        if word.endswith('/F.'):
            word = word[:-3]
        if word.endswith('/a0p+') or word.endswith('/d0p+') or word.endswith('/a3p+'):
            word = word[:-5]
        if not WORD_REGEXP.search(remove_accent(word)):
            return get_random_word()
        return word


def display_uncomplete_word(word, *proposals):
    """display response with proposals
    """
    display = ['-'] * len(word)
    for idx, char in enumerate(remove_accent(word)):
        if char in proposals:
            display[idx] = word[idx]
    return ''.join(display)


def display_misses(word, *proposals):
    """display all proposals
    """
    if proposals[-1] == None:
        props = proposals[:-1]
    else:
        props = proposals
    ret = list(set(props) - set(list(remove_accent(word))))
    ret.sort()
    return ' '.join(ret)


def validate_misses(misses):
    if display_proposals_left(misses) == 0:
        raise ValueError('No more guest possible')


def display_proposals_left(misses):
    if not misses:
        return NB_PROPOSALS
    return max(NB_PROPOSALS - len(misses.split(' ')), 0)


def display_proposal(word, *proposals):
    if display_uncomplete_word(word, *proposals) == word:
        return False
    return display_proposals_left(display_misses(word, *proposals)) != 0


class ProposalOption(RegexpOption):
    __slots__ = tuple()
    _regexp = re.compile(r'^[a-z]$')
    _display_name = 'proposal'


tiramisu_web = None
def get_hangman_json(body=None):
    global tiramisu_web
    if tiramisu_web is None:
        options = []
        proposal = None
        word = UnicodeOption('word',
                             'Word',
                             properties=('hidden', 'force_store_value'),
                             callback=get_random_word)
        proposals = [ParamOption(word)]
        for idx in range(PROPOSALS_LEN):
            requires = [{'option': 'self', 'expected': None, 'action': 'hidden', 'inverse': True}]
            if proposal is not None:
                display = BoolOption('display{}'.format(idx),
                                     'Display {}'.format(idx),
                                     properties=('hidden',),
                                     callback=display_proposal,
                                     callback_params=Params(tuple(proposals)))
                options.append(display)
                requires.append({'option': proposal, 'expected': None, 'action': 'disabled'})
                requires.append({'option': display, 'expected': False, 'action': 'disabled'})

            proposal = ProposalOption('guess{}'.format(idx),
                                      'Guess {}'.format(idx),
                                      requires=requires)
            #FIXME maximum recursion ...
            #if proposals:
            #    proposal.impl_add_consistency('not_equal', proposals[0])

            proposals.append(ParamOption(proposal, True))
            options.append(proposal)
        #
        proposal_word = UnicodeOption('proposal_word',
                                      'Word',
                                      properties=('frozen',),
                                      callback=display_uncomplete_word,
                                      callback_params=Params(tuple(proposals)))
        misses = UnicodeOption('misses',
                               'Misses',
                               properties=('frozen',),
                               callback=display_misses,
                               callback_params=Params(tuple(proposals)),
                               validator=validate_misses)
        proposals_left = IntOption('proposals_left',
                                   'Proposals left',
                                   properties=('frozen',),
                                   callback=display_proposals_left,
                                   callback_params=Params(ParamOption(misses)))
        descr = OptionDescription('proposals', 'Suggesting letters', options)
        root = OptionDescription('root', 'root', [word, proposal_word, misses, proposals_left, descr])
        api = getapi(Config(root))
        tiramisu_web = TiramisuWeb(api, clearable='none', remotable='minimum')
    if body is None:
        values = tiramisu_web.get_jsonform()
    else:
        values = tiramisu_web.set_updates(body)
        from pprint import pprint
        pprint(values)
    return values



class RequestHandler(SimpleHTTPRequestHandler):
    """main
    """
    def do_GET(self):
        if self.path == '/variables':
            global tiramisu_web
            tiramisu_web = None
            self.send_response(200)
            self.send_header('Content-Type', 'application/json')
            self.end_headers()
            values = get_hangman_json()
            self.wfile.write(dumps(values).encode())
        else:
            if self.path == '/':
                self.path = '/src/index.html'
            elif self.path == '/example.html':
                self.path = '/src/example.html'
            elif self.path.startswith('/bower_components'):
                self.path = '/src' + self.path
            elif self.path.startswith('/javascript'):
                self.path = '/src' + self.path
            SimpleHTTPRequestHandler.do_GET(self)

    def do_POST(self):
        """Post data
        """
        content_len = int(self.headers.get('content-length', 0))
        body = loads(self.rfile.read(content_len))
        splitted_path = unquote(self.path).split('/', 4)
        print("POST on {}".format(self.path))
        from pprint import pprint
        pprint(body['updates'])
        if splitted_path[1] == 'variables':
            values = get_hangman_json(body)
            #assert list(api.value.mandatory_warnings()) == []
            #print(api.option.make_dict(flatten=True))

            text = "OK"
            type_ = 'info'
            #for value in values.get('updates', []):
            #    if value.get('invalid'):
            #        text = "Some variables are invalids"
            #        type_ = 'error'
            #        pprint(values)
            #        break

            self.send_response(200)
            self.send_header('Content-Type', 'application/json')
            self.end_headers()
            values['message'] = {'text': text, 'type': type_}
            self.wfile.write(dumps(values).encode())


def main(port=8000):
    """Launches the http server
    """
    print('Listening on localhost:%s' % port)
    server = HTTPServer(('', port), RequestHandler)
    server.serve_forever()


if __name__ == "__main__":
    main()
