# -*- coding: utf-8 -*-

import warnings
from copy import copy
from collections import OrderedDict
from tiramisu.error import ValueWarning, PropertiesOptionError, RequirementError, ConfigError
from tiramisu.option import DynSymLinkOption
try:
    from tiramisu.api import TIRAMISU_VERSION
    from tiramisu import Config
except ImportError:
    TIRAMISU_VERSION = 2
    from tiramisu.config import Config

if TIRAMISU_VERSION == 2:
    from tiramisu.option.option import _RegexpOption as RegexpOption
    def getapi(config):
        return config

    from .tiramisu2 import TiramisuWebRoot
    # only for tests
    from tiramisu.setting import groups
    from tiramisu.option import OptionDescription
    class MasterSlaves(OptionDescription):
        def __init__(self, *args, **kwargs):
            # force class name
            self.__class__.__name__ = 'OptionDescription'
            super(MasterSlaves, self).__init__(*args, **kwargs)
            self.impl_set_group_type(groups.master)

    def Params(args=None, kwargs={}):
        kwargs_ = {}
        if kwargs:
            for key, val in kwargs.items():
                kwargs_[key] = tuple([val])
        if args is not None:
            if not isinstance(args, list):
                args = [args]
            kwargs_[''] = tuple(args)
        return kwargs_

    def ParamOption(option, prop=False):
        return (option, prop)

    def ParamValue(val):
        return val

    def ParamContext():
        return None

else:
    from tiramisu.option.option import RegexpOption
    from .tiramisu3 import TiramisuWebRoot
    # only for tests
    from tiramisu import MasterSlaves, getapi
    from tiramisu import Params, ParamOption, ParamValue, ParamContext


TYPES = {'StrOption': 'string',
         'UnicodeOption': 'string',
         'IntOption': 'number',
         'FloatOption': 'number',
         'IPOption': 'string',
         'ChoiceOption': 'string',
         'BoolOption': 'boolean',
         'PasswordOption': 'password',
         'EmailOption': 'string',
         'UsernameOption': 'string',
         'FilenameOption': 'string',
         'PortOption': 'number',
         'DateOption': 'date',
         'DomainnameOption': 'domain'
        }
INPUTS = ['string',
          'number',
          'password',
          'domain']

ACTION_HIDE = ['hidden', 'disabled']


#return always warning (even if same warning is already returned)
warnings.simplefilter("always", ValueWarning)


def tiramisu_copy(val):
    return val


def _(val):
    return val


class Callbacks(object):
    def __init__(self, tiramisu_web):
        self.tiramisu_web = tiramisu_web
        self.clearable = tiramisu_web.clearable
        self.remotable = tiramisu_web.remotable

    def _iter_callback_params(self,
                              child,
                              path,
                              callback,
                              callback_params,
                              form,
                              options):
        self.tiramisu_web.manage_callbacks(self, path, child, options, callback, callback_params, form)

    def add(self,
            path,
            childapi,
            options,
            form,
            force_store_value):
        if not self.tiramisu_web.isoptiondescription(childapi):
            callback, callback_params = self.tiramisu_web.get_callbacks(childapi)
            if callback is not None:
                if force_store_value and self.clearable != 'all':
                    return
                #if callback_params == {}:
                #    if self.remotable == 'none':
                #        raise Exception('not remotable')
                #    form.setdefault(options[self.tiramisu_web.get_option(childapi)], {})['remote'] = True
                #else:
                self._iter_callback_params(self.tiramisu_web.get_option(childapi),
                                           path,
                                           callback,
                                           callback_params,
                                           form,
                                           options)


class Consistencies(object):
    def __init__(self, tiramisu_web):
        self.not_equal = []
        self.options = {}
        self.tiramisu_web = tiramisu_web

    def add(self, path, childapi, form):
        child = self.tiramisu_web.get_option(childapi)
        if isinstance(child, DynSymLinkOption):
            child = child._impl_getopt()
        self.options[child] = path
        if not self.tiramisu_web.isoptiondescription(childapi):
            for consistency in self.tiramisu_web.get_consistencies(childapi):
                if consistency[0] == '_cons_not_equal':
                    options = []
                    for option in consistency[1]:
                        if TIRAMISU_VERSION == 3:
                            option = option()
                        options.append(option)
                    self.not_equal.append(options)

    def process(self, form):
        not_equals = []
        for not_equal in self.not_equal:
            not_equal_option = []
            for option in not_equal:
                not_equal_option.append(self.options[option])
            for idx in not_equal_option:
                options = copy(not_equal_option)
                options.remove(idx)
                form[idx]['not_equal'] = options


class Requires(object):
    def __init__(self, tiramisu_web):
        self.requires = {}
        self.options = {}
        self.tiramisu_web = tiramisu_web
        self.api = tiramisu_web.api
        self.remotable = tiramisu_web.remotable

    def add(self, path, childapi, form):
        #collect id of all options
        child = self.tiramisu_web.get_option(childapi)
        if isinstance(child, DynSymLinkOption):
            child = child._impl_getopt()
        self.options[child] = path
        current_action = None

        self.tiramisu_web.manage_requires(self,
                                          childapi,
                                          path,
                                          form,
                                          ACTION_HIDE,
                                          current_action)

    def is_remote(self, idx, form):
        if self.remotable == 'all':
            return True
        else:
            return form.get(idx) and form[idx].get('remote', False)

    def process(self, form, hidden_options):
        dependencies = {}
        for idx, values in self.requires.items():
            if 'default' in values:
                for option in values['default'].get('show', []):
                    #hidden_options[idx] = True
                    if not self.is_remote(option, form):
                        dependencies.setdefault(option,
                                                {'default': {}, 'expected': {}}
                                               )['default'].setdefault('show', [])
                        if not idx in dependencies[option]['default']['show']:
                            dependencies[option]['default']['show'].append(idx)
                for option in values['default'].get('hide', []):
                    #hidden_options[idx] = True
                    if not self.is_remote(option, form):
                        dependencies.setdefault(option,
                                                {'default': {}, 'expected': {}}
                                               )['default'].setdefault('hide', [])
                        if not idx in dependencies[option]['default']['hide']:
                            dependencies[option]['default']['hide'].append(idx)
            for expected, actions in values['expected'].items():
                if expected is None:
                    expected = ''
                for option in actions.get('show', []):
                    #hidden_options[idx] = True
                    if not self.is_remote(option, form):
                        dependencies.setdefault(option,
                                                {'expected': {}}
                                               )['expected'].setdefault(expected,
                                                                        {}).setdefault('show', [])
                        if idx not in dependencies[option]['expected'][expected]['show']:
                            dependencies[option]['expected'][expected]['show'].append(idx)
                for option in actions.get('hide', []):
                    #hidden_options[idx] = True
                    if not self.is_remote(option, form):
                        dependencies.setdefault(option,
                                                {'expected': {}}
                                               )['expected'].setdefault(expected,
                                                                        {}).setdefault('hide', [])
                        if idx not in dependencies[option]['expected'][expected]['hide']:
                            dependencies[option]['expected'][expected]['hide'].append(idx)
        for path, dependency in dependencies.items():
            form[path]['dependencies'] = dependency


class TiramisuWeb(TiramisuWebRoot):
    def __init__(self, api, root=None, clearable="all", remotable="minimum"):
        if isinstance(api, Config):
            api = getapi(api)
        self.set_api(api, root)
        self.read_write()
        #self.context = config.cfgimpl_get_context()
        self.form = {}
        self.requires = None
        self.callbacks = None
        self.hidden_options = {}
        #all, minimum, none
        self.clearable = clearable
        #all, minimum, none
        self.remotable = remotable

    def get_schema(self, root, subchildapi, init=False):
        def add_help(obj, childapi):
            hlp = self.get_option_help(childapi)
            if hlp is not None:
                obj['help'] = hlp

        schema = OrderedDict()
        if init:
            init = True
            self.form = OrderedDict()
            self.requires = Requires(self)
            self.consistencies = Consistencies(self)
            self.callbacks = Callbacks(self)
        else:
            init = False
        if subchildapi is None:
            subchildapi = self.get_unrestraint(root)
        for path, childapi in self.get_list(root, subchildapi):
            props = self.get_properties(childapi, path, apply_requires=False)
            self.requires.add(path, childapi, self.form)
            self.consistencies.add(path, childapi, self.form)
            self.callbacks.add(path, childapi, self.requires.options, self.form, 'force_store_value' in props)
            childapi_option = self.get_childapi_option(childapi)
            if self.isoptiondescription(childapi):
                properties = self.get_schema(path, childapi)
                obj = {'name': path,
                       'properties': properties}
                if self.get_option_ismasterslaves(childapi_option):
                    obj['type'] = 'array'
                else:
                    obj['type'] = 'object'
                obj['title'] = self.get_option_doc(childapi_option)
                add_help(obj, childapi)
                schema[path] = obj
            else:
                child = self.get_option_option(childapi_option)
                childtype = child.__class__.__name__
                if childtype == 'DynSymLinkOption':
                    childtype = child._impl_getopt().__class__.__name__
                web_type = TYPES.get(childtype, 'string')
                obj = {'name': path,
                       'title': self.get_option_doc(childapi_option),
                       'type': web_type}
                value = self.get_option_default(childapi_option)
                if value != [] and value is not None:
                    obj['value'] = value
                    if self.clearable != 'none':
                        self.form.setdefault(path, {})['clearable'] = True
                add_help(obj, childapi)

                is_multi = self.get_option_ismulti(childapi_option)
                if is_multi:
                    obj['isMulti'] = is_multi
                    default = self.get_option_get_default_multi(childapi_option)
                    if default not in [None, []]:
                        obj['default'] = default
                        if self.clearable != 'none':
                            self.form.setdefault(path, {})['clearable'] = True

                if self.get_option_issubmulti(childapi_option):
                    obj['isSubMulti'] = True

                if 'auto_freeze' in props:
                    obj['autoFreeze'] = True

                # apply Option differencies
                if self.clearable == 'all':
                    self.form.setdefault(path, {})['clearable'] = True
                if self.remotable == 'all' or self.get_option_has_dependency(childapi_option):
                    #if self.remotable == 'none':
                    #    raise Exception('{} is not remotable'.format(childapi.option.doc()))
                    self.form.setdefault(path, {})['remote'] = True
                if childtype ==  'ChoiceOption':
                    obj['enum'] = self.get_option_values(childapi)
                    empty_is_required = not self.get_isslave(childapi) and self.get_ismulti(childapi)
                    if (empty_is_required and not 'empty' in props) or \
                            (not empty_is_required and not 'mandatory' in props):
                        obj['enum'] = [''] + list(obj['enum'])
                    self.form.setdefault(path, {})['type'] = 'choice'
                elif web_type in INPUTS:
                    self.form.setdefault(path, {})['type'] = 'input'
                if web_type == 'number':
                    self.form.setdefault(path, {})['allowedpattern'] = '[0-9]'
                if isinstance(child, RegexpOption):
                    self.form.setdefault(path, {})['pattern'] = child._regexp.pattern
                if childtype == 'DomainnameOption':
                    self.form.setdefault(path, {})['pattern'] = child._get_extra('_domain_re').pattern
                if childtype == 'IPOption':
                    #FIXME only from 0.0.0.0 to 255.255.255.255
                    self.form.setdefault(path, {})['pattern'] = r'^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$'
                if childtype == 'FloatOption':
                    self.form.setdefault(path, {})['step'] = 'any'
                if childtype == 'PortOption':
                    self.form.setdefault(path, {})['min'] = child._get_extra('_min_value')
                    self.form.setdefault(path, {})['max'] = child._get_extra('_max_value')
                #FIXME:
                #'properties': child.impl_getproperties()
                #FIXME pour master/slave voir : http://ulion.github.io/jsonform/playground/?example=schema-array
                schema[path] = obj
        if init:
            self.requires.process(self.form, self.hidden_options)
            self.consistencies.process(self.form)
            del self.requires
            del self.consistencies
            return schema
        else:
            return schema

    # propriete:
    #   hidden
    #   mandatory
    #   editable

    # FIXME model:
    # #optionnel mais qui bouge
    # choices/suggests
    # warning
    #
    # #bouge
    # owner
    # properties


    def get_model(self,
                  path,
                  model=None,
                  subchildapi=None,
                  force_error_msg=None,
                  force_value=None,
                  hide=False,
                  masterslave=False,
                  order=None):
        if model is None:
            model = []
        if subchildapi is None:
            subchildapi = self.get_unrestraint(path)
        if not masterslave:
            # not a slave or a master
            for subpath, childapi in self.get_list(path, subchildapi):
                obj = self._gen_option_model(model,
                                             childapi,
                                             subpath,
                                             force_error_msg,
                                             force_value,
                                             hide,
                                             order)
                if not self.isoptiondescription(childapi):
                    self._get_value(childapi,
                                    force_error_msg,
                                    subpath,
                                    None,
                                    force_value,
                                    obj)
                if order is not None:
                    order.append(subpath)
                if list(obj.keys()) != ['key']:
                    model.append(obj)
        else:
            iterator = self.get_list(path, subchildapi)
            # master
            masterpath, childapi = next(iterator)
            obj = self._gen_option_model(model,
                                         childapi,
                                         masterpath,
                                         force_error_msg,
                                         force_value,
                                         hide,
                                         order)
            self._get_value(childapi,
                            force_error_msg,
                            masterpath,
                            None,
                            force_value,
                            obj)
            if order is not None:
                order.append(masterpath)
            if list(obj.keys()) != ['key']:
                model.append(obj)
            slaves = []
            for slave_path, slave_childapi in iterator:
                # slaves
                slaves.append(slave_path)
                obj = self._gen_option_model(model,
                                             slave_childapi,
                                             slave_path,
                                             force_error_msg,
                                             force_value,
                                             hide,
                                             order)
                if order is not None:
                    order.append(slave_path)
                if list(obj.keys()) != ['key']:
                    model.append(obj)
            for index in range(self.get_value_len(childapi, masterpath)):
                for slave_path in slaves:
                    slave_childapi = self.get_unrestraint_child_index(slave_path, index=index)
                    obj = self._gen_option_model(model,
                                                 slave_childapi,
                                                 slave_path,
                                                 force_error_msg,
                                                 force_value,
                                                 hide,
                                                 order,
                                                 index=index)
                    obj['index'] = index
                    self._get_value(slave_childapi,
                                    force_error_msg,
                                    slave_path,
                                    index,
                                    force_value,
                                    obj)
                    if set(obj.keys()) != set(['key', 'index']):
                        model.append(obj)
        return model


    def _gen_option_variable(self,
                             childapi,
                             path,
                             index,
                             hide):
        obj = {'key': path}
        isslave = self.get_isslave(childapi)
        if index is None and isslave:
            apply_requires = False
        else:
            apply_requires = True
        props = self.get_properties(childapi,
                                    path,
                                    index=index,
                                    apply_requires=apply_requires)
        if self.hidden_options.get(path, False):
            obj['hidden'] = True
        hidden = self.hidden_options.get(path, False) or hide
        if not isslave and self.get_ismulti(childapi):
            if 'empty' in props:
                obj['required'] = True
                props.remove('empty')
            if 'mandatory' in props:
                obj['needs_len'] = True
                props.remove('mandatory')
        elif 'mandatory' in props:
            obj['required'] = True
            props.remove('mandatory')
        if 'frozen' in props:
            props.remove('frozen')
        if 'hidden' in props:
            obj['hidden'] = True
            props.remove('hidden')
        if 'disabled' in props:
            obj['hidden'] = True
            props.remove('disabled')
        if props != set():
            obj['properties'] = list(props)
        return obj

    def _gen_option_model(self,
                          model,
                          childapi,
                          path,
                          force_error_msg,
                          force_value,
                          hide,
                          order,
                          index=None):
        if self.isoptiondescription(childapi):
            props = self.get_properties(childapi, path)
            obj = {'key': path}
            if props != set():
                obj['properties'] = list(props)
            if 'hidden' in props or 'disabled' in props:
                obj['hidden'] = True
            if not hide:
                hide = self.hidden_options.get(path, False)
            self.get_model(path,
                           model,
                           childapi,
                           force_error_msg=force_error_msg,
                           force_value=force_value,
                           hide=hide,
                           masterslave=self.get_option_ismasterslaves(self.get_childapi_option(childapi)),
                           order=order)
            if TIRAMISU_VERSION == 3:
                old_properties = childapi.option_bag.properties
                del childapi.option_bag.properties
            try:
                self.get_option(self.get_child(path))
            except PropertiesOptionError:
                obj['hidden'] = True
            if TIRAMISU_VERSION == 3:
                childapi.option_bag.properties = old_properties
        else:
            obj = self._gen_option_variable(childapi,
                                            path,
                                            index,
                                            hide)
        return obj

    def _get_value(self,
                   childapi,
                   force_error_msg,
                   path,
                   index,
                   force_value,
                   obj):
        props = set(obj.get('properties', []))
        hidden = obj.get('hidden', False),
        with warnings.catch_warnings(record=True) as warns:
            if TIRAMISU_VERSION == 3:
                old_properties = childapi.option_bag.config_bag.properties
            try:
                pass
                #if index is None:
                #    childapi = self.get_child(path)
                #else:
                #    childapi = self.get_child_index(path, index)
                if TIRAMISU_VERSION == 3:
                    del childapi.option_bag.config_bag.properties
                value = self.get_value(childapi,
                                       path,
                                       props,
                                       obj.get('index'),
                                       True)
            except (PropertiesOptionError, ValueError, RequirementError, ConfigError) as err:
                obj['hidden'] = True
                #if index is None:
                #    childapi = self.get_unrestraint(path)
                #else:
                #    childapi = self.get_unrestraint_child_index(path, index)
                if TIRAMISU_VERSION == 3:
                    childapi.option_bag.config_bag.properties = old_properties
                self._get_value_with_exception(obj,
                                               err,
                                               hidden)
                value = self.get_value(childapi,
                                       path,
                                       props,
                                       obj.get('index'),
                                       False)
            if TIRAMISU_VERSION == 3:
                childapi.option_bag.config_bag.properties = old_properties
        if force_error_msg is not None:  # and var_path == path:
            # here because could have PropertiesOptionError
            if TIRAMISU_VERSION == 2 and isinstance(force_value, list):
                force_value = force_value.copy()
            obj.update({'value': force_value,
                        'error': force_error_msg,
                        'invalid': True})
            return
        if value is not None and value != []:
            if TIRAMISU_VERSION == 2 and isinstance(value, list):
                value = list(value)
            obj['value'] = value
            obj['owner'] = self.get_owner(childapi, path, obj.get('index'))
        if warns != []:
            obj['warnings'] = "\n".join([str(warn.message) for warn in warns])
            obj['hasWarnings'] = True

    def _get_value_with_exception(self,
                                  obj,
                                  value,
                                  hidden):
        if isinstance(value, PropertiesOptionError):
            #props |= set(value.proptype)
            if not hidden:
                obj['error'] = str(value).decode('utf8')
                obj['invalid'] = True
            #else:
            #    value = self.get_default(childapi)
            #    if value is not None and value != []:
            #        if TIRAMISU_VERSION == 2 and isinstance(value, list):
            #            value = value.copy()
            #        ret['value'] = value
        else:
            obj['error'] = str(value)
            obj['invalid'] = True

    def get_form(self, form):
        ret = []
        buttons = []
        dict_form = OrderedDict()
        for form_ in form:
            if 'key' in form_:
                dict_form[form_['key']] = form_
            elif form_.get('type') == 'submit':
                if 'cmd' not in form_:
                    form_['cmd'] = 'submit'
                buttons.append(form_)
            else:
                raise Exception('unknown form {}'.format(form_))

        for key, form_ in self.form.items():
            form_['key'] = key
            if key in dict_form:
                form_.update(dict_form[key])
            ret.append(form_)
        ret.extend(buttons)
        return ret

    def apply_updates(self, oripath, body, model_ori):
        updates = body.get('updates', [])
        self.hidden_options = OrderedDict()
        #for option in model_ori:
        #    if option.get('hide', False):
        #        self.hidden_options[option['key']] = True
        for update in updates:
            path = update['name']
            if oripath is not None and not path.startswith(oripath):
                raise Exception('not in current area')
            childapi = self.get_child_index(path, None)
            childapi_option = self.get_childapi_option(childapi)
            if self.get_option_isslave(childapi_option):
                childapi = self.get_child_index(path, update['index'])
            if update['action'] == 'modify':
                with warnings.catch_warnings():
                    warnings.simplefilter("ignore")
                    self.mod_value(childapi, path, update.get('index'), update['value'])
            elif update['action'] == 'delete':
                self.del_value(childapi, path, update.get('index'))
            elif update['action'] == 'add':
                if self.get_option_ismulti(childapi_option):
                    self.add_value(childapi, path, update['value'])
                else:
                    raise Exception('only multi option can have action "add", but "{}" is not a multi'.format(path))
            else:
                raise Exception('unknown action')

    def get_jsonform(self, form=[]):
        rootpath = self.root
        ret = {
               'schema': self.get_schema(rootpath, None, True),
               'model': self.get_model(rootpath),
               'form': self.get_form(form)
        }
        #import pprint
        #pp = pprint.PrettyPrinter(indent=1)
        #pp.pprint(ret)

        return ret

    def set_updates(self, body):
        root_path = self.root
        if 'model' in body:
            old_model = body['model']
        else:
            old_model = self.get_model(root_path)
        self.apply_updates(root_path, body, old_model)
        order = []
        new_model = self.get_model(root_path,
                                   order=order)
        values = {'updates': list_keys(old_model, new_model, order),
                  'model': new_model}
        return values


def list_keys(model_a, model_b, ordered_key):
    model_a_dict = {}
    model_b_dict = {}
    for model in model_a:
        model_a_dict[model['key']] = model
    for model in model_b:
        model_b_dict[model['key']] = model

    keys_a = set(model_a_dict.keys())
    keys_b = set(model_b_dict.keys())

    keys = list(keys_a ^ keys_b)

    for key in keys_a & keys_b:
        keys_mod_a = set(model_a_dict[key].keys())
        keys_mod_b = set(model_b_dict[key].keys())
        if keys_mod_a != keys_mod_b:
            keys.append(key)
        else:
            for skey in keys_mod_a:
                if model_a_dict[key][skey] != model_b_dict[key][skey]:
                    keys.append(key)
                    break
    def sort_key(key):
        try:
            return ordered_key.index(key)
        except ValueError:
            return -1
    return sorted(keys, key=sort_key)
