# -*- coding: utf-8 -*-
from json import load, dump, loads, dumps
from os import listdir
from os.path import dirname, abspath, join, normpath, splitext, isfile
from tiramisu.option import OptionDescription
import sys

FILENAME = []
try:
    from tiramisu.api import TIRAMISU_VERSION
    from tiramisu import Config

    def fake_config(config):
        return config

    def make_dict(config):
        return config.config.dict()
except ImportError:
    TIRAMISU_VERSION = 2
    from tiramisu.config import Config
    class FakeApi:
        def __init__(self, config):
            self.config = config
            self.value = self

        def option(self, path, index=None):
            self.path = path
            self.index = index
            return self

        def set(self, value):
            self.config.setattr(self.path, value, index=self.index)

    def fake_config(config):
        return FakeApi(config)

    def make_dict(config):
        return config.make_dict(fullpath=True)


from tiramisu_web import TiramisuWeb
from data.unicode1 import get_description as get_description_unicode1
from data.unicode1_mod_value import get_description as get_description_unicode1_mod_value, \
                                    get_values as get_values_unicode1_mod_value
from data.unicode1_multi import get_description as get_description_unicode1_multi
from data.unicode1_multi_mod_value import get_description as get_description_unicode1_multi_mod_value, \
                                          get_values as get_values_unicode1_multi_mod_value
from data.unicode1_master_slaves_value import get_description as get_description_unicode1_master_slaves_value, \
                                              get_values as get_values_unicode1_master_slaves_value


def datapath():
    path_= join(normpath(dirname(abspath(__file__))), 'data')
    if path_ not in sys.path:
        sys.path.insert(1, path_)
    return path_


def list_data():
    #return ['choice1_requires.py']
    datadir = datapath()
    filenames = listdir(datadir)
    filenames.sort()
    ret = []
    for filename in filenames:
        if filename.endswith('.py') and not filename.startswith('__'):
            ret.append(filename)
    return ret


def load_config(filename,
             add_extra_od=False,
             remote='minimum',
             clearable='minimum',
             root=None):
    modulepath = splitext(filename)[0]
    mod = __import__(modulepath)
    descr = mod.get_description()
    if add_extra_od:
        descr = OptionDescription('root', '', [descr])
    config = Config(descr)
    if 'get_values' in dir(mod):
        mod.get_values(fake_config(config), add_extra_od)
    rootconfig = config
    if add_extra_od:
        if TIRAMISU_VERSION == 2:
            config = getattr(rootconfig, modulepath)

    tiramisu = TiramisuWeb(config,
                           remotable=remote,
                           clearable=clearable,
                           root=root)
    form = [{'title': 'Configurer',
             'type': 'submit'}]
    if 'get_form' in dir(mod):
        form.extend(mod.get_form(add_extra_od))
    values = loads(dumps(tiramisu.get_jsonform(form)))
    return values


def parse_expected(schema, all_options):
    for key, value in schema['properties'].items():
        if 'properties' in value:
            parse_expected(value, all_options)
        else:
            all_options.append(key)


def del_property(expected, prop):
    new_form = []
    for form in expected['form']:
        if prop in form:
            del form[prop]
        if list(form.keys()) != ['key']:
            new_form.append(form)
    return new_form


def add_property(expected, prop, prop_value):
    all_options = []
    for key, root in expected['schema'].items():
        if 'properties' in root:
            parse_expected(root, all_options)
        else:
            all_options.append(key)

    ordered_options = all_options.copy()
    new_form = {}
    buttons = []
    for form in expected['form']:
        if 'key' in form:
            form[prop] = prop_value
            new_form[form['key']] = form
            all_options.remove(form['key'])
        else:
            # for button
            buttons.append(form)
    for option in all_options:
        new_form[option] = {'key': option, prop: prop_value}
    ordered_form = []
    for option in ordered_options:
        ordered_form.append(new_form[option])
    ordered_form.extend(buttons)
    return ordered_form


def test_jsons():
    debug = False
    #debug = True
    datadir = datapath()
    if debug:
        print()
    #for clearable in ['minimum']:
    for clearable in ['minimum', 'none', 'all']:
        if debug:
            print('==> clearable', clearable)
        #for remote in ['all']:
        for remote in ['minimum', 'none', 'all']:
            if debug:
                print('  ==> remotable', remote)
            filenames = list_data()
            for filename in filenames:
                if filename in FILENAME:
                    continue
                modulepath = splitext(filename)[0]
                if debug:
                    print("    ", filename)
                values = load_config(filename,
                                     remote=remote,
                                     clearable=clearable)
                #
                if not isfile(join(datadir, modulepath + '.json')) and \
                        clearable == 'minimum' and \
                        remote == 'minimum':
                    with open(join(datadir, modulepath + '.json'), 'w') as fh:
                        dump(values, fh, indent=2)
                with open(join(datadir, modulepath + '.json'), 'r') as fh:
                    expected = loads(fh.read())
                if clearable == 'none':
                    expected['form'] = del_property(expected, 'clearable')
                if remote == 'none' and 'tiramisu' in expected:
                    del expected['tiramisu']
                if clearable == 'all':
                    expected['form'] = add_property(expected, 'clearable', True)
                if remote == 'all':
                    expected['form'] = add_property(expected, 'remote', True)
                    expected['form'] = del_property(expected, 'dependencies')
                    expected['form'] = del_property(expected, 'copy')
                # properties are unordered
                for model in expected['model']:
                    if 'properties' in model:
                        model['properties'] = set(model['properties'])
                for model in values['model']:
                    if 'properties' in model:
                        model['properties'] = set(model['properties'])
                #from pprint import pprint
                #pprint(values)
                #print('----------------')
                #pprint(expected)
                assert values == expected


def test_jsons_subconfig():
    debug = False
    #debug = True
    datadir = datapath()
    if debug:
        print()
    clearable = 'minimum'
    remote = 'minimum'
    filenames = list_data()
    for filename in filenames:
        if filename in FILENAME:
            continue
        modulepath = splitext(filename)[0]
        if debug:
            print("    ", filename)
        values = load_config(filename, add_extra_od=True, root=modulepath)
        #
        with open(join(datadir, modulepath + '.json'), 'r') as fh:
            expected = loads(fh.read())
        # properties are unordered
        for model in expected['model']:
            if 'properties' in model:
                model['properties'] = set(model['properties'])
        for model in values['model']:
            if 'properties' in model:
                model['properties'] = set(model['properties'])
        #add root
        def change_key(schema):
            new_schema = {}
            for key_schema, val_schema in schema.items():
                key = modulepath + '.' + key_schema
                val_schema['name'] = key
                if 'properties' in val_schema:
                    val_schema['properties'] = change_key(val_schema['properties'])
                new_schema[key] = val_schema
            return new_schema

        expected['schema'] = change_key(expected['schema'])
        for form in expected['form']:
            if 'key' in form:
                form['key'] = modulepath + '.' + form['key']
            if 'copy' in form:
                for idx, noteq in enumerate(form['copy']):
                    form['copy'][idx] = modulepath + '.' + noteq
            if 'not_equal' in form:
                for idx, noteq in enumerate(form['not_equal']):
                    form['not_equal'][idx] = modulepath + '.' + noteq
            if 'dependencies' in form:
                for dependency in form['dependencies'].values():
                    for val1 in dependency.values():
                        if isinstance(val1, list):
                            for idx, lst in enumerate(val1):
                                val1[idx] = modulepath + '.' + lst
                        else:
                            for val2 in val1.values():
                                if isinstance(val2, list):
                                    for idx, lst in enumerate(val2):
                                        val2[idx] = modulepath + '.' + lst

        for model in expected['model']:
            if 'key' in model:
                model['key'] = modulepath + '.' + model['key']
        #from pprint import pprint
        #pprint(values)
        #print('----------------')
        #pprint(expected)
        assert values == expected


#import cProfile
#cProfile.run("test_jsons()", 'pouet')
#import pstats
#stats = pstats.Stats('pouet')
#stats.sort_stats('cumulative')
#stats.print_stats()
#test_jsons()
def test_updates_simple_unicode_modify():
    descr = get_description_unicode1()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode': None}

            body = {'updates': [{"action": "modify",
                                 "name": "options.unicode",
                                 "value": "val"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['options.unicode'],
                              'model': [{'key': 'options.unicode',
                                         'owner': 'user',
                                         'value': 'val'}]}
            assert make_dict(config) == {'options.unicode': 'val'}


def test_updates_simple_unicode_modify_sub():
    descr = get_description_unicode1()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(OptionDescription('root', '', [descr]))
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1')
            assert make_dict(config) == {'unicode1.options.unicode': None}

            body = {'updates': [{"action": "modify",
                                 "name": "unicode1.options.unicode",
                                 "value": "val"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['unicode1.options.unicode'],
                              'model': [{'key': 'unicode1.options.unicode',
                                         'owner': 'user',
                                         'value': 'val'}]}
            assert make_dict(config) == {'unicode1.options.unicode': 'val'}


def test_updates_simple_unicode_mod_value_modify():
    descr = get_description_unicode1_mod_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_mod_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode': 'a'}

            body = {'updates': [{"action": "modify",
                                 "name": "options.unicode",
                                 "value": "val"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['options.unicode'],
                              'model': [{'key': 'options.unicode',
                                         'owner': 'user',
                                         'value': 'val'}]}
            assert make_dict(config) == {'options.unicode': 'val'}


def test_updates_simple_unicode_mod_value_modify_sub():
    descr = get_description_unicode1_mod_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_mod_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_mod_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_mod_value')
            assert make_dict(config) == {'unicode1_mod_value.options.unicode': 'a'}

            body = {'updates': [{"action": "modify",
                                 "name": "unicode1_mod_value.options.unicode",
                                 "value": "val"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['unicode1_mod_value.options.unicode'],
                              'model': [{'key': 'unicode1_mod_value.options.unicode',
                                         'owner': 'user',
                                         'value': 'val'}]}
            assert make_dict(config) == {'unicode1_mod_value.options.unicode': 'val'}


def test_updates_simple_unicode_mod_value_delete():
    descr = get_description_unicode1_mod_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_mod_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode': 'a'}

            body = {'updates': [{"action": "delete",
                                 "name": "options.unicode"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['options.unicode'],
                              'model': []}
            assert make_dict(config) == {'options.unicode': None}


def test_updates_simple_unicode_mod_value_delete_sub():
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_mod_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_mod_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_mod_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_mod_value')
            assert make_dict(config) == {'unicode1_mod_value.options.unicode': 'a'}

            body = {'updates': [{"action": "delete",
                                 "name": "unicode1_mod_value.options.unicode"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['unicode1_mod_value.options.unicode'],
                              'model': []}
            assert make_dict(config) == {'unicode1_mod_value.options.unicode': None}


def test_updates_simple_unicode_multi_modify():
    descr = get_description_unicode1_multi()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(OptionDescription('root', '', [descr]))
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_multi
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_multi')
            assert make_dict(config) == {'unicode1_multi.options.unicode': []}

            body = {'updates': [{"action": "add",
                                 "name": "unicode1_multi.options.unicode",
                                 "index": 0,
                                 "value": "val"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['unicode1_multi.options.unicode'],
                              'model': [{'key': 'unicode1_multi.options.unicode',
                                         'owner': 'user',
                                         'required': True,
                                         'value': ['val']}]}
            assert make_dict(config) == {'unicode1_multi.options.unicode': ['val']}


def test_updates_simple_unicode_multi_modify_sub():
    descr = get_description_unicode1_multi()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(OptionDescription('root', '', [descr]))
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_multi
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_multi')
            assert make_dict(config) == {'unicode1_multi.options.unicode': []}

            body = {'updates': [{"action": "add",
                                 "name": "unicode1_multi.options.unicode",
                                 "index": 0,
                                 "value": "val"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['unicode1_multi.options.unicode'],
                              'model': [{'key': 'unicode1_multi.options.unicode',
                                         'owner': 'user',
                                         'required': True,
                                         'value': ['val']}]}
            assert make_dict(config) == {'unicode1_multi.options.unicode': ['val']}


def test_updates_simple_unicode_multi_mod_value_delete():
    descr = get_description_unicode1_multi_mod_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_multi_mod_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode': ['c', 'd', 'e']}

            body = {'updates': [{"action": "delete",
                                 "name": "options.unicode"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['options.unicode'],
                              'model': [{'key': 'options.unicode',
                                         'owner': 'default',
                                         'required': True,
                                         'value': ['a', 'b']}]}
            assert make_dict(config) == {'options.unicode': ['a', 'b']}


def test_updates_simple_unicode_multi_mod_value_delete_sub():
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_multi_mod_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_multi_mod_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_multi_mod_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_multi_mod_value')
            assert make_dict(config) == {'unicode1_multi_mod_value.options.unicode': ['c', 'd', 'e']}

            body = {'updates': [{"action": "delete",
                                 "name": "unicode1_multi_mod_value.options.unicode"}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['unicode1_multi_mod_value.options.unicode'],
                              'model': [{'key': 'unicode1_multi_mod_value.options.unicode',
                                         'owner': 'default',
                                         'required': True,
                                         'value': ['a', 'b']}]}
            assert make_dict(config) == {'unicode1_multi_mod_value.options.unicode': ['a', 'b']}


def test_updates_simple_unicode_multi_mod_value_modify():
    descr = get_description_unicode1_multi_mod_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_multi_mod_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode': ['c', 'd', 'e']}

            body = {'updates': [{"action": 'modify',
                                 "name": 'options.unicode',
                                 "index": 1,
                                 "value": 'f'}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['options.unicode'],
                              'model': [{'key': 'options.unicode',
                                         'owner': 'user',
                                         'required': True,
                                         'value': ['c', 'f', 'e']}]}
            assert make_dict(config) == {'options.unicode': ['c', 'f', 'e']}


def test_updates_simple_unicode_multi_mod_value_modify_sub():
    descr = get_description_unicode1_multi_mod_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_multi_mod_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_multi_mod_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_multi_mod_value')
            assert make_dict(config) == {'unicode1_multi_mod_value.options.unicode': ['c', 'd', 'e']}

            body = {'updates': [{"action": 'modify',
                                 "name": 'unicode1_multi_mod_value.options.unicode',
                                 "index": 1,
                                 "value": 'f'}]}
            values = tiramisu.set_updates(body)

            assert values == {'updates': ['unicode1_multi_mod_value.options.unicode'],
                              'model': [{'key': 'unicode1_multi_mod_value.options.unicode',
                                         'owner': 'user',
                                         'required': True,
                                         'value': ['c', 'f', 'e']}]}
            assert make_dict(config) == {'unicode1_multi_mod_value.options.unicode': ['c', 'f', 'e']}


def test_updates_unicode1_master_slaves_value_delete_master():
    descr = get_description_unicode1_master_slaves_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_master_slaves_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test'],
                                              'options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "delete",
                                 "name": "options.unicode.unicode"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['options.unicode.unicode',
                                    'options.unicode.unicode1',
                                    'options.unicode.unicode2',
                                    'options.unicode.unicode3'],
                        'model': [{'key': 'options.unicode.unicode',
                                   'owner': 'default',
                                   'required': True,
                                   'value': ['val1', 'val2']},
                                  {'index': 0,
                                   'key': 'options.unicode.unicode2',
                                   'owner': 'default',
                                   'value': 'slave2'},
                                  {'index': 1,
                                   'key': 'options.unicode.unicode2',
                                   'owner': 'default',
                                   'value': 'slave2'}]}

            assert sorted(values['updates']) == sorted(expected['updates'])
            # only last 2 options is important
            assert values['updates'] == expected['updates']
            expected['updates'] = values['updates']
            assert values == expected
            assert make_dict(config) == {'options.unicode.unicode': ['val1', 'val2'],
                                              'options.unicode.unicode1': [None, None],
                                              'options.unicode.unicode2': ['slave2', 'slave2'],
                                              'options.unicode.unicode3': [None, None]}


def test_updates_unicode1_master_slaves_value_delete_master_sub():
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_master_slaves_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_master_slaves_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_master_slaves_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_master_slaves_value')

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "delete",
                                 "name": "unicode1_master_slaves_value.options.unicode.unicode"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['unicode1_master_slaves_value.options.unicode.unicode',
                                    'unicode1_master_slaves_value.options.unicode.unicode1',
                                    'unicode1_master_slaves_value.options.unicode.unicode2',
                                    'unicode1_master_slaves_value.options.unicode.unicode3'],
                        'model': [{'key': 'unicode1_master_slaves_value.options.unicode.unicode',
                                   'owner': 'default',
                                   'required': True,
                                   'value': ['val1', 'val2']},
                                  {'index': 0,
                                   'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'owner': 'default',
                                   'value': 'slave2'},
                                  {'index': 1,
                                   'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'owner': 'default',
                                   'value': 'slave2'}]}

            assert sorted(values['updates']) == sorted(expected['updates'])
            # only last 2 options is important
            assert values['updates'] == expected['updates']
            expected['updates'] = values['updates']
            assert values == expected
            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val1', 'val2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': [None, None],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['slave2', 'slave2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, None]}


def test_updates_unicode1_master_slaves_value_reduce_master():
    descr = get_description_unicode1_master_slaves_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_master_slaves_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test'],
                                              'options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "delete",
                                 "index": 1,
                                 "name": "options.unicode.unicode"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['options.unicode.unicode',
                                    'options.unicode.unicode1',
                                    'options.unicode.unicode2',
                                    'options.unicode.unicode3'],
                        'model': [{'key': 'options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3'],
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'}]}
            assert values == expected

            assert make_dict(config) == {'options.unicode.unicode': ['val3'],
                                              'options.unicode.unicode1': ['super1'],
                                              'options.unicode.unicode2': ['pas test'],
                                              'options.unicode.unicode3': [None]}


def test_updates_unicode1_master_slaves_value_reduce_master_sub():
    descr = get_description_unicode1_master_slaves_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_master_slaves_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_master_slaves_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_master_slaves_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_master_slaves_value')
            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "delete",
                                 "index": 1,
                                 "name": "unicode1_master_slaves_value.options.unicode.unicode"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['unicode1_master_slaves_value.options.unicode.unicode',
                                    'unicode1_master_slaves_value.options.unicode.unicode1',
                                    'unicode1_master_slaves_value.options.unicode.unicode2',
                                    'unicode1_master_slaves_value.options.unicode.unicode3'],
                        'model': [{'key': 'unicode1_master_slaves_value.options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3'],
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'}]}
            assert values == expected

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None]}


def test_updates_unicode1_master_slaves_value_add_master():
    descr = get_description_unicode1_master_slaves_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_master_slaves_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test'],
                                              'options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "add",
                                 "index": 0,
                                 "name": "options.unicode.unicode",
                                 "value": "val5"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['options.unicode.unicode',
                                    'options.unicode.unicode2'],
                        'model': [{'key': 'options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val4', 'val5'],
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'test',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 2,
                                   'value': 'slave2',
                                   'owner': 'default'}]}
            assert values == expected

            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4', 'val5'],
                                              'options.unicode.unicode1': ['super1', 'super2', None],
                                              'options.unicode.unicode2': ['pas test', 'test', 'slave2'],
                                              'options.unicode.unicode3': [None, 'super', None]}



def test_updates_unicode1_master_slaves_value_add_master_sub():
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_master_slaves_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_master_slaves_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_master_slaves_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_master_slaves_value')

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "add",
                                 "index": 0,
                                 "name": "unicode1_master_slaves_value.options.unicode.unicode",
                                 "value": "val5"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['unicode1_master_slaves_value.options.unicode.unicode',
                                    'unicode1_master_slaves_value.options.unicode.unicode2'],
                        'model': [{'key': 'unicode1_master_slaves_value.options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val4', 'val5'],
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'test',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 2,
                                   'value': 'slave2',
                                   'owner': 'default'}]}
            assert values == expected

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4', 'val5'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2', None],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test', 'slave2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super', None]}


def test_updates_unicode1_master_slaves_value_modify_master():
    descr = get_description_unicode1_master_slaves_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_master_slaves_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test'],
                                              'options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "modify",
                                 "name": "options.unicode.unicode",
                                 "index": 1,
                                 "value": "val5"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['options.unicode.unicode'],
                        'model': [{'key': 'options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val5'],
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'test',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'}]}
            assert values == expected

            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val5'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test'],
                                              'options.unicode.unicode3': [None, 'super']}


def test_updates_unicode1_master_slaves_value_modify_master_sub():
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_master_slaves_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_master_slaves_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_master_slaves_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_master_slaves_value')

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "modify",
                                 "name": "unicode1_master_slaves_value.options.unicode.unicode",
                                 "index": 1,
                                 "value": "val5"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['unicode1_master_slaves_value.options.unicode.unicode'],
                        'model': [{'key': 'unicode1_master_slaves_value.options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val5'],
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'test',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'}]}
            assert values == expected

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val5'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}


def test_updates_unicode1_master_slaves_value_delete_slave():
    descr = get_description_unicode1_master_slaves_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_master_slaves_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test'],
                                              'options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "delete",
                                 "name": "options.unicode.unicode2",
                                 "index": 1}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['options.unicode.unicode2'],
                        'model': [{'key': 'options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val4'],
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'slave2',
                                   'owner': 'default'},
                                  {'key': 'options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'}]}

            assert values == expected
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'slave2'],
                                              'options.unicode.unicode3': [None, 'super']}


def test_updates_unicode1_master_slaves_value_delete_slave_sub():
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_master_slaves_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_master_slaves_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_master_slaves_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_master_slaves_value')

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "delete",
                                 "name": "unicode1_master_slaves_value.options.unicode.unicode2",
                                 "index": 1}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['unicode1_master_slaves_value.options.unicode.unicode2'],
                        'model': [{'key': 'unicode1_master_slaves_value.options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val4'],
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'slave2',
                                   'owner': 'default'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'}]}

            assert values == expected
            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'slave2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}


def test_updates_unicode1_master_slaves_value_modify_slave():
    descr = get_description_unicode1_master_slaves_value()
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            config = Config(descr)
            get_values_unicode1_master_slaves_value(fake_config(config))
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable)
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test'],
                                              'options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "modify",
                                 "name": "options.unicode.unicode2",
                                 "index": 1,
                                 "value": "test2"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['options.unicode.unicode2'],
                        'model': [{'key': 'options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val4'],
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'test2',
                                   'owner': 'user'},
                                  {'key': 'options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'}]}

            assert values == expected
            assert make_dict(config) == {'options.unicode.unicode': ['val3', 'val4'],
                                              'options.unicode.unicode1': ['super1', 'super2'],
                                              'options.unicode.unicode2': ['pas test', 'test2'],
                                              'options.unicode.unicode3': [None, 'super']}


def test_updates_unicode1_master_slaves_value_modify_slave_sub():
    for clearable in ['none', 'minimum', 'all']:
        for remote in ['none', 'minimum', 'all']:
            descr = get_description_unicode1_master_slaves_value()
            config = Config(OptionDescription('root', '', [descr]))
            get_values_unicode1_master_slaves_value(fake_config(config), True)
            if TIRAMISU_VERSION == 2:
                rootconfi = config
                config = config.unicode1_master_slaves_value
            tiramisu = TiramisuWeb(config, remotable=remote, clearable=clearable, root='unicode1_master_slaves_value')

            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                      'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                      'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test'],
                                      'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}

            body = {'updates': [{"action": "modify",
                                 "name": "unicode1_master_slaves_value.options.unicode.unicode2",
                                 "index": 1,
                                 "value": "test2"}]}
            values = tiramisu.set_updates(body)

            expected = {'updates': ['unicode1_master_slaves_value.options.unicode.unicode2'],
                        'model': [{'key': 'unicode1_master_slaves_value.options.unicode.unicode',
                                   'required': True,
                                   'value': ['val3', 'val4'],
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 0,
                                   'value': 'super1',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 0,
                                   'value': 'pas test',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode1',
                                   'index': 1,
                                   'value': 'super2',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode2',
                                   'index': 1,
                                   'value': 'test2',
                                   'owner': 'user'},
                                  {'key': 'unicode1_master_slaves_value.options.unicode.unicode3',
                                   'index': 1,
                                   'value': 'super',
                                   'owner': 'user'}]}

            assert values == expected
            assert make_dict(config) == {'unicode1_master_slaves_value.options.unicode.unicode': ['val3', 'val4'],
                                         'unicode1_master_slaves_value.options.unicode.unicode1': ['super1', 'super2'],
                                         'unicode1_master_slaves_value.options.unicode.unicode2': ['pas test', 'test2'],
                                         'unicode1_master_slaves_value.options.unicode.unicode3': [None, 'super']}
