# -*- coding: utf-8 -*-

"""theses tests are much more to test that config, option description, vs...
**it's there** and answers via attribute access"""

import autopath
from py.test import raises

from tiramisu.config import Config, SubConfig
from tiramisu.option import IntOption, FloatOption, StrOption, ChoiceOption, \
    BoolOption, UnicodeOption, OptionDescription
from tiramisu.error import ConflictError, ConfigError, PropertiesOptionError
import weakref


def make_description():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc", properties=('mandatory', ))
    boolop = BoolOption('boolop', 'Test boolean option op', default=True, properties=('hidden',))
    wantref_option = BoolOption('wantref', 'Test requires', default=False)
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False)

    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption])
    descr = OptionDescription('tiram', '', [gcgroup, booloption, objspaceoption,
                                            wantref_option, stroption,
                                            wantframework_option,
                                            intoption, boolop])
    return descr


def test_base_config():
    """making a :class:`tiramisu.config.Config()` object
    and a :class:`tiramisu.option.OptionDescription()` object
    """
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    assert cfg.dummy is False
    dm = cfg.unwrap_from_path('dummy')
    assert dm._name == 'dummy'


def test_not_config():
    assert raises(TypeError, "Config('str')")


def test_base_path():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    assert cfg._impl_path is None
    base = OptionDescription('config', '', [descr])
    cfg = Config(base)
    assert cfg._impl_path is None
    assert cfg.tiramisu._impl_path == 'tiramisu'
    nbase = OptionDescription('baseconfig', '', [base])
    cfg = Config(nbase)
    assert cfg._impl_path is None
    assert cfg.config._impl_path == 'config'
    assert cfg.config.tiramisu._impl_path == 'config.tiramisu'


def test_reset_value():
    descr = make_description()
    cfg = Config(descr)
    assert cfg.gc.dummy is False
    cfg.gc.dummy = True
    assert cfg.gc.dummy is True


def test_base_config_and_groups():
    descr = make_description()
    # overrides the booloption default value
    config = Config(descr)
    config.bool = False
    assert config.gc.name == 'ref'
    assert config.bool is False
    nm = config.unwrap_from_path('gc.name')
    assert nm._name == 'name'
    gc = config.unwrap_from_path('gc')
    assert gc._name == 'gc'
    #nm = config.unwrap_from_name('name')
    #assert nm._name == 'name'


def test_base_config_force_permissive():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    config.cfgimpl_get_settings().setpermissive(('hidden',))
    raises(PropertiesOptionError, "config.getattr('boolop')")
    assert config.getattr('boolop', force_permissive=True) is True


def test_base_config_force_permissives():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    setting = config.cfgimpl_get_settings()
    raises(PropertiesOptionError, "config.getattr('boolop')")
    assert config.getattr('boolop', force_permissives=set(['hidden'])) is True
    setting.remove('validator')
    setting.remove('hidden')
    config.boolop = 1
    setting.append('validator')
    setting.append('hidden')
    raises(ValueError, "config.getattr('boolop', force_permissives=set(['hidden']))")


def test_base_config_force_permissives_path():
    descr = make_description()
    od = OptionDescription('o', '', [descr])
    config = Config(od)
    config.read_write()
    setting = config.cfgimpl_get_settings()
    raises(PropertiesOptionError, "config.getattr('tiram.boolop')")
    assert config.getattr('tiram.boolop', force_permissives=set(['hidden'])) is True
    setting.remove('validator')
    setting.remove('hidden')
    config.tiram.boolop = 1
    setting.append('validator')
    setting.append('hidden')
    raises(ValueError, "config.getattr('tiram.boolop', force_permissives=set(['hidden']))")


def test_base_config_in_a_tree():
    "how options are organized into a tree, see :ref:`tree`"
    descr = make_description()
    config = Config(descr)
    config.bool = False
    assert config.gc.name == 'ref'
    config.gc.name = 'framework'
    assert config.gc.name == 'framework'
    assert getattr(config, "gc.name") == 'framework'

    assert config.objspace == 'std'
    config.objspace = 'thunk'
    assert config.objspace == 'thunk'

    assert config.gc.float == 2.3
    assert config.int == 0
    config.gc.float = 3.4
    config.int = 123
    assert config.gc.float == 3.4
    assert config.int == 123

    assert not config.wantref

    assert config.str == "abc"
    config.str = "def"
    assert config.str == "def"

    raises(AttributeError, 'config.gc.foo = "bar"')

    config = Config(descr)
    config.bool = False
    assert config.gc.name == 'ref'
    config.wantframework = True


def test_cfgimpl_get_home_by_path():
    " :meth:`tiramisu.config.SubConfig.cfgimpl_get_home_by_path()` to retrieve a path"
    descr = make_description()
    config = Config(descr)
    config.bool = False
    assert config.cfgimpl_get_home_by_path('gc.dummy')[1] == 'dummy'
    assert config.cfgimpl_get_home_by_path('dummy')[1] == 'dummy'


def test_not_valid_properties():
    raises(TypeError, "stroption = StrOption('str', 'Test string option', default='abc', properties=['mandatory',])")


def test_information_config():
    descr = make_description()
    config = Config(descr)
    string = 'some informations'
    config.impl_set_information('info', string)
    assert config.impl_get_information('info') == string
    raises(ValueError, "config.impl_get_information('noinfo')")
    assert config.impl_get_information('noinfo', 'default') == 'default'


def test_config_impl_get_path_by_opt():
    descr = make_description()
    config = Config(descr)
    dummy = config.unwrap_from_path('gc.dummy')
    boo = config.unwrap_from_path('bool')
    unknown = IntOption('test', '')
    assert config.cfgimpl_get_description().impl_get_path_by_opt(boo) == 'bool'
    assert config.cfgimpl_get_description().impl_get_path_by_opt(dummy) == 'gc.dummy'
    raises(AttributeError, "config.cfgimpl_get_description().impl_get_path_by_opt(unknown)")


def test_config_impl_get_opt_by_path():
    descr = make_description()
    config = Config(descr)
    dummy = config.unwrap_from_path('gc.dummy')
    boo = config.unwrap_from_path('bool')
    assert config.cfgimpl_get_description().impl_get_opt_by_path('bool') == boo
    assert config.cfgimpl_get_description().impl_get_opt_by_path('gc.dummy') == dummy
    raises(AttributeError, "config.cfgimpl_get_description().impl_get_opt_by_path('gc.unknown')")


def test_information_display():
    g1 = IntOption('g1', '', 1)
    g2 = StrOption('g2', '', 'héhé')
    g3 = UnicodeOption('g3', '', u'héhé')
    g4 = BoolOption('g4', '', True)
    g5 = StrOption('g5', '')
    d1 = OptionDescription('od', '', [g1, g2, g3, g4, g5])
    root = OptionDescription('root', '', [d1])
    config = Config(root)
    config.od == """g1 = 1
g2 = héhé
g3 = héhé
g4 = True
g5 = None"""
    config == '[od]'


def test_get_modified_values():
    g1 = IntOption('g1', '', 1)
    g2 = StrOption('g2', '', 'héhé')
    g3 = UnicodeOption('g3', '', u'héhé')
    g4 = BoolOption('g4', '', True)
    g5 = StrOption('g5', '')
    d1 = OptionDescription('od', '', [g1, g2, g3, g4, g5])
    root = OptionDescription('root', '', [d1])
    config = Config(root)
    assert config.cfgimpl_get_values().get_modified_values() == {}
    config.od.g5 = 'yes'
    assert config.cfgimpl_get_values().get_modified_values() == {'od.g5': ('user', 'yes')}
    config.od.g4 = True
    assert config.cfgimpl_get_values().get_modified_values() == {'od.g5': ('user', 'yes'), 'od.g4': ('user', True)}


def test_has_value():
    g1 = IntOption('g1', '', 1)
    g2 = StrOption('g2', '', 'héhé')
    g3 = UnicodeOption('g3', '', u'héhé')
    g4 = BoolOption('g4', '', True)
    g5 = StrOption('g5', '')
    d1 = OptionDescription('od', '', [g1, g2, g3, g4, g5])
    root = OptionDescription('root', '', [d1])
    config = Config(root)
    assert not g5 in config.cfgimpl_get_values()
    config.od.g5 = 'yes'
    assert g5 in config.cfgimpl_get_values()


def test_values_not_setitem():
    g1 = IntOption('g1', '', 1)
    g2 = StrOption('g2', '', 'héhé')
    g3 = UnicodeOption('g3', '', u'héhé')
    g4 = BoolOption('g4', '', True)
    g5 = StrOption('g5', '')
    d1 = OptionDescription('od', '', [g1, g2, g3, g4, g5])
    root = OptionDescription('root', '', [d1])
    config = Config(root)
    raises(ConfigError, "config.cfgimpl_get_values()[g1] = 2")


def test_duplicated_option():
    g1 = IntOption('g1', '', 1)
    #in same OptionDescription
    raises(ConflictError, "d1 = OptionDescription('od', '', [g1, g1])")
    d1 = OptionDescription('od1', '', [g1])
    d2 = OptionDescription('od2', '', [g1])
    root = OptionDescription('root', '', [d1, d2])
    #in different OptionDescription
    raises(ConflictError, "config = Config(root)")


def test_cannot_assign_value_to_option_description():
    descr = make_description()
    cfg = Config(descr)
    raises(TypeError, "cfg.gc = 3")


def test_config_multi():
    i1 = IntOption('test1', '', multi=True)
    i2 = IntOption('test2', '', multi=True, default_multi=1)
    i3 = IntOption('test3', '', default=[2], multi=True, default_multi=1)
    od = OptionDescription('test', '', [i1, i2, i3])
    config = Config(od)
    assert config.test1 == []
    assert config.test2 == []
    config.test2.append()
    assert config.test2 == [1]
    assert config.test3 == [2]
    config.test3.append()
    assert config.test3 == [2, 1]


def test_no_validation():
    i1 = IntOption('test1', '')
    od = OptionDescription('test', '', [i1])
    c = Config(od)
    setting = c.cfgimpl_get_settings()
    c.test1 = 1
    raises(ValueError, 'c.test1 = "yes"')
    assert c.test1 == 1
    setting.remove('validator')
    c.test1 = "yes"
    assert c.test1 == "yes"
    setting.append('validator')
    raises(ValueError, 'c.test1')
    del(c.test1)
    assert c.test1 is None


def test_delete_config_with_subconfig():
    test = IntOption('test', '')
    multi = IntOption('multi', '', multi=True)
    od = OptionDescription('od', '', [test, multi])
    odroot = OptionDescription('odroot', '', [od])
    c = Config(odroot)
    sub = c.od
    val = c.cfgimpl_get_values()
    setting = c.cfgimpl_get_settings()
    val[test]
    val[multi]
    setting[test]
    sub.make_dict()
    del(c)
    raises(ConfigError, 'val[test]')
    raises(ConfigError, 'val[multi]')
    raises(ConfigError, 'setting[test]')
    raises(ConfigError, 'sub.make_dict()')


def test_config_weakref():
    o = OptionDescription('val', '', [])
    o2 = OptionDescription('val', '', [o])
    c = Config(o2)
    SubConfig(o, weakref.ref(c))
    raises(ValueError, "SubConfig(o, c)")
    s = SubConfig(o, weakref.ref(c))
    assert s._cfgimpl_get_context() == c
    del(c)
    raises(ConfigError, "s._cfgimpl_get_context()")


def test_config_str():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    gcdummy1 = BoolOption('dummy1', 'dummy', default=False, properties=('disabled',))
    o = OptionDescription('o', '', [gcdummy, gcdummy1])
    descr = OptionDescription('tiramisu', '', [o])
    cfg = Config(descr)
    cfg.read_only()
    str(cfg)
    str(cfg.o)
