import autopath
from py.test import raises

from tiramisu.setting import groups
from tiramisu.config import Config
from tiramisu.option import ChoiceOption, BoolOption, IntOption, FloatOption, \
    StrOption, OptionDescription, SymLinkOption
from tiramisu.error import PropertiesOptionError, ConflictError, SlaveError, ConfigError


def return_val():
    return 'val'


def return_concat(*args):
    return '.'.join(list(args))


def return_list(value=None):
    return ['val', 'val']


def return_list2(*args):
    l = []
    for arg in args:
        if isinstance(arg, list):
            l.extend(arg)
        else:
            l.append(arg)
    return l


def return_value(value=None):
    return value


def return_value2(*args, **kwargs):
    value = list(args)
    value.extend(kwargs.values())
    return value


def return_calc(i, j, k):
    return i + j + k


def is_config(config, **kwargs):
    if isinstance(config, Config):
        return 'yes'
    else:
        return 'no'


def make_description():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    intoption2 = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=True)
    wantref_option = BoolOption('wantref', 'Test requires', default=False,
                                requires=({'option': boolop, 'expected': True, 'action': 'hidden'},))
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False,
                                      requires=({'option': boolop, 'expected': True, 'action': 'hidden'},))
    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption, intoption2])
    descr = OptionDescription('constraints', '', [gcgroup, booloption, objspaceoption,
                              wantref_option, stroption,
                              wantframework_option,
                              intoption, boolop])
    return descr


def make_description_duplicates():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    ## dummy 1
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=True)
    wantref_option = BoolOption('wantref', 'Test requires', default=False,
                                requires=({'option': boolop, 'expected': True, 'action': 'hidden'},))
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False,
                                      requires=({'option': boolop, 'expected': True, 'action': 'hidden'},))
    # dummy2 (same path)
    gcdummy2 = BoolOption('dummy', 'dummy2', default=True)
    # dummy3 (same name)
    gcdummy3 = BoolOption('dummy', 'dummy2', default=True)
    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, gcdummy2, floatoption])
    descr = OptionDescription('constraints', '', [gcgroup, booloption, objspaceoption,
                              wantref_option, stroption,
                              wantframework_option,
                              intoption, boolop, gcdummy3])
    return descr


def test_identical_paths():
    """If in the schema (the option description) there is something that
    have the same name, an exection is raised
    """
    raises(ConflictError, "make_description_duplicates()")


def make_description_requires():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)

    floatoption = FloatOption('float', 'Test float option', default=2.3)

    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default="abc",
                          requires=({'option': intoption, 'expected': 1, 'action': 'hidden'},))

    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption])
    descr = OptionDescription('constraints', '', [gcgroup, booloption, objspaceoption,
                              stroption, intoption])
    return descr


def test_hidden_if_in():
    descr = make_description_requires()
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    cfg.read_write()
    stroption = cfg.unwrap_from_path('str')
    assert not 'hidden' in setting[stroption]
    cfg.int = 1
    raises(PropertiesOptionError, "cfg.str")
    raises(PropertiesOptionError, 'cfg.str="uvw"')
    assert 'hidden' in setting[stroption]


def test_hidden_if_in_with_group():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)

    floatoption = FloatOption('float', 'Test float option', default=2.3)

    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default="abc")
    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption],
                                requires=({'option': intoption, 'expected': 1, 'action': 'hidden'},))
    descr = OptionDescription('constraints', '', [gcgroup, booloption,
                              objspaceoption, stroption, intoption])
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    cfg.read_write()
    assert not 'hidden' in setting[stroption]
    cfg.int = 1
    raises(PropertiesOptionError, "cfg.gc.name")


def test_disabled_with_group():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)

    floatoption = FloatOption('float', 'Test float option', default=2.3)

    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default="abc")
    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption],
                                requires=({'option': intoption, 'expected': 1, 'action': 'disabled'},))
    descr = OptionDescription('constraints', '', [gcgroup, booloption,
                              objspaceoption, stroption, intoption])
    cfg = Config(descr)
    cfg.read_write()
    assert cfg.gc.name
    cfg.int = 1
    raises(PropertiesOptionError, "cfg.gc.name")
#____________________________________________________________


def make_description_callback():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy')
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=True)
    wantref_option = BoolOption('wantref', 'Test requires', default=False,
                                requires=({'option': boolop, 'expected': True, 'action': 'hidden'},))
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False,
                                      requires=({'option': boolop, 'expected': True, 'action': 'hidden'},))
    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption])
    descr = OptionDescription('constraints', '', [gcgroup, booloption, objspaceoption,
                              wantref_option, stroption,
                              wantframework_option,
                              intoption, boolop])
    return descr


def test_has_callback():
    descr = make_description_callback()
    # here the owner is 'default'
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    config.bool = False
    # because dummy has a callback
    dummy = config.unwrap_from_path('gc.dummy')
    setting.append('freeze')
    setting[dummy].append('frozen')
    raises(PropertiesOptionError, "config.gc.dummy = True")


def test_freeze_and_has_callback():
    descr = make_description_callback()
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    config.bool = False
    setting = config.cfgimpl_get_settings()
    setting.append('freeze')
    dummy = config.unwrap_from_path('gc.dummy')
    setting[dummy].append('frozen')
    raises(PropertiesOptionError, "config.gc.dummy = True")


def test_callback():
    val1 = StrOption('val1', "", callback=return_val)
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == 'val'
    cfg.val1 = 'new-val'
    assert cfg.val1 == 'new-val'
    del(cfg.val1)
    assert cfg.val1 == 'val'


def test_callback_params_without_callback():
    raises(ValueError, "StrOption('val2', '', callback_params={'': ('yes',)})")


def test_callback_invalid():
    raises(ValueError, 'val1 = StrOption("val1", "", callback="string")')
    raises(ValueError, 'val1 = StrOption("val1", "", callback=return_val, callback_params="string")')
    val1 = StrOption('val1', "", 'val')
    raises(ValueError, "StrOption('val2', '', callback=return_value, callback_params={'': 'string'})")
    raises(ValueError, "StrOption('val4', '', callback=return_value, callback_params={'value': (('string', False),)})")
    raises(ValueError, "StrOption('val4', '', callback=return_value, callback_params={'value': ((val1, 'string'),)})")
    raises(ValueError, "StrOption('val4', '', callback=return_value, callback_params={'value': ((val1, False, 'unknown'),)})")
    raises(ValueError, "StrOption('val4', '', callback=return_value, callback_params={'value': ((val1,),)})")


def test_callback_with_context():
    val1 = StrOption("val1", "", callback=is_config, callback_params={'': ((None,),), 'value': ('string',)})
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    assert cfg.val1 == 'yes'


def test_callback_with_context_named():
    val1 = StrOption("val1", "", callback=is_config, callback_params={'config': ((None,),)})
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    assert cfg.val1 == 'yes'


def test_callback_with_error():
    val1 = StrOption("val1", "", callback=is_config, callback_params={'': ('string',), 'value': ('string',)})
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    assert cfg.val1 == 'no'


def test_callback_value():
    val1 = StrOption('val1', "", 'val')
    val2 = StrOption('val2', "", callback=return_value, callback_params={'': ((val1, False),)})
    val3 = StrOption('val3', "", callback=return_value, callback_params={'': ('yes',)})
    val4 = StrOption('val4', "", callback=return_value, callback_params={'value': ((val1, False),)})
    val5 = StrOption('val5', "", callback=return_value, callback_params={'value': ('yes',)})
    maconfig = OptionDescription('rootconfig', '', [val1, val2, val3, val4, val5])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == 'val'
    assert cfg.val2 == 'val'
    assert cfg.val4 == 'val'
    cfg.val1 = 'new-val'
    assert cfg.val1 == 'new-val'
    assert cfg.val2 == 'new-val'
    assert cfg.val4 == 'new-val'
    del(cfg.val1)
    assert cfg.val1 == 'val'
    assert cfg.val2 == 'val'
    assert cfg.val3 == 'yes'
    assert cfg.val4 == 'val'
    assert cfg.val5 == 'yes'


def test_callback_value_tuple():
    val1 = StrOption('val1', "", 'val1')
    val2 = StrOption('val2', "", 'val2')
    val3 = StrOption('val3', "", callback=return_concat, callback_params={'': ((val1, False), (val2, False))})
    val4 = StrOption('val4', "", callback=return_concat, callback_params={'': ('yes', 'no')})
    raises(ValueError, "StrOption('val4', '', callback=return_concat, callback_params={'value': ('yes', 'no')})")
    maconfig = OptionDescription('rootconfig', '', [val1, val2, val3, val4])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == 'val1'
    assert cfg.val2 == 'val2'
    assert cfg.val3 == 'val1.val2'
    assert cfg.val4 == 'yes.no'
    cfg.val1 = 'new-val'
    assert cfg.val3 == 'new-val.val2'
    del(cfg.val1)
    assert cfg.val3 == 'val1.val2'


def test_callback_value_force_permissive():
    val1 = StrOption('val1', "", 'val', properties=('disabled',))
    val2 = StrOption('val2', "", callback=return_value, callback_params={'': ((val1, False),)})
    val3 = StrOption('val3', "", callback=return_value, callback_params={'': ((val1, True),)})
    maconfig = OptionDescription('rootconfig', '', [val1, val2, val3])
    cfg = Config(maconfig)
    cfg.read_only()
    raises(ConfigError, "cfg.val2")
    assert cfg.val3 is None


def test_callback_symlink():
    val1 = StrOption('val1', "", 'val')
    val2 = SymLinkOption('val2', val1)
    val3 = StrOption('val3', "", callback=return_value, callback_params={'': ((val2, False),)})
    maconfig = OptionDescription('rootconfig', '', [val1, val2, val3])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == 'val'
    assert cfg.val3 == 'val'
    cfg.val1 = 'new-val'
    assert cfg.val1 == 'new-val'
    assert cfg.val3 == 'new-val'
    del(cfg.val1)
    assert cfg.val1 == 'val'
    assert cfg.val3 == 'val'


def test_callback_list():
    val1 = StrOption('val1', "", callback=return_list)
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(ValueError, "cfg.val1")


def test_callback_multi():
    val1 = StrOption('val1', "", callback=return_val, multi=True)
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == ['val']
    cfg.val1 = ['new-val']
    assert cfg.val1 == ['new-val']
    cfg.val1.append('new-val2')
    assert cfg.val1 == ['new-val', 'new-val2']
    del(cfg.val1)
    assert cfg.val1 == ['val']


def test_callback_multi_value():
    val1 = StrOption('val1', "", ['val'], multi=True)
    val2 = StrOption('val2', "", multi=True, callback=return_value, callback_params={'': ((val1, False),)})
    val3 = StrOption('val3', "", multi=True, callback=return_value, callback_params={'': ('yes',)})
    val4 = StrOption('val4', "", multi=True, callback=return_list2, callback_params={'': ((val1, False), 'yes')})
    maconfig = OptionDescription('rootconfig', '', [val1, val2, val3, val4])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == ['val']
    assert cfg.val2 == ['val']
    assert cfg.val4 == ['val', 'yes']
    cfg.val1 = ['new-val']
    assert cfg.val1 == ['new-val']
    assert cfg.val2 == ['new-val']
    assert cfg.val4 == ['new-val', 'yes']
    cfg.val1.append('new-val2')
    assert cfg.val1 == ['new-val', 'new-val2']
    assert cfg.val2 == ['new-val', 'new-val2']
    assert cfg.val4 == ['new-val', 'new-val2', 'yes']
    del(cfg.val1)
    assert cfg.val1 == ['val']
    assert cfg.val2 == ['val']
    assert cfg.val3 == ['yes']
    assert cfg.val4 == ['val', 'yes']
    cfg.val2.append('new')
    assert cfg.val1 == ['val']
    assert cfg.val2 == ['val', 'new']


def test_callback_multi_list():
    val1 = StrOption('val1', "", callback=return_list, multi=True)
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == ['val', 'val']
    cfg.val1 = ['new-val']
    assert cfg.val1 == ['new-val']
    cfg.val1.append('new-val2')
    assert cfg.val1 == ['new-val', 'new-val2']
    del(cfg.val1)
    assert cfg.val1 == ['val', 'val']


def test_callback_multi_list_extend():
    val1 = StrOption('val1', "", callback=return_list2, callback_params={'': (['1', '2', '3'], ['4', '5'])}, multi=True)
    maconfig = OptionDescription('rootconfig', '', [val1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1 == ['1', '2', '3', '4', '5']


def test_callback_master_and_slaves_master():
    val1 = StrOption('val1', "", multi=True, callback=return_val)
    val2 = StrOption('val2', "", multi=True)
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1.val1 == ['val']
    cfg.val1.val1.append()
    assert cfg.val1.val1 == ['val', 'val']
    assert cfg.val1.val2 == [None, None]


def test_callback_master_and_slaves_master2():
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True, default_multi='val2')
    val3 = StrOption('val3', "", multi=True, callback=return_value, callback_params={'': ((val2, False),)})
    val4 = StrOption('val4', "", multi=True, callback=return_value, callback_params={'': ((val3, False),)})
    interface1 = OptionDescription('val1', '', [val1, val2, val3, val4])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    cfg.val1.val1.append('val')
    assert cfg.val1.val4 == ['val2']
    assert cfg.val1.val3 == ['val2']
    assert cfg.val1.val2 == ['val2']


def test_callback_master_and_slaves_master_list():
    val1 = StrOption('val1', "", multi=True, callback=return_list)
    val2 = StrOption('val2', "", multi=True)
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1.val1 == ['val', 'val']
    assert cfg.val1.val2 == [None, None]
    cfg.val1.val1.append()
    assert cfg.val1.val1 == ['val', 'val', None]
    assert cfg.val1.val2 == [None, None, None]
    del(cfg.val1.val1)
    assert cfg.val1.val1 == ['val', 'val']
    assert cfg.val1.val2 == [None, None]
    assert cfg.val1.val1.pop(1)
    assert cfg.val1.val1 == ['val']
    assert cfg.val1.val2 == [None]


def test_callback_master_and_slaves_slave():
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True, callback=return_val)
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1.val1 == []
    assert cfg.val1.val2 == []
    #
    cfg.val1.val1 = ['val1']
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == ['val']
    #
    cfg.val1.val1.append('val2')
    assert cfg.val1.val1 == ['val1', 'val2']
    assert cfg.val1.val2 == ['val', 'val']
    #
    cfg.val1.val1 = ['val1', 'val2', 'val3']
    assert cfg.val1.val1 == ['val1', 'val2', 'val3']
    assert cfg.val1.val2 == ['val', 'val', 'val']
    #
    cfg.val1.val1.pop(2)
    assert cfg.val1.val1 == ['val1', 'val2']
    assert cfg.val1.val2 == ['val', 'val']
    #
    cfg.val1.val2 = ['val2', 'val2']
    assert cfg.val1.val2 == ['val2', 'val2']
    #
    cfg.val1.val1.append('val3')
    assert cfg.val1.val2 == ['val2', 'val2', 'val']


def test_callback_master_and_slaves():
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True, callback=return_val)
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()


def test_callback_master_and_slaves_slave_cal():
    val3 = StrOption('val3', "", multi=True)
    val1 = StrOption('val1', "", multi=True, callback=return_value, callback_params={'': ((val3, False),)})
    val2 = StrOption('val2', "", multi=True, callback=return_val)
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1, val3])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val3 == []
    assert cfg.val1.val1 == []
    assert cfg.val1.val2 == []
    cfg.val1.val1 = ['val1']
    cfg.val3 = ['val1']
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == ['val']
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == ['val']
    del(cfg.val1.val1)
    cfg.val1.val2 = ['val']
    cfg.val3 = ['val1', 'val2']
    assert cfg.val1.val2 == ['val', 'val']
    assert cfg.val1.val1 == ['val1', 'val2']
    cfg.val1.val2 = ['val1', 'val2']
    cfg.val3.pop(1)
    # cannot remove slave's value because master is calculated
    # so raise
    raises(SlaveError, "cfg.val1.val1")
    raises(SlaveError, "cfg.val1.val2")
    cfg.val3 = ['val1', 'val2', 'val3']
    assert cfg.val1.val2 == ['val1', 'val2', 'val']


def test_callback_master_and_slaves_slave_cal2():
    val3 = StrOption('val3', "", ['val', 'val'], multi=True)
    val1 = StrOption('val1', "", multi=True, callback=return_value, callback_params={'': ((val3, False),)})
    val2 = StrOption('val2', "", ['val2', 'val2'], multi=True)
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1, val3])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val3 == ['val', 'val']
    assert cfg.val1.val1 == ['val', 'val']
    assert cfg.val1.val2 == ['val2', 'val2']
    cfg.val3.pop(1)
#    # cannot remove slave's value because master is calculated
#    # so raise
    raises(SlaveError, "cfg.val1.val1")
    raises(SlaveError, "cfg.val1.val2")
    cfg.val3 = ['val', 'val']
    assert cfg.val3 == ['val', 'val']
    assert cfg.val1.val1 == ['val', 'val']
    assert cfg.val1.val2 == ['val2', 'val2']
    raises(SlaveError, "cfg.val1.val1 = ['val']")
    assert cfg.val3 == ['val', 'val']
    assert cfg.val1.val1 == ['val', 'val']
    assert cfg.val1.val2 == ['val2', 'val2']


def test_callback_master_and_slaves_slave_list():
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True, callback=return_list)
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1.val2 == []
    cfg.val1.val1 = ['val1', 'val2']
    assert cfg.val1.val1 == ['val1', 'val2']
    assert cfg.val1.val2 == ['val', 'val']
    cfg.val1.val1 = ['val1']
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == ['val']


def test_callback_master_and_slaves_value():
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True, callback=return_value, callback_params={'': ((val1, False),)})
    val3 = StrOption('val3', "", multi=True, callback=return_value, callback_params={'': ('yes',)})
    val4 = StrOption('val4', '', multi=True, default=['val10', 'val11'])
    val5 = StrOption('val5', "", multi=True, callback=return_value, callback_params={'': ((val4, False),)})
    val6 = StrOption('val6', "", multi=True, callback=return_value, callback_params={'': ((val5, False),)})
    interface1 = OptionDescription('val1', '', [val1, val2, val3, val5, val6])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1, val4])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1.val1 == []
    assert cfg.val1.val2 == []
    assert cfg.val1.val3 == []
    assert cfg.val1.val5 == []
    assert cfg.val1.val6 == []
    #
    cfg.val1.val1 = ['val1']
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == ['val1']
    assert cfg.val1.val3 == ['yes']
    assert cfg.val1.val5 == ['val10']
    assert cfg.val1.val6 == ['val10']
    #
    cfg.val1.val1.append('val2')
    assert cfg.val1.val1 == ['val1', 'val2']
    assert cfg.val1.val2 == ['val1', 'val2']
    assert cfg.val1.val3 == ['yes', 'yes']
    assert cfg.val1.val5 == ['val10', 'val11']
    assert cfg.val1.val6 == ['val10', 'val11']
    #
    cfg.val1.val1 = ['val1', 'val2', 'val3']
    assert cfg.val1.val1 == ['val1', 'val2', 'val3']
    assert cfg.val1.val2 == ['val1', 'val2', 'val3']
    assert cfg.val1.val3 == ['yes', 'yes', 'yes']
    assert cfg.val1.val5 == ['val10', 'val11', None]
    assert cfg.val1.val6 == ['val10', 'val11', None]
    #
    cfg.val1.val1.pop(2)
    assert cfg.val1.val1 == ['val1', 'val2']
    assert cfg.val1.val2 == ['val1', 'val2']
    assert cfg.val1.val3 == ['yes', 'yes']
    assert cfg.val1.val5 == ['val10', 'val11']
    assert cfg.val1.val6 == ['val10', 'val11']
    #
    cfg.val1.val2 = ['val2', 'val2']
    cfg.val1.val3 = ['val2', 'val2']
    cfg.val1.val5 = ['val2', 'val2']
    assert cfg.val1.val2 == ['val2', 'val2']
    assert cfg.val1.val3 == ['val2', 'val2']
    assert cfg.val1.val5 == ['val2', 'val2']
    assert cfg.val1.val6 == ['val2', 'val2']
    #
    cfg.val1.val1.append('val3')
    assert cfg.val1.val2 == ['val2', 'val2', 'val3']
    assert cfg.val1.val3 == ['val2', 'val2', 'yes']
    assert cfg.val1.val5 == ['val2', 'val2', None]
    assert cfg.val1.val6 == ['val2', 'val2', None]
    cfg.cfgimpl_get_settings().remove('cache')
    cfg.val4 = ['val10', 'val11', 'val12']
    #if value is already set, not updated !
    cfg.val1.val1.pop(2)
    cfg.val1.val1.append('val3')
    cfg.val1.val1 = ['val1', 'val2', 'val3']
    assert cfg.val1.val5 == ['val2', 'val2', 'val12']
    assert cfg.val1.val6 == ['val2', 'val2', 'val12']


def test_callback_master_and_slaves_disabled():
    val = BoolOption('val', '', multi=True)
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True, callback=return_value, callback_params={'': ((val, False),)})
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [val, interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    #
    assert cfg.val == []
    assert cfg.val1.val1 == []
    assert cfg.val1.val2 == []
    cfg.val1.val1.append('val1')
    assert cfg.val == []
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == [None]
    #
    cfg.val.append(True)
    cfg.cfgimpl_get_settings()[cfg.cfgimpl_get_description().val1.val2].append('disabled')
    assert cfg.val == [True]
    assert cfg.val1.val1 == ['val1']
    raises(PropertiesOptionError, 'cfg.val1.val2')
    #
    cfg.val1.val1.append('val1_1')
    assert cfg.val == [True]
    assert cfg.val1.val1 == ['val1', 'val1_1']
    raises(PropertiesOptionError, 'cfg.val1.val2')
    #
    cfg.cfgimpl_get_settings()[cfg.cfgimpl_get_description().val1.val2].remove('disabled')
    assert cfg.val == [True]
    assert cfg.val1.val1 == ['val1', 'val1_1']
    raises(ValueError, 'cfg.val1.val2')
    #
    cfg.val = []
    assert cfg.val1.val1 == ['val1', 'val1_1']
    assert cfg.val1.val2 == [None, None]
    #
    cfg.val1.val2 = [None, None]
    cfg.val = [True, True]
    cfg.cfgimpl_get_settings()[cfg.cfgimpl_get_description().val1.val2].append('disabled')
    cfg.val1.val1.pop(1)
    assert cfg.val == [True, True]
    assert cfg.val1.val1 == ['val1']
    raises(PropertiesOptionError, 'cfg.val1.val2')
    #
    cfg.val1.val1.append('val1_2')
    assert cfg.val == [True, True]
    assert cfg.val1.val1 == ['val1', 'val1_2']
    raises(PropertiesOptionError, 'cfg.val1.val2')
    #
    cfg.cfgimpl_get_settings()[cfg.cfgimpl_get_description().val1.val2].remove('disabled')
    assert cfg.val == [True, True]
    assert cfg.val1.val1 == ['val1', 'val1_2']
    #[None, None] + pop() + append() => [None, True]
    raises(ValueError, 'assert cfg.val1.val2')


def test_callback_master_and_slaves_requires():
    val = StrOption('val', '', 'val')
    valreq = StrOption('valreq', '', 'val')
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True, callback=return_value, callback_params={'': ((val1, False),)})
    val3 = StrOption('val3', "", multi=True, callback=return_value, callback_params={'': ('yes',)})
    val4 = StrOption('val4', '', multi=True, default=['val10', 'val11'])
    val5 = StrOption('val5', "", multi=True, callback=return_value, callback_params={'': ((val4, False),)})
    val6 = StrOption('val6', "", multi=True, callback=return_value, callback_params={'': ((val, False),)},
                     requires=({'option': valreq, 'expected': 'val_disabled', 'action': 'disabled'},))
    interface1 = OptionDescription('val1', '', [val1, val2, val3, val5, val6])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [val, valreq, interface1, val4])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1.val1 == []
    assert cfg.val1.val2 == []
    assert cfg.val1.val3 == []
    assert cfg.val1.val5 == []
    assert cfg.val1.val6 == []
    #
    cfg.val1.val1 = ['val1']
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == ['val1']
    assert cfg.val1.val3 == ['yes']
    assert cfg.val1.val5 == ['val10']
    assert cfg.val1.val6 == ['val']
    #
    cfg.valreq = 'val_disabled'
    assert cfg.val1.val1 == ['val1']
    assert cfg.val1.val2 == ['val1']
    assert cfg.val1.val3 == ['yes']
    assert cfg.val1.val5 == ['val10']
    raises(PropertiesOptionError, 'cfg.val1.val6')
    assert cfg.make_dict() == {'val1.val2': ['val1'], 'val1.val3': ['yes'], 'val1.val1': ['val1'], 'val1.val5': ['val10'], 'val': 'val', 'valreq': 'val_disabled', 'val4': ['val10', 'val11']}
    #
    cfg.cfgimpl_get_settings()[cfg.cfgimpl_get_description().val].append('disabled')
    cfg.cfgimpl_get_settings()[cfg.cfgimpl_get_description().val1.val3].append('disabled')
    raises(PropertiesOptionError, 'cfg.val1.val6')
    assert cfg.make_dict() == {'val1.val2': ['val1'], 'val1.val1': ['val1'], 'val1.val5': ['val10'], 'val4': ['val10', 'val11'], 'valreq': 'val_disabled'}
    #
    cfg.valreq = 'val'
    raises(ConfigError, 'cfg.val1.val6')


def test_callback_master():
    val2 = StrOption('val2', "", multi=True, callback=return_value)
    val1 = StrOption('val1', "", multi=True, callback=return_value, callback_params={'': ((val2, False),)})
    interface1 = OptionDescription('val1', '', [val1, val2])
    raises(ValueError, "interface1.impl_set_group_type(groups.master)")


def test_callback_master_and_other_master_slave():
    val1 = StrOption('val1', "", multi=True)
    val2 = StrOption('val2', "", multi=True)
    val3 = StrOption('val3', "", multi=True)
    val4 = StrOption('val4', '', multi=True, default=['val10', 'val11'])
    val5 = StrOption('val5', "", multi=True, callback=return_value, callback_params={'': ((val1, False),)})
    val6 = StrOption('val6', "", multi=True, callback=return_value, callback_params={'': ((val2, False),)})
    interface1 = OptionDescription('val1', '', [val1, val2, val3])
    interface1.impl_set_group_type(groups.master)
    interface2 = OptionDescription('val4', '', [val4, val5, val6])
    interface2.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1, interface2])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val4.val4 == ['val10', 'val11']
    assert cfg.val4.val5 == [None, None]
    assert cfg.val4.val6 == [None, None]
    cfg.val1.val1 = ['yes']
    assert cfg.val4.val4 == ['val10', 'val11']
    assert cfg.val4.val5 == ['yes', None]
    assert cfg.val4.val6 == [None, None]
    cfg.val1.val2 = ['no']
    assert cfg.val4.val4 == ['val10', 'val11']
    assert cfg.val4.val5 == ['yes', None]
    assert cfg.val4.val6 == ['no', None]
    cfg.val1.val1 = ['yes', 'yes', 'yes']
    cfg.val1.val2 = ['no', 'no', 'no']
    assert cfg.val4.val4 == ['val10', 'val11']
    assert cfg.val4.val5 == ['yes', 'yes']
    assert cfg.val4.val6 == ['no', 'no']


def test_callback_different_type():
    val = IntOption('val', "", default=2)
    val_ = IntOption('val_', "", default=3)
    val1 = IntOption('val1', "", multi=True)
    val2 = IntOption('val2', "", multi=True, callback=return_calc, callback_params={'': ((val, False), (val1, False)), 'k': ((val_, False),)})
    interface1 = OptionDescription('val1', '', [val1, val2])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('rootconfig', '', [interface1, val, val_])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val1.val1 == []
    assert cfg.val1.val2 == []
    cfg.val1.val1 = [1]
    assert cfg.val1.val1 == [1]
    assert cfg.val1.val2 == [6]
    cfg.val1.val1 = [1, 3]
    assert cfg.val1.val1 == [1, 3]
    assert cfg.val1.val2 == [6, 8]
    cfg.val1.val1 = [1, 3, 5]
    assert cfg.val1.val1 == [1, 3, 5]
    assert cfg.val1.val2 == [6, 8, 10]


def test_callback_hidden():
    opt1 = BoolOption('opt1', '')
    opt2 = BoolOption('opt2', '', callback=return_value, callback_params={'': ((opt1, False),)})
    od1 = OptionDescription('od1', '', [opt1], properties=('hidden',))
    od2 = OptionDescription('od2', '', [opt2])
    maconfig = OptionDescription('rootconfig', '', [od1, od2])
    cfg = Config(maconfig)
    cfg.cfgimpl_get_settings().setpermissive(('hidden',))
    cfg.read_write()
    raises(PropertiesOptionError, 'cfg.od1.opt1')
    cfg.od2.opt2


def test_callback_two_disabled():
    opt1 = BoolOption('opt1', '', properties=('disabled',))
    opt2 = BoolOption('opt2', '', callback=return_value, callback_params={'': ((opt1, False),)}, properties=('disabled',))
    od1 = OptionDescription('od1', '', [opt1])
    od2 = OptionDescription('od2', '', [opt2])
    maconfig = OptionDescription('rootconfig', '', [od1, od2])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(PropertiesOptionError, 'cfg.od2.opt2')


def test_callback_calculating_invalid():
    opt1 = IntOption('opt1', '', 1)
    opt2 = BoolOption('opt2', '', callback=return_value, callback_params={'': ((opt1, False),)})
    od1 = OptionDescription('od1', '', [opt1])
    od2 = OptionDescription('od2', '', [opt2])
    maconfig = OptionDescription('rootconfig', '', [od1, od2])
    cfg = Config(maconfig)
    setting = cfg.cfgimpl_get_settings()
    cfg.read_write()
    raises(ValueError, 'cfg.od2.opt2')
    setting[opt2].append('disabled')
    raises(PropertiesOptionError, 'cfg.od2.opt2')


def test_callback_calculating_disabled():
    opt1 = BoolOption('opt1', '', properties=('disabled',))
    opt2 = BoolOption('opt2', '', callback=return_value, callback_params={'': ((opt1, False),)})
    od1 = OptionDescription('od1', '', [opt1])
    od2 = OptionDescription('od2', '', [opt2])
    maconfig = OptionDescription('rootconfig', '', [od1, od2])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(ConfigError, 'cfg.od2.opt2')


def test_callback_calculating_mandatory():
    opt1 = BoolOption('opt1', '', properties=('disabled',))
    opt2 = BoolOption('opt2', '', callback=return_value, callback_params={'': ((opt1, False),)}, properties=('mandatory',))
    od1 = OptionDescription('od1', '', [opt1])
    od2 = OptionDescription('od2', '', [opt2])
    maconfig = OptionDescription('rootconfig', '', [od1, od2])
    cfg = Config(maconfig)
    cfg.read_only()
    raises(ConfigError, 'cfg.od2.opt2')


def test_callback_two_disabled_multi():
    opt1 = BoolOption('opt1', '', properties=('disabled',))
    opt2 = BoolOption('opt2', '', callback=return_value, callback_params={'': ((opt1, False),)}, properties=('disabled',), multi=True)
    od1 = OptionDescription('od1', '', [opt1])
    od2 = OptionDescription('od2', '', [opt2])
    maconfig = OptionDescription('rootconfig', '', [od1, od2])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(PropertiesOptionError, 'cfg.od2.opt2')


def test_callback_multi_list_params():
    val1 = StrOption('val1', "", multi=True, default=['val1', 'val2'])
    val2 = StrOption('val2', "", multi=True, callback=return_list, callback_params={'': ((val1, False),)})
    oval2 = OptionDescription('val2', '', [val2])
    maconfig = OptionDescription('rootconfig', '', [val1, oval2])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val2.val2 == ['val', 'val']


def test_callback_multi_list_params_key():
    val1 = StrOption('val1', "", multi=True, default=['val1', 'val2'])
    val2 = StrOption('val2', "", multi=True, callback=return_list, callback_params={'value': ((val1, False),)})
    oval2 = OptionDescription('val2', '', [val2])
    maconfig = OptionDescription('rootconfig', '', [val1, oval2])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.val2.val2 == ['val', 'val']
