"test all types of option default values for options, add new option in a descr"
import autopath
from py.test import raises

from tiramisu.config import Config
from tiramisu.setting import owners
from tiramisu.error import PropertiesOptionError, ConfigError
from tiramisu.option import IntOption, FloatOption, StrOption, ChoiceOption, \
    BoolOption, OptionDescription


def make_description():
    gcoption = ChoiceOption('name', 'GC name', ['ref', 'framework'], 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ['std', 'thunk'], 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=True)
    wantref_option = BoolOption('wantref', 'Test requires', default=False,
                                requires=['boolop'])
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False,
                                      requires=['boolop'])

    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption])
    descr = OptionDescription('tiramisu', '', [gcgroup, booloption, objspaceoption,
                                               wantref_option, stroption,
                                               wantframework_option,
                                               intoption, boolop])
    return descr


#____________________________________________________________
# default values
def test_default_is_none():
    """
    Most constructors take a ``default`` argument that specifies the default
    value of the option. If this argument is not supplied the default value is
    assumed to be ``None``.
    """
    dummy1 = BoolOption('dummy1', 'doc dummy')
    dummy2 = BoolOption('dummy2', 'doc dummy')
    group = OptionDescription('group', '', [dummy1, dummy2])
    config = Config(group)
    # so when the default value is not set, there is actually a default value
    assert config.dummy1 is None
    assert config.dummy2 is None


def test_set_defaut_value_from_option_object():
    """Options have an available default setting and can give it back"""
    b = BoolOption("boolean", "", default=False)
    assert b.impl_getdefault() is False


def test_force_default_on_freeze():
    "a frozen option wich is forced returns his default"
    dummy1 = BoolOption('dummy1', 'doc dummy', default=False, properties=('force_default_on_freeze',))
    dummy2 = BoolOption('dummy2', 'doc dummy', default=True)
    group = OptionDescription('group', '', [dummy1, dummy2])
    config = Config(group)
    config.read_write()
    owners.addowner("frozen")
    config.dummy1 = True
    config.dummy2 = False
    assert config.getowner(dummy1) == owners.user
    assert config.getowner(dummy2) == owners.user
    setting = config.cfgimpl_get_settings()
    setting[dummy1].append('frozen')
    setting[dummy2].append('frozen')
    assert config.dummy1 is False
    assert config.dummy2 is False
    assert config.getowner(dummy1) == owners.default
    assert config.getowner(dummy2) == owners.user
    raises(PropertiesOptionError, "config.cfgimpl_get_values().setowner(dummy1, owners.frozen)")
    raises(PropertiesOptionError, "config.cfgimpl_get_values().setowner(dummy2, owners.frozen)")
    raises(PropertiesOptionError, "del(config.dummy1)")
    setting[dummy1].remove('frozen')
    del(config.dummy1)
    setting[dummy1].append('frozen')
    raises(ConfigError, "config.cfgimpl_get_values().setowner(dummy1, owners.frozen)")


def test_force_default_on_freeze_multi():
    dummy1 = BoolOption('dummy1', 'doc dummy', default=[False], properties=('force_default_on_freeze',), multi=True)
    dummy2 = BoolOption('dummy2', 'doc dummy', default=[True], multi=True)
    group = OptionDescription('group', '', [dummy1, dummy2])
    config = Config(group)
    config.read_write()
    owners.addowner("frozenmulti")
    config.dummy1.append(True)
    config.dummy2.append(False)
    assert config.getowner(dummy1) == owners.user
    assert config.getowner(dummy2) == owners.user
    setting = config.cfgimpl_get_settings()
    setting[dummy1].append('frozen')
    setting[dummy2].append('frozen')
    assert config.dummy1 == [False]
    assert config.dummy2 == [True, False]
    assert config.getowner(dummy1) == owners.default
    assert config.getowner(dummy2) == owners.user
    raises(PropertiesOptionError, "config.cfgimpl_get_values().setowner(dummy1, owners.frozenmulti)")
    raises(PropertiesOptionError, "config.cfgimpl_get_values().setowner(dummy2, owners.frozenmulti)")
    raises(PropertiesOptionError, "del(config.dummy1)")
    setting[dummy1].remove('frozen')
    del(config.dummy1)
    setting[dummy1].append('frozen')
    raises(ConfigError, "config.cfgimpl_get_values().setowner(dummy1, owners.frozenmulti)")


def test_overrides_changes_option_value():
    "with config.override(), the default is changed and the value is changed"
    descr = OptionDescription("test", "", [
        BoolOption("b", "", default=False)])
    config = Config(descr)
    config.b = True


def test_choice_with_no_default():
    descr = OptionDescription("test", "", [
        ChoiceOption("backend", "", ("c", "cli"))])
    config = Config(descr)
    assert config.backend is None
    config.backend = "c"


def test_choice_with_default():
    descr = OptionDescription("test", "", [
        ChoiceOption("backend", "", ("c", "cli"), default="cli")])
    config = Config(descr)
    assert config.backend == "cli"
