# Copyright (C) 2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________

"""Config's informations are, by default, volatiles. This means, all values and
settings changes will be lost.

The storage is the system Tiramisu uses to communicate with various DB.
You can specified a persistent storage.

Storage is basic components used to set Config informations in DB.
The primary "entry point" class is the StorageType and it's public
configurator ``set_storage()``.
"""


from time import time
from tiramisu.error import ConfigError
from tiramisu.i18n import _


class StorageType(object):
    """Object to store storage's type. If a Config is already set,
    default storage is store as selected storage. You cannot change it
    after.
    """
    default_storage = 'dictionary'
    storage_type = None
    mod = None

    def set(self, name):
        if self.storage_type is not None:
            if self.storage_type == name:
                return
            raise ConfigError(_('storage_type is already set, cannot rebind it'))
        self.storage_type = name

    def get(self):
        if self.storage_type is None:
            self.storage_type = self.default_storage
        storage = self.storage_type
        if self.mod is None:
            modulepath = 'tiramisu.storage.{0}'.format(storage)
            mod = __import__(modulepath)
            for token in modulepath.split(".")[1:]:
                mod = getattr(mod, token)
            self.mod = mod
        return self.mod


storage_type = StorageType()


def set_storage(name, **kwargs):
    """Change storage's configuration

    :params name: is the storage name. If storage is already set, cannot
        reset storage name

    Other attributes are differents according to the selected storage's name
    """
    storage_type.set(name)
    setting = storage_type.get().setting
    for option, value in kwargs.items():
        try:
            getattr(setting, option)
            setattr(setting, option, value)
        except AttributeError:
            raise ValueError(_('option {0} not already exists in storage {1}'
                               '').format(option, name))


def _impl_getstate_setting():
    setting = storage_type.get().setting
    state = {'name': storage_type.storage_type}
    for var in dir(setting):
        if not var.startswith('_'):
            state[var] = getattr(setting, var)
    return state


def get_storage(session_id, persistent, test):
    """all used when __setstate__ a Config
    """
    return storage_type.get().Storage(session_id, persistent, test)


def get_storages(context, session_id, persistent):
    def gen_id(config):
        return str(id(config)) + str(time())

    if session_id is None:
        session_id = gen_id(context)
    imp = storage_type.get()
    storage = imp.Storage(session_id, persistent)
    return imp.Settings(storage), imp.Values(storage)


def list_sessions():
    """List all available session (persistent or not persistent)
    """
    return storage_type.get().list_sessions()


def delete_session(session_id):
    """Delete a selected session, be careful, you can deleted a session
    use by an other instance
    :params session_id: id of session to delete
    """
    return storage_type.get().delete_session(session_id)


__all__ = (set_storage, list_sessions, delete_session)
