# -*- coding: utf-8 -*-
"default plugin for value: set it in a simple dictionary"
# Copyright (C) 2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________

from ..util import Cache


class Values(Cache):
    __slots__ = ('_values', '_informations', '__weakref__')

    def __init__(self, storage):
        """init plugin means create values storage
        """
        self._values = {}
        self._informations = {}
        # should init cache too
        super(Values, self).__init__(storage)

    # value
    def setvalue(self, path, value, owner):
        """set value for a path
        a specified value must be associated to an owner
        """
        self._values[path] = (owner, value)

    def getvalue(self, path):
        """get value for a path
        return: only value, not the owner
        """
        return self._values[path][1]

    def hasvalue(self, path):
        """if path has a value
        return: boolean
        """
        return path in self._values

    def resetvalue(self, path):
        """remove value means delete value in storage
        """
        del(self._values[path])

    def get_modified_values(self):
        """return all values in a dictionary
        example: {'path1': (owner, 'value1'), 'path2': (owner, 'value2')}
        """
        return self._values

    # owner
    def setowner(self, path, owner):
        """change owner for a path
        """
        self._values[path] = (owner, self._values[path][1])

    def getowner(self, path, default):
        """get owner for a path
        return: owner object
        """
        return self._values.get(path, (default, None))[0]

    def set_information(self, key, value):
        """updates the information's attribute
        (which is a dictionary)

        :param key: information's key (ex: "help", "doc"
        :param value: information's value (ex: "the help string")
        """
        self._informations[key] = value

    def get_information(self, key):
        """retrieves one information's item

        :param key: the item string (ex: "help")
        """
        if key in self._informations:
            return self._informations[key]
        else:
            raise ValueError("not found")
