# -*- coding: utf-8 -*-
"default plugin for cache: set it in a simple dictionary"
# Copyright (C) 2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________
from tiramisu.setting import owners


class SerializeObject(object):
    def __getstate__(self):
        ret = {}
        for key in dir(self):
            if not key.startswith('__'):
                ret[key] = getattr(self, key)
        return ret


class Cache(object):
    __slots__ = ('_cache', '_storage')
    key_is_path = False

    def __init__(self, storage):
        self._cache = {}
        self._storage = storage

    def __getstate__(self):
        slots = set()
        for subclass in self.__class__.__mro__:
            if subclass is not object:
                slots.update(subclass.__slots__)
        slots -= frozenset(['__weakref__', '_storage'])
        states = {}
        for slot in slots:
            try:
                value = getattr(self, slot)
                #value has owners object, need 'str()' it
                if slot == '_values':
                    _value = {}
                    for key, values in value.items():
                        vals = list(values)
                        vals[0] = str(vals[0])
                        _value[key] = tuple(vals)
                    states[slot] = _value
                else:
                    states[slot] = value
            except AttributeError:
                pass
        return states

    def __setstate__(self, states):
        for key, value in states.items():
            #value has owners object, need to reconstruct it
            if key == '_values':
                _value = {}
                for key_, values_ in value.items():
                    vals = list(values_)
                    try:
                        vals[0] = getattr(owners, vals[0])
                    except AttributeError:
                        owners.addowner(vals[0])
                        vals[0] = getattr(owners, vals[0])
                    _value[key_] = tuple(vals)
                value = _value
            setattr(self, key, value)

    def setcache(self, path, val, time):
        self._cache[path] = (val, time)

    def getcache(self, path, exp):
        value, created = self._cache[path]
        if created is None or exp <= created:
            return True, value
        return False, None

    def hascache(self, path):
        """ path is in the cache

        :param path: the path's option
        """
        return path in self._cache

    def reset_expired_cache(self, exp):
        for key in tuple(self._cache.keys()):
            val, created = self._cache[key]
            if created is not None and exp > created:
                del(self._cache[key])

    def reset_all_cache(self):
        "empty the cache"
        self._cache.clear()

    def get_cached(self, context):
        """return all values in a dictionary
        example: {'path1': ('value1', 'time1'), 'path2': ('value2', 'time2')}
        """
        return self._cache
