# -*- coding: utf-8 -*-
"takes care of the option's values and multi values"
# Copyright (C) 2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________
from time import time
from copy import copy
import sys
import weakref
from tiramisu.error import ConfigError, SlaveError, PropertiesOptionError
from tiramisu.setting import owners, multitypes, expires_time, undefined
from tiramisu.autolib import carry_out_calculation
from tiramisu.i18n import _
from tiramisu.option import SymLinkOption, OptionDescription


class Values(object):
    """The `Config`'s root is indeed  in charge of the `Option()`'s values,
    but the values are physicaly located here, in `Values`, wich is also
    responsible of a caching utility.
    """
    __slots__ = ('context', '_p_', '__weakref__')

    def __init__(self, context, storage):
        """
        Initializes the values's dict.

        :param context: the context is the home config's values

        """
        self.context = weakref.ref(context)
        # the storage type is dictionary or sqlite3
        self._p_ = storage

    def _getcontext(self):
        """context could be None, we need to test it
        context is None only if all reference to `Config` object is deleted
        (for example we delete a `Config` and we manipulate a reference to
        old `SubConfig`, `Values`, `Multi` or `Settings`)
        """
        context = self.context()
        if context is None:
            raise ConfigError(_('the context does not exist anymore'))
        return context

    def _getdefault(self, opt):
        """
        actually retrieves the default value

        :param opt: the `option.Option()` object
        """
        meta = self._getcontext().cfgimpl_get_meta()
        if meta is not None:
            value = meta.cfgimpl_get_values()[opt]
            if isinstance(value, Multi):
                value = list(value)
        else:
            value = opt.impl_getdefault()
        if opt.impl_is_multi():
            return copy(value)
        else:
            return value

    def _getvalue(self, opt, path):
        """actually retrieves the value

        :param opt: the `option.Option()` object
        :returns: the option's value (or the default value if not set)
        """
        if not self._p_.hasvalue(path):
            # if there is no value
            value = self._getdefault(opt)
        else:
            # if there is a value
            value = self._p_.getvalue(path)
        return value

    def get_modified_values(self):
        return self._p_.get_modified_values()

    def __contains__(self, opt):
        """
        implements the 'in' keyword syntax in order provide a pythonic way
        to kow if an option have a value

        :param opt: the `option.Option()` object
        """
        path = self._get_opt_path(opt)
        return self._contains(path)

    def _contains(self, path):
        return self._p_.hasvalue(path)

    def __delitem__(self, opt, force_permissive=False):
        """overrides the builtins `del()` instructions"""
        self.reset(opt, force_permissive=force_permissive)

    def reset(self, opt, path=None, force_permissive=False):
        if path is None:
            path = self._get_opt_path(opt)
        context = self._getcontext()
        context.cfgimpl_get_settings().validate_properties(opt,
                                                           False,
                                                           True,
                                                           path,
                                                           force_permissive=force_permissive)
        hasvalue = self._p_.hasvalue(path)
        if hasvalue:
            setting = context.cfgimpl_get_settings()
            opt.impl_validate(opt.impl_getdefault(),
                              context, 'validator' in setting)
            context.cfgimpl_reset_cache()
        if (opt.impl_is_multi() and
                opt.impl_get_multitype() == multitypes.master):
            #for remove slave values
            for slave in opt.impl_get_master_slaves():
                pslave = self._get_opt_path(slave)
                try:
                    self._p_.resetvalue(pslave)
                except KeyError:
                    pass
        if hasvalue:
            self._p_.resetvalue(path)

    def _isempty(self, opt, value):
        "convenience method to know if an option is empty"
        empty = opt._empty
        if (not opt.impl_is_multi() and (value is None or value == empty)) or \
           (opt.impl_is_multi() and (value == [] or
                                     None in value or empty in value)):
            return True
        return False

    def _getcallback_value(self, opt, index=None, max_len=None):
        """
        retrieves a value for the options that have a callback

        :param opt: the `option.Option()` object
        :param index: if an option is multi, only calculates the nth value
        :type index: int
        :param is_default: if value is default value (not appended value for example)
        :type is_default: bool or None
        :returns: a calculated value
        """
        callback, callback_params = opt._callback
        if callback_params is None:
            callback_params = {}
        return carry_out_calculation(opt, config=self._getcontext(),
                                     callback=callback,
                                     callback_params=callback_params,
                                     index=index, max_len=max_len)

    def __getitem__(self, opt):
        "enables us to use the pythonic dictionary-like access to values"
        return self.getitem(opt)

    def getitem(self, opt, path=None, validate=True, force_permissive=False,
                force_properties=None, validate_properties=True,
                force_permissives=None):
        if path is None:
            path = self._get_opt_path(opt)
        ntime = None
        setting = self._getcontext().cfgimpl_get_settings()
        if 'cache' in setting and self._p_.hascache(path):
            if 'expire' in setting:
                ntime = int(time())
            is_cached, value = self._p_.getcache(path, ntime)
            if is_cached:
                if opt.impl_is_multi() and not isinstance(value, Multi):
                    #load value so don't need to validate if is not a Multi
                    value = Multi(value, self.context, opt, path, validate=False)
                return value
        val = self._getitem(opt, path, validate, force_permissive,
                            force_properties, validate_properties,
                            force_permissives)
        if 'cache' in setting and validate and validate_properties and \
                force_permissive is False and force_properties is None:
            if 'expire' in setting:
                if ntime is None:
                    ntime = int(time())
                ntime = ntime + expires_time
            self._p_.setcache(path, val, ntime)

        return val

    def _getitem(self, opt, path, validate, force_permissive, force_properties,
                 validate_properties, force_permissives):
        # options with callbacks
        context = self._getcontext()
        setting = context.cfgimpl_get_settings()
        is_frozen = 'frozen' in setting[opt]
        # For calculating properties, we need value (ie for mandatory value).
        # If value is calculating with a PropertiesOptionError's option
        # _getcallback_value raise a ConfigError.
        # We can not raise ConfigError if this option should raise
        # PropertiesOptionError too. So we get config_error and raise
        # ConfigError if properties did not raise.
        config_error = None
        # if value has callback and is not set
        # or frozen with force_default_on_freeze
        if opt.impl_has_callback() and (
                self._is_default_owner(path, opt) or
                (is_frozen and 'force_default_on_freeze' in setting[opt])):
            lenmaster = None
            no_value_slave = False
            if (opt.impl_is_multi() and
                    opt.impl_get_multitype() == multitypes.slave):
                masterp = self._get_opt_path(opt.impl_get_master_slaves())
                mastervalue = context.getattr(masterp, validate=False,
                                              force_permissive=force_permissive)
                lenmaster = len(mastervalue)
                if lenmaster == 0:
                    value = []
                    no_value_slave = True

            if not no_value_slave:
                try:
                    value = self._getcallback_value(opt, max_len=lenmaster)
                except ConfigError as err:
                    # cannot assign config_err directly in python 3.3
                    config_error = err
                    value = None
                else:
                    if (opt.impl_is_multi() and
                            opt.impl_get_multitype() == multitypes.slave):
                        if not isinstance(value, list):
                            value = [value for i in range(lenmaster)]
            if config_error is None:
                if opt.impl_is_multi():
                    value = Multi(value, self.context, opt, path, validate,
                                  force_permissive=force_permissive)
        # frozen and force default
        elif is_frozen and 'force_default_on_freeze' in setting[opt]:
            value = self._getdefault(opt)
            if opt.impl_is_multi():
                value = Multi(value, self.context, opt, path, validate,
                              force_permissive=force_permissive)
        else:
            value = self._getvalue(opt, path)
            if opt.impl_is_multi():
                # load value so don't need to validate if is not a Multi
                value = Multi(value, self.context, opt, path, validate=validate,
                              force_permissive=force_permissive)
        if config_error is None and validate:
            try:
                opt.impl_validate(value, context, 'validator' in setting)
            except ValueError, err:
                config_error = err
                value = None
        if config_error is None and self._is_default_owner(path, opt) and \
                'force_store_value' in setting[opt]:
            if isinstance(value, Multi):
                item = list(value)
            else:
                item = value
            self.setitem(opt, item, path, is_write=False,
                         force_permissive=force_permissive)
        if validate_properties:
            if config_error is not None:
                # should not raise PropertiesOptionError if option is
                # mandatory
                if force_permissives is None:
                    force_permissives = set(['mandatory'])
                else:
                    force_permissives.add('mandatory')
            setting.validate_properties(opt, False, False, value=value, path=path,
                                        force_permissive=force_permissive,
                                        force_properties=force_properties,
                                        force_permissives=force_permissives)
        if config_error is not None:
            raise config_error
        return value

    def __setitem__(self, opt, value):
        raise ConfigError(_('you must only set value with config'))

    def setitem(self, opt, value, path, force_permissive=False,
                is_write=True):
        # is_write is, for example, used with "force_store_value"
        # user didn't change value, so not write
        # valid opt
        context = self._getcontext()
        opt.impl_validate(value, context,
                          'validator' in context.cfgimpl_get_settings())
        if opt.impl_is_multi():
            value = Multi(value, self.context, opt, path, setitem=True)
            # Save old value
            if opt.impl_get_multitype() == multitypes.master and \
                    self._p_.hasvalue(path):
                old_value = self._p_.getvalue(path)
                old_owner = self._p_.getowner(path, None)
            else:
                old_value = undefined
                old_owner = undefined
        self._setvalue(opt, path, value, force_permissive=force_permissive,
                       is_write=is_write)
        if opt.impl_is_multi() and opt.impl_get_multitype() == multitypes.master:
            try:
                value._valid_master(force_permissive=force_permissive)
            except Exception, err:
                if old_value is not undefined:
                    self._p_.setvalue(path, old_value, old_owner)
                else:
                    self._p_.resetvalue(path)
                raise err

    def _setvalue(self, opt, path, value, force_permissive=False,
                  force_properties=None,
                  is_write=True, validate_properties=True):
        context = self._getcontext()
        context.cfgimpl_reset_cache()
        if validate_properties:
            setting = context.cfgimpl_get_settings()
            setting.validate_properties(opt, False, is_write,
                                        value=value, path=path,
                                        force_permissive=force_permissive,
                                        force_properties=force_properties)
        owner = context.cfgimpl_get_settings().getowner()
        if isinstance(value, Multi):
            value = list(value)
        self._p_.setvalue(path, value, owner)

    def getowner(self, opt):
        """
        retrieves the option's owner

        :param opt: the `option.Option` object
        :returns: a `setting.owners.Owner` object
        """
        if isinstance(opt, SymLinkOption):
            opt = opt._opt
        path = self._get_opt_path(opt)
        return self._getowner(path, opt)

    def _getowner(self, path, opt):
        context = self._getcontext()
        setting = context.cfgimpl_get_settings()
        setting_opt = setting._getproperties(opt, path=path, read_only=True)
        if 'frozen' in setting_opt and \
                'force_default_on_freeze' in setting_opt:
            return owners.default
        owner = self._p_.getowner(path, owners.default)
        meta = self._getcontext().cfgimpl_get_meta()
        if owner is owners.default and meta is not None:
            owner = meta.cfgimpl_get_values()._getowner(path, opt)
        return owner

    def setowner(self, opt, owner):
        """
        sets a owner to an option

        :param opt: the `option.Option` object
        :param owner: a valid owner, that is a `setting.owners.Owner` object
        """
        if not isinstance(owner, owners.Owner):
            raise TypeError(_("invalid generic owner {0}").format(str(owner)))

        path = self._get_opt_path(opt)
        self._setowner(path, opt, owner)

    def _setowner(self, path, opt, owner):
        if not self._p_.hasvalue(path):
            raise ConfigError(_('no value for {0} cannot change owner to {1}'
                                '').format(path, owner))
        self._getcontext().cfgimpl_get_settings().validate_properties(opt,
                                                                      False,
                                                                      True,
                                                                      path)
        self._p_.setowner(path, owner)

    def is_default_owner(self, opt):
        """
        :param config: *must* be only the **parent** config
                       (not the toplevel config)
        :return: boolean
        """
        path = self._get_opt_path(opt)
        return self._is_default_owner(path, opt)

    def _is_default_owner(self, path, opt):
        return self._getowner(path, opt) == owners.default

    def reset_cache(self, only_expired):
        """
        clears the cache if necessary
        """
        if only_expired:
            self._p_.reset_expired_cache(int(time()))
        else:
            self._p_.reset_all_cache()

    def _get_opt_path(self, opt):
        """
        retrieve the option's path in the config

        :param opt: the `option.Option` object
        :returns: a string with points like "gc.dummy.my_option"
        """
        return self._getcontext().cfgimpl_get_description().impl_get_path_by_opt(opt)

    # information
    def set_information(self, key, value):
        """updates the information's attribute

        :param key: information's key (ex: "help", "doc"
        :param value: information's value (ex: "the help string")
        """
        self._p_.set_information(key, value)

    def get_information(self, key, default=None):
        """retrieves one information's item

        :param key: the item string (ex: "help")
        """
        try:
            return self._p_.get_information(key)
        except ValueError:
            if default is not None:
                return default
            else:
                raise ValueError(_("information's item not found: {0}").format(
                    key))

    def mandatory_warnings(self, force_permissive=False, is_apply_req=True,
                           validate=True):
        """convenience function to trace Options that are mandatory and
        where no value has been set

        :param force_permissive: do raise with permissives properties
        :type force_permissive: `bool`
        :param is_apply_req: apply requires when getting properties, calculated
                             properties will not apply
        :type is_apply_req: `bool`
        :param validate: validate value when calculating properties
        :type validate: `bool`

        :returns: generator of mandatory Option's path

        """
        #if value in cache, properties are not calculated
        self.reset_cache(False)
        context = self.context()
        values = context.cfgimpl_get_values()
        settings = context.cfgimpl_get_settings()

        def impl_getpaths(opt, _currpath=None):
            if _currpath is None:
                _currpath = []
            for option in opt.impl_getchildren():
                # symlink already tested with option
                if isinstance(option, SymLinkOption):
                    continue
                attr = option._name
                path = '.'.join(_currpath + [attr])
                if isinstance(option, OptionDescription):
                    try:
                        context.getattr(path,
                                        force_properties=frozenset(('mandatory',)),
                                        force_permissive=force_permissive)
                    except PropertiesOptionError:
                        pass
                    except Exception, err:
                        raise err
                    else:
                        for path_ in impl_getpaths(option, _currpath=_currpath + [attr]):
                            yield path_
                else:
                    prop = settings._getproperties(option, path, is_apply_req=is_apply_req, read_only=True)
                    if 'mandatory' in prop:
                        try:
                            values.getitem(opt=option, path=path, validate=validate,
                                           force_properties=frozenset(('mandatory',)),
                                           force_permissive=force_permissive)
                        except PropertiesOptionError as err:
                            if err.proptype == ['mandatory']:
                                yield path
                        except ConfigError as err:
                            if validate:
                                raise err
                            else:
                                #assume that uncalculated value is an empty value
                                yield path
        for path in impl_getpaths(context.cfgimpl_get_description()):
            yield path

    def force_cache(self):
        """parse all option to force data in cache
        """
        context = self.context()
        if not 'cache' in context.cfgimpl_get_settings():
            raise ConfigError(_('can force cache only if cache '
                                'is actived in config'))
        #remove all cached properties and value to update "expired" time
        context.cfgimpl_reset_cache()
        for path in context.cfgimpl_get_description().impl_getpaths(
                include_groups=True):
            try:
                context.getattr(path)
            except PropertiesOptionError:
                pass

    def __getstate__(self):
        return {'_p_': self._p_}

    def _impl_setstate(self, storage):
        self._p_._storage = storage

    def __setstate__(self, states):
        self._p_ = states['_p_']


# ____________________________________________________________
# multi types


class Multi(list):
    """multi options values container
    that support item notation for the values of multi options"""
    __slots__ = ('opt', 'path', 'context')

    def __init__(self, value, context, opt, path, validate=True,
                 setitem=False, force_permissive=False):
        """
        :param value: the Multi wraps a list value
        :param context: the home config that has the values
        :param opt: the option object that have this Multi value
        :param setitem: only if set a value
        :param force_permissive: force permissive
        """
        if isinstance(value, Multi):
            raise ValueError(_('{0} is already a Multi ').format(opt._name))
        self.opt = opt
        self.path = path
        if not isinstance(context, weakref.ReferenceType):
            raise ValueError('context must be a Weakref')
        self.context = context
        if not isinstance(value, list):
            value = [value]
        if validate and self.opt.impl_get_multitype() == multitypes.slave:
            value = self._valid_slave(value, setitem,
                                      force_permissive=force_permissive)
        elif not setitem and validate and \
                self.opt.impl_get_multitype() == multitypes.master:
            self._valid_master(force_permissive=force_permissive)
        super(Multi, self).__init__(value)

    def _getcontext(self):
        """context could be None, we need to test it
        context is None only if all reference to `Config` object is deleted
        (for example we delete a `Config` and we manipulate a reference to
        old `SubConfig`, `Values`, `Multi` or `Settings`)
        """
        context = self.context()
        if context is None:
            raise ConfigError(_('the context does not exist anymore'))
        return context

    def _valid_slave(self, value, setitem, force_permissive=False):
        #if slave, had values until master's one
        context = self._getcontext()
        values = context.cfgimpl_get_values()
        masterp = context.cfgimpl_get_description().impl_get_path_by_opt(
            self.opt.impl_get_master_slaves())
        mastervalue = context.getattr(masterp, validate=False,
                                      force_permissive=force_permissive)
        masterlen = len(mastervalue)
        valuelen = len(value)
        if valuelen > masterlen or (valuelen < masterlen and setitem):
            raise SlaveError(_("invalid len for the slave: {0}"
                               " which has {1} as master").format(
                                   self.opt._name, masterp))
        elif valuelen < masterlen:
            for num in range(0, masterlen - valuelen):
                if self.opt.impl_has_callback():
                    # if callback add a value, but this value will not change
                    # anymore automaticly (because this value has owner)
                    index = value.__len__()
                    value.append(values._getcallback_value(self.opt,
                                                           index=index))
                else:
                    value.append(self.opt.impl_getdefault_multi())
        #else: same len so do nothing
        return value

    def _valid_master(self, force_permissive=False):
        #masterlen = len(value)
        context = self._getcontext()
        values = context.cfgimpl_get_values()
        setting = context.cfgimpl_get_settings()
        for slave in self.opt._master_slaves:
            path = values._get_opt_path(slave)
            try:
                setting.validate_properties(slave, False, False, path,
                                            force_permissive=force_permissive)
            except PropertiesOptionError:
                continue
            Multi(values._getvalue(slave, path), self.context, slave, path,
                  force_permissive=force_permissive)

    def __setitem__(self, index, value):
        self._validate(value, index)
        #assume not checking mandatory property
        super(Multi, self).__setitem__(index, value)
        self._getcontext().cfgimpl_get_values()._setvalue(self.opt, self.path, self)

    def append(self, value=undefined, force=False):
        """the list value can be updated (appened)
        only if the option is a master
        :param force: store valid without valid/change value for slaves for
                      masterslaves
        :type force: bool
        """
        context = self._getcontext()
        index = self.__len__()
        if not force:
            if self.opt.impl_get_multitype() == multitypes.slave:
                raise SlaveError(_("cannot append a value on a multi option {0}"
                                   " which is a slave").format(self.opt._name))
            elif self.opt.impl_get_multitype() == multitypes.master:
                values = context.cfgimpl_get_values()
                if value is undefined and self.opt.impl_has_callback():
                    value = values._getcallback_value(self.opt, index=index)
        if value is undefined:
            value = self.opt.impl_getdefault_multi()
        self._validate(value, index)
        super(Multi, self).append(value)
        context.cfgimpl_get_values()._setvalue(self.opt, self.path,
                                               self,
                                               validate_properties=not force)
        if not force and self.opt.impl_get_multitype() == multitypes.master:
            setting = context.cfgimpl_get_settings()
            for slave in self.opt.impl_get_master_slaves():
                path = values._get_opt_path(slave)
                if not values._is_default_owner(path, slave):
                    try:
                        setting.validate_properties(slave, False, False, path)
                    except PropertiesOptionError:
                        continue
                    if slave.impl_has_callback():
                        dvalue = values._getcallback_value(slave, index=index)
                    else:
                        dvalue = slave.impl_getdefault_multi()
                    old_value = values.getitem(slave, path, validate=False,
                                               validate_properties=False)
                    if len(old_value) + 1 != self.__len__():
                        raise SlaveError(_("invalid len for the slave: {0}"
                                           " which has {1} as master").format(
                                               self.opt._name, self.__len__()))
                    values.getitem(slave, path, validate=False,
                                   validate_properties=False).append(
                                       dvalue, force=True)

    def sort(self, cmp=None, key=None, reverse=False):
        if self.opt.impl_get_multitype() in [multitypes.slave,
                                             multitypes.master]:
            raise SlaveError(_("cannot sort multi option {0} if master or slave"
                               "").format(self.opt._name))
        if sys.version_info[0] >= 3:
            if cmp is not None:
                raise ValueError(_('cmp is not permitted in python v3 or greater'))
            super(Multi, self).sort(key=key, reverse=reverse)
        else:
            super(Multi, self).sort(cmp=cmp, key=key, reverse=reverse)
        self._getcontext().cfgimpl_get_values()._setvalue(self.opt, self.path, self)

    def reverse(self):
        if self.opt.impl_get_multitype() in [multitypes.slave,
                                             multitypes.master]:
            raise SlaveError(_("cannot reverse multi option {0} if master or "
                               "slave").format(self.opt._name))
        super(Multi, self).reverse()
        self._getcontext().cfgimpl_get_values()._setvalue(self.opt, self.path, self)

    def insert(self, index, obj):
        if self.opt.impl_get_multitype() in [multitypes.slave,
                                             multitypes.master]:
            raise SlaveError(_("cannot insert multi option {0} if master or "
                               "slave").format(self.opt._name))
        super(Multi, self).insert(index, obj)
        self._getcontext().cfgimpl_get_values()._setvalue(self.opt, self.path, self)

    def extend(self, iterable):
        if self.opt.impl_get_multitype() in [multitypes.slave,
                                             multitypes.master]:
            raise SlaveError(_("cannot extend multi option {0} if master or "
                               "slave").format(self.opt._name))
        super(Multi, self).extend(iterable)
        self._getcontext().cfgimpl_get_values()._setvalue(self.opt, self.path, self)

    def _validate(self, value, force_index):
        if value is not None:
            try:
                self.opt.impl_validate(value, context=self._getcontext(),
                                       force_index=force_index)
            except ValueError as err:
                raise ValueError(_("invalid value {0} "
                                   "for option {1}: {2}"
                                   "").format(str(value),
                                              self.opt._name, err))

    def pop(self, index, force=False):
        """the list value can be updated (poped)
        only if the option is a master

        :param index: remove item a index
        :type index: int
        :param force: force pop item (withoud check master/slave)
        :type force: boolean
        :returns: item at index
        """
        context = self._getcontext()
        if not force:
            if self.opt.impl_get_multitype() == multitypes.slave:
                raise SlaveError(_("cannot pop a value on a multi option {0}"
                                   " which is a slave").format(self.opt._name))
            if self.opt.impl_get_multitype() == multitypes.master:
                values = context.cfgimpl_get_values()
                for slave in self.opt.impl_get_master_slaves():
                    if not values.is_default_owner(slave):
                        #get multi without valid properties
                        values.getitem(slave, validate=False,
                                       validate_properties=False
                                       ).pop(index, force=True)
        #set value without valid properties
        ret = super(Multi, self).pop(index)
        context.cfgimpl_get_values()._setvalue(self.opt, self.path, self, validate_properties=not force)
        return ret
