# coding: utf-8
"frozen and hidden values"
from autopath import do_autopath
do_autopath()

from py.test import raises

from tiramisu.setting import owners, groups
from tiramisu.option import ChoiceOption, BoolOption, IntOption, FloatOption, \
    StrOption, OptionDescription, SymLinkOption
from tiramisu.config import Config
from tiramisu.error import PropertiesOptionError, ConfigError


#____________________________________________________________
#freeze
def make_description_freeze():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=[True], multi=True)
    wantref_option = BoolOption('wantref', 'Test requires', default=False, properties=('force_store_value',),
                                requires=({'option': booloption, 'expected': True, 'action': 'hidden'},))
    wantref2_option = BoolOption('wantref2', 'Test requires', default=False, properties=('force_store_value', 'hidden'))
    wantref3_option = BoolOption('wantref3', 'Test requires', default=[False], multi=True, properties=('force_store_value',))
    st2 = SymLinkOption('st2', wantref3_option)
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False,
                                      requires=({'option': booloption, 'expected': True, 'action': 'hidden'},))

    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption])
    descr = OptionDescription('tiramisu', '', [gcgroup, booloption, objspaceoption,
                              wantref_option, wantref2_option, wantref3_option, st2, stroption,
                              wantframework_option,
                              intoption, boolop])
    return descr


def return_val():
    return 1


def return_val2(value):
    return value


def return_val3(context, value):
    return value


def test_freeze_whole_config():
    descr = make_description_freeze()
    conf = Config(descr)
    setting = conf.cfgimpl_get_settings()
    conf.read_write()
    setting.append('everything_frozen')
    assert conf.gc.dummy is False
    prop = []
    try:
        conf.gc.dummy = True
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'frozen' in prop
    assert conf.gc.dummy is False
    #
    setting.remove('everything_frozen')
    conf.gc.dummy = True
    assert conf.gc.dummy is True
    setting.append('everything_frozen')
    owners.addowner("everythingfrozen")
    prop = []
    try:
        conf.cfgimpl_get_values().setowner(descr.gc.dummy, owners.everythingfrozen)
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'frozen' in prop


def test_freeze_one_option():
    "freeze an option "
    descr = make_description_freeze()
    conf = Config(descr)
    setting = conf.cfgimpl_get_settings()
    conf.read_write()
    #freeze only one option
    dummy = conf.unwrap_from_path('gc.dummy')
    setting[dummy].append('frozen')
    assert conf.gc.dummy is False
    prop = []
    try:
        conf.gc.dummy = True
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'frozen' in prop


def test_frozen_value():
    "setattr a frozen value at the config level"
    s = StrOption("string", "", default="string")
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    setting.append('frozen')
    setting[s].append('frozen')
    prop = []
    try:
        config.string = "egg"
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'frozen' in prop


def test_freeze():
    "freeze a whole configuration object"
    descr = make_description_freeze()
    conf = Config(descr)
    setting = conf.cfgimpl_get_settings()
    conf.read_write()
    setting.append('frozen')
    name = conf.unwrap_from_path("gc.name")
    setting[name].append('frozen')
    prop = []
    try:
        conf.gc.name = 'framework'
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'frozen' in prop


def test_freeze_multi():
    descr = make_description_freeze()
    conf = Config(descr)
    setting = conf.cfgimpl_get_settings()
    conf.read_write()
    setting.append('frozen')
    obj = conf.unwrap_from_path('boolop')
    setting[obj].append('frozen')
    prop = []
    try:
        conf.boolop = [True]
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'frozen' in prop


def test_freeze_get_multi():
    descr = make_description_freeze()
    conf = Config(descr)
    setting = conf.cfgimpl_get_settings()
    conf.read_write()
    setting.append('frozen')
    valmulti = conf.boolop
    valmulti.append(False)
    obj = conf.unwrap_from_path('boolop')
    setting[obj].append('frozen')
    prop = []
    try:
        valmulti.append(False)
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'frozen' in prop


def test_force_store_value():
    descr = make_description_freeze()
    conf = Config(descr)
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'wantref': ('forced', False),
                                                                   'wantref2': ('forced', False),
                                                                   'wantref3': ('forced', (False,))}
    conf.wantref = True
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'wantref': ('user', True),
                                                                   'wantref2': ('forced', False),
                                                                   'wantref3': ('forced', (False,))}
    del(conf.wantref)
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'wantref': ('forced', False),
                                                                   'wantref2': ('forced', False),
                                                                   'wantref3': ('forced', (False,))}


def test_force_store_value_no_requirement():
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    try:
        BoolOption('wantref', 'Test requires', default=False,
                   requires=({'option': booloption, 'expected': True, 'action': 'force_store_value'},))
    except ValueError:
        pass


def test_force_store_value_masterslaves_slave():
    b = IntOption('int', 'Test int option', multi=True)
    c = StrOption('str', 'Test string option', multi=True, properties=('force_store_value',))
    descr = OptionDescription("int", "", [b, c])
    descr.impl_set_group_type(groups.master)
    raises(ConfigError, "conf = Config(descr)")


def test_force_store_value_masterslaves():
    b = IntOption('int', 'Test int option', multi=True, properties=('force_store_value',))
    c = StrOption('str', 'Test string option', multi=True)
    descr = OptionDescription("int", "", [b, c])
    descr.impl_set_group_type(groups.master)
    conf = Config(descr)
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'int': ('forced', ())}


def test_force_store_value_callback():
    b = IntOption('int', 'Test int option', properties=('force_store_value',), callback=return_val)
    descr = OptionDescription("int", "", [b])
    conf = Config(descr)
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'int': ('forced', 1)}


def test_force_store_value_callback_params():
    b = IntOption('int', 'Test int option', properties=('force_store_value',), callback=return_val2, callback_params={'value': (2,)})
    descr = OptionDescription("int", "", [b])
    conf = Config(descr)
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'int': ('forced', 2)}


def test_force_store_value_callback_params_2():
    b = IntOption('int', 'Test int option', properties=('force_store_value',), callback=return_val3, callback_params={'': ((None,),), 'value': (2,)})
    descr = OptionDescription("int", "", [b])
    conf = Config(descr)
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'int': ('forced', 2)}


def test_force_store_value_callback_params_with_opt():
    a = IntOption('val1', "", 2)
    b = IntOption('int', 'Test int option', properties=('force_store_value',), callback=return_val2, callback_params={'value': ((a, False),)})
    descr = OptionDescription("int", "", [a, b])
    conf = Config(descr)
    assert conf.cfgimpl_get_values()._p_.get_modified_values() == {'int': ('forced', 2)}
