# coding: utf-8
from autopath import do_autopath
do_autopath()

from tiramisu.setting import groups, owners
from tiramisu.config import Config
from tiramisu.option import ChoiceOption, BoolOption, IntOption, \
    StrOption, OptionDescription
from tiramisu.error import SlaveError, PropertiesOptionError

from py.test import raises


def make_description():
    numero_etab = StrOption('numero_etab', "identifiant de l'établissement")
    nom_machine = StrOption('nom_machine', "nom de la machine", default="eoleng")
    nombre_interfaces = IntOption('nombre_interfaces', "nombre d'interfaces à activer",
                                  default=1)
    activer_proxy_client = BoolOption('activer_proxy_client', "utiliser un proxy",
                                      default=False)
    mode_conteneur_actif = BoolOption('mode_conteneur_actif', "le serveur est en mode conteneur",
                                      default=False)
    mode_conteneur_actif2 = BoolOption('mode_conteneur_actif2', "le serveur est en mode conteneur2",
                                       default=False, properties=('hidden',))

    adresse_serveur_ntp = StrOption('serveur_ntp', "adresse serveur ntp", multi=True)
    time_zone = ChoiceOption('time_zone', 'fuseau horaire du serveur',
                             ('Paris', 'Londres'), 'Paris')

    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé")
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau")

    master = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1 = OptionDescription('interface1', '', [master])
    interface1.impl_set_group_type(groups.family)

    general = OptionDescription('general', '', [numero_etab, nom_machine,
                                nombre_interfaces, activer_proxy_client,
                                mode_conteneur_actif, mode_conteneur_actif2,
                                adresse_serveur_ntp, time_zone])
    general.impl_set_group_type(groups.family)
    new = OptionDescription('new', '', [], properties=('hidden',))
    new.impl_set_group_type(groups.family)
    creole = OptionDescription('creole', 'first tiramisu configuration', [general, interface1, new])
    descr = OptionDescription('baseconfig', 'baseconifgdescr', [creole])
    return descr


def test_base_config():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    assert config.creole.general.activer_proxy_client is False
    assert config.creole.general.nom_machine == "eoleng"
    assert config.find_first(byname='nom_machine', type_='value') == "eoleng"
    result = {'general.numero_etab': None, 'general.nombre_interfaces': 1,
              'general.serveur_ntp': [], 'interface1.ip_admin_eth0.ip_admin_eth0': None,
              'general.mode_conteneur_actif': False, 'general.time_zone': 'Paris',
              'interface1.ip_admin_eth0.netmask_admin_eth0': None, 'general.nom_machine':
              'eoleng', 'general.activer_proxy_client': False}
    assert config.creole.make_dict() == result
    result = {'serveur_ntp': [], 'mode_conteneur_actif': False,
              'ip_admin_eth0': None, 'time_zone': 'Paris', 'numero_etab': None,
              'netmask_admin_eth0': None, 'nom_machine': 'eoleng', 'activer_proxy_client':
              False, 'nombre_interfaces': 1}
    assert config.creole.make_dict(flatten=True) == result


def test_make_dict_filter():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    subresult = {'numero_etab': None, 'nombre_interfaces': 1,
                 'serveur_ntp': [], 'mode_conteneur_actif': False,
                 'time_zone': 'Paris', 'nom_machine': 'eoleng',
                 'activer_proxy_client': False}
    result = {}
    for key, value in subresult.items():
        result['general.' + key] = value
    assert config.creole.make_dict(withoption='numero_etab') == result
    raises(AttributeError, "config.creole.make_dict(withoption='numero_etab', withvalue='toto')")
    assert config.creole.make_dict(withoption='numero_etab', withvalue=None) == result
    assert config.creole.general.make_dict(withoption='numero_etab') == subresult


def test_get_group_type():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    grp = config.unwrap_from_path('creole.general')
    assert grp.impl_get_group_type() == groups.family
    assert grp.impl_get_group_type() == 'family'
    assert isinstance(grp.impl_get_group_type(), groups.GroupType)
    raises(TypeError, 'grp.impl_set_group_type(groups.default)')


def test_iter_on_groups():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    result = list(config.creole.iter_groups(group_type=groups.family))
    group_names = [res[0] for res in result]
    assert group_names == ['general', 'interface1']
    for i in config.creole.iter_groups(group_type=groups.family):
        #test StopIteration
        break


def test_iter_on_groups_force_permissive():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    config.cfgimpl_get_settings().setpermissive(('hidden',))
    result = list(config.creole.general.__iter__(force_permissive=True))
    group_names = [res[0] for res in result]
    ass = ['numero_etab', 'nom_machine', 'nombre_interfaces',
           'activer_proxy_client', 'mode_conteneur_actif',
           'mode_conteneur_actif2', 'serveur_ntp', 'time_zone']
    assert group_names == ass
    group_names = [res[0] for res in config.creole.general]
    ass.remove('mode_conteneur_actif2')
    assert group_names == ass


def test_iter_group_on_groups_force_permissive():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    config.cfgimpl_get_settings().setpermissive(('hidden',))
    result = list(config.creole.iter_groups(group_type=groups.family,
                                            force_permissive=True))
    group_names = [res[0] for res in result]
    assert group_names == ['general', 'interface1', 'new']


def test_iter_on_groups_props():
    descr = make_description()
    config = Config(descr)
    config.read_write()
    config.cfgimpl_get_settings()[descr.creole.interface1].append('disabled')
    result = list(config.creole.iter_groups(group_type=groups.family))
    group_names = [res[0] for res in result]
    assert group_names == ['general']


def test_iter_on_empty_group():
    config = Config(OptionDescription("name", "descr", []))
    config.read_write()
    result = list(config.iter_groups())
    assert result == []
    for i in config.iter_groups():
        pass
    for i in config:
        pass
    assert [] == list(config)


def test_iter_not_group():
    config = Config(OptionDescription("name", "descr", []))
    config.read_write()
    raises(TypeError, "list(config.iter_groups(group_type='family'))")


def test_groups_with_master():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    assert interface1.impl_get_group_type() == groups.master


def test_groups_with_master_in_config():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    Config(interface1)
    assert interface1.impl_get_group_type() == groups.master


def test_groups_with_master_hidden_in_config():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True, properties=('hidden',))
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    cfg = Config(interface1)
    cfg.read_write()
    cfg.cfgimpl_get_settings().setpermissive(('hidden',))
    cfg.getattr('ip_admin_eth0', force_permissive=True)
    cfg.getattr('netmask_admin_eth0', force_permissive=True)
    raises(PropertiesOptionError, "cfg.getattr('ip_admin_eth0')")
    raises(PropertiesOptionError, "cfg.getattr('netmask_admin_eth0')")


def test_groups_with_master_hidden_in_config2():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    cfg = Config(interface1)
    cfg.read_write()
    cfg.cfgimpl_get_settings().setpermissive(('hidden',))
    cfg.getattr('ip_admin_eth0', force_permissive=True)
    cfg.getattr('netmask_admin_eth0', force_permissive=True)
    cfg.getattr('ip_admin_eth0')
    raises(PropertiesOptionError, "cfg.getattr('netmask_admin_eth0')")
    cfg.ip_admin_eth0.append('192.168.1.1')
    assert cfg.ip_admin_eth0 == ['192.168.1.1']
    raises(PropertiesOptionError, "cfg.netmask_admin_eth0")
    del(cfg.ip_admin_eth0)
    assert cfg.ip_admin_eth0 == []
    #del
    cfg.ip_admin_eth0.append('192.168.1.1')
    assert cfg.ip_admin_eth0 == ['192.168.1.1']
    cfg.cfgimpl_get_settings().remove('hidden')
    assert cfg.netmask_admin_eth0 == [None]
    cfg.netmask_admin_eth0 = ['255.255.255.0']
    assert cfg.netmask_admin_eth0 == ['255.255.255.0']
    cfg.cfgimpl_get_settings().append('hidden')
    del(cfg.ip_admin_eth0)
    cfg.ip_admin_eth0.append('192.168.1.1')
    cfg.cfgimpl_get_settings().remove('hidden')
    assert cfg.netmask_admin_eth0 == [None]


def test_groups_with_master_hidden_in_config3():
    #if master is hidden, slave are hidden too
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True, properties=('hidden',))
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    cfg = Config(interface1)
    cfg.read_write()
    cfg.cfgimpl_get_settings().setpermissive(('hidden',))
    cfg.getattr('ip_admin_eth0', force_permissive=True)
    cfg.getattr('netmask_admin_eth0', force_permissive=True)
    raises(PropertiesOptionError, "cfg.getattr('ip_admin_eth0')")
    raises(PropertiesOptionError, "cfg.getattr('netmask_admin_eth0')")


def test_allowed_groups():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1
    raises(ValueError, "interface1.impl_set_group_type('toto')")


def test_values_with_master_disabled_master():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.ip_admin_eth0.pop(0)
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.netmask_admin_eth0 = ["192.168.230.145"]
    del(cfg.ip_admin_eth0.netmask_admin_eth0)
    cfg.cfgimpl_get_settings()[ip_admin_eth0].append('disabled')
    raises(PropertiesOptionError, "cfg.ip_admin_eth0.netmask_admin_eth0[0] = '192.168.230.145'")


def test_master_not_valid_name():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    invalid_group = OptionDescription('interface1', '', [ip_admin_eth0, netmask_admin_eth0])
    invalid_group
    raises(ValueError, "invalid_group.impl_set_group_type(groups.master)")


def test_sub_group_in_master_group():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    subgroup = OptionDescription("subgroup", '', [])
    invalid_group = OptionDescription('ip_admin_eth0', '', [subgroup, ip_admin_eth0, netmask_admin_eth0])
    invalid_group
    raises(ValueError, "invalid_group.impl_set_group_type(groups.master)")


def test_group_always_has_multis():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau")
    group = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    group
    raises(ValueError, "group.impl_set_group_type(groups.master)")


#____________________________________________________________
def test_values_with_master_and_slaves():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    owner = cfg.cfgimpl_get_settings().getowner()
    assert interface1.impl_get_group_type() == groups.master
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == []
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    assert cfg.ip_admin_eth0.ip_admin_eth0 == ["192.168.230.145"]
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == [None]
    assert cfg.getowner(ip_admin_eth0) == owner
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    cfg.ip_admin_eth0.ip_admin_eth0 = ["192.168.230.145", "192.168.230.147"]
    raises(SlaveError, 'cfg.ip_admin_eth0.netmask_admin_eth0.append(None)')
    raises(SlaveError, 'cfg.ip_admin_eth0.netmask_admin_eth0.pop(0)')


def test_reset_values_with_master_and_slaves():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    owner = cfg.cfgimpl_get_settings().getowner()
    assert interface1.impl_get_group_type() == groups.master
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    assert cfg.getowner(ip_admin_eth0) == owner
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    del(cfg.ip_admin_eth0.ip_admin_eth0)
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    assert cfg.ip_admin_eth0.ip_admin_eth0 == []
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == []
    #reset
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.cfgimpl_get_values().reset(ip_admin_eth0)
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    assert cfg.ip_admin_eth0.ip_admin_eth0 == []
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == []


def test_reset_values_with_master_and_slaves_default():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True, default=['192.168.230.145'])
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    owner = cfg.cfgimpl_get_settings().getowner()
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default

    cfg.ip_admin_eth0.ip_admin_eth0[0] = "192.168.230.146"
    assert cfg.getowner(ip_admin_eth0) == owner
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    del(cfg.ip_admin_eth0.ip_admin_eth0)
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    assert cfg.ip_admin_eth0.ip_admin_eth0 == ['192.168.230.145']
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == [None]

    cfg.ip_admin_eth0.ip_admin_eth0[0] = "192.168.230.146"
    cfg.ip_admin_eth0.netmask_admin_eth0[0] = "255.255.255.0"
    assert cfg.getowner(ip_admin_eth0) == owner
    assert cfg.getowner(netmask_admin_eth0, 0) == owner
    del(cfg.ip_admin_eth0.ip_admin_eth0)
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0, 0) == owners.default
    assert cfg.ip_admin_eth0.ip_admin_eth0 == ['192.168.230.145']
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == [None]

    cfg.ip_admin_eth0.netmask_admin_eth0[0] = "255.255.255.0"
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0, 0) == owner
    del(cfg.ip_admin_eth0.ip_admin_eth0)
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    assert cfg.ip_admin_eth0.ip_admin_eth0 == ['192.168.230.145']
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == [None]


def test_reset_values_with_master_and_slaves_setitem():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True, default=['192.168.230.145'])
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    owner = cfg.cfgimpl_get_settings().getowner()
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default

    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.146")
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.146")
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.146")
    assert cfg.ip_admin_eth0.netmask_admin_eth0[-1] == None
    assert cfg.ip_admin_eth0.netmask_admin_eth0[3] == None
    assert cfg.getowner(netmask_admin_eth0, 3) == owners.default
    cfg.ip_admin_eth0.netmask_admin_eth0[-1] = "255.255.255.0"
    assert cfg.ip_admin_eth0.netmask_admin_eth0[-1] == "255.255.255.0"
    assert cfg.ip_admin_eth0.netmask_admin_eth0[3] == "255.255.255.0"
    assert cfg.getowner(netmask_admin_eth0, 3) == owner
    #
    assert cfg.getowner(netmask_admin_eth0, 2) == owners.default
    cfg.ip_admin_eth0.netmask_admin_eth0[2] = "255.255.0.0"
    assert cfg.ip_admin_eth0.netmask_admin_eth0[-2] == "255.255.0.0"
    assert cfg.getowner(netmask_admin_eth0, 2) == owner


def test_values_with_master_and_slaves_slave():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == []
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0']")
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0']
    cfg.ip_admin_eth0.netmask_admin_eth0[0] = '255.255.255.0'
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0', '255.255.255.0']")
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0 = []")
    del(cfg.ip_admin_eth0.netmask_admin_eth0)
    cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0']
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == ['255.255.255.0', None]
    cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0', '255.255.255.0']
    raises(SlaveError, 'cfg.ip_admin_eth0.netmask_admin_eth0.pop(1)')
    #reset
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.cfgimpl_get_values().reset(ip_admin_eth0)
    assert cfg.ip_admin_eth0.ip_admin_eth0 == []
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == []


def test_values_with_master_and_slaves_master():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.ip_admin_eth0 = ["192.168.230.145"]
    cfg.ip_admin_eth0.ip_admin_eth0 = ["192.168.230.145", "192.168.230.145"]
    cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0', '255.255.255.0']
    raises(SlaveError, 'cfg.ip_admin_eth0.ip_admin_eth0 = ["192.168.230.145"]')
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == ['255.255.255.0', '255.255.255.0']
    cfg.ip_admin_eth0.ip_admin_eth0.pop(1)
    assert cfg.ip_admin_eth0.ip_admin_eth0 == ["192.168.230.145"]
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == ['255.255.255.0']
    del(cfg.ip_admin_eth0.ip_admin_eth0)
    assert cfg.ip_admin_eth0.ip_admin_eth0 == []
    assert cfg.ip_admin_eth0.netmask_admin_eth0 == []


def test_values_with_master_and_slaves_master_error():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    cfg.ip_admin_eth0.ip_admin_eth0 = ["192.168.230.145", "192.168.230.145"]
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0']")
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0', '255.255.255.0', '255.255.255.0']")
    cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0', '255.255.255.0']
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0']")
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0 = ['255.255.255.0', '255.255.255.0', '255.255.255.0']")


def test_values_with_master_owner():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    owner = cfg.cfgimpl_get_settings().getowner()
    assert cfg.getowner(ip_admin_eth0) == owners.default
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    assert cfg.getowner(ip_admin_eth0) == owner
    assert cfg.getowner(netmask_admin_eth0) == owners.default
    cfg.ip_admin_eth0.ip_admin_eth0.pop(0)
    assert cfg.getowner(ip_admin_eth0) == owner
    assert cfg.getowner(netmask_admin_eth0) == owners.default


def test_values_with_master_disabled():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.ip_admin_eth0.pop(0)
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.netmask_admin_eth0 = ["192.168.230.145"]
    cfg.ip_admin_eth0.ip_admin_eth0.pop(0)
    del(cfg.ip_admin_eth0.netmask_admin_eth0)
    cfg.cfgimpl_get_settings()[netmask_admin_eth0].append('disabled')
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.ip_admin_eth0.pop(0)

    #delete with value in disabled var
    cfg.cfgimpl_get_settings()[netmask_admin_eth0].remove('disabled')
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.netmask_admin_eth0 = ["192.168.230.145"]
    cfg.cfgimpl_get_settings()[netmask_admin_eth0].append('disabled')
    cfg.ip_admin_eth0.ip_admin_eth0.pop(0)

    #append with value in disabled var
    cfg.cfgimpl_get_settings()[netmask_admin_eth0].remove('disabled')
    cfg.ip_admin_eth0.ip_admin_eth0.append("192.168.230.145")
    cfg.ip_admin_eth0.netmask_admin_eth0 = ["192.168.230.145"]
    cfg.cfgimpl_get_settings()[netmask_admin_eth0].append('disabled')
    cfg.ip_admin_eth0.ip_admin_eth0.append('192.168.230.43')


def test_multi_insert():
    var = StrOption('var', '', ['ok'], multi=True)
    od = OptionDescription('od', '', [var])
    c = Config(od)
    c.read_write()
    assert c.var == ['ok']
    assert c.getowner(var) == owners.default
    c.var.insert(0, 'nok')
    assert c.var == ['nok', 'ok']
    assert c.getowner(var) != owners.default
    raises(ValueError, 'c.var.insert(0, 1)')


def test_multi_insert_master():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(SlaveError, "cfg.ip_admin_eth0.ip_admin_eth0.insert(0, 'nok')")
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0.insert(0, 'nok')")


def test_multi_sort():
    var = StrOption('var', '', ['ok', 'nok'], multi=True)
    od = OptionDescription('od', '', [var])
    c = Config(od)
    c.read_write()
    assert c.var == ['ok', 'nok']
    assert c.getowner(var) == owners.default
    c.var.sort()
    assert c.var == ['nok', 'ok']
    assert c.getowner(var) != owners.default


def test_multi_sort_master():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(SlaveError, "cfg.ip_admin_eth0.ip_admin_eth0.sort()")
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0.sort()")


def test_multi_reverse():
    var = StrOption('var', '', ['ok', 'nok'], multi=True)
    od = OptionDescription('od', '', [var])
    c = Config(od)
    c.read_write()
    assert c.var == ['ok', 'nok']
    assert c.getowner(var) == owners.default
    c.var.reverse()
    assert c.var == ['nok', 'ok']
    assert c.getowner(var) != owners.default


def test_multi_reverse_master():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(SlaveError, "cfg.ip_admin_eth0.ip_admin_eth0.reverse()")
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0.reverse()")


def test_multi_extend():
    var = StrOption('var', '', ['ok', 'nok'], multi=True)
    od = OptionDescription('od', '', [var])
    c = Config(od)
    c.read_write()
    assert c.var == ['ok', 'nok']
    assert c.getowner(var) == owners.default
    c.var.extend(['pok'])
    assert c.var == ['ok', 'nok', 'pok']
    assert c.getowner(var) != owners.default
    raises(ValueError, 'c.var.extend([1])')


def test_multi_extend_master():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    raises(SlaveError, "cfg.ip_admin_eth0.ip_admin_eth0.extend(['ok'])")
    raises(SlaveError, "cfg.ip_admin_eth0.netmask_admin_eth0.extend(['ok'])")


def test_multi_non_valid_value():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    maconfig = OptionDescription('toto', '', [ip_admin_eth0])
    cfg = Config(maconfig)
    cfg.read_write()
    cfg.ip_admin_eth0 = ['a']
    raises(ValueError, 'cfg.ip_admin_eth0[0] = 1')


def test_multi_master_default_slave():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip réseau autorisé", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "masque du sous-réseau", default_multi="255.255.255.0", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    maconfig = OptionDescription('toto', '', [interface1])
    cfg = Config(maconfig)
    cfg.read_write()
    cfg.ip_admin_eth0.ip_admin_eth0.append('192.168.1.1')
    cfg.cfgimpl_reset_cache()
    assert cfg.ip_admin_eth0.ip_admin_eth0 == ['192.168.1.1']
