# -*- coding: utf-8 -*-
# Copyright (C) 2012-2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________
"user defined exceptions"


# Exceptions for an Option
class PropertiesOptionError(AttributeError):
    "attempt to access to an option with a property that is not allowed"
    def __init__(self, msg, proptype):
        self.proptype = proptype
        super(PropertiesOptionError, self).__init__(msg)


#____________________________________________________________
# Exceptions for a Config
class ConfigError(Exception):
    """attempt to change an option's owner without a value
    or in case of `_cfgimpl_descr` is None
    or if a calculation cannot be carried out"""
    pass


class ConflictError(Exception):
    "duplicate options are present in a single config"
    pass


#____________________________________________________________
# miscellaneous exceptions
class RequirementError(Exception):
    """a recursive loop occurs in the requirements tree
    requires
    """
    pass


class SlaveError(Exception):
    "problem with a slave's value length"
    pass


class ConstError(TypeError):
    "no uniq value in _NameSpace"
    pass


#Warning
class ValueWarning(UserWarning):  # pragma: optional cover
    """Option could warn user and not raise ValueError.

    Example:

    >>> import warnings
    >>> from tiramisu.error import ValueWarning
    >>> from tiramisu.option import StrOption, OptionDescription
    >>> from tiramisu.config import Config
    >>> warnings.simplefilter("always", ValueWarning)
    >>> def a(val):
    ...  raise ValueError('pouet')
    ...
    >>> s=StrOption('s', '', validator=a, warnings_only=True)
    >>> o=OptionDescription('o', '', [s])
    >>> c=Config(o)
    >>> c.s = 'val'
    StrOption:0: ValueWarning: invalid value val for option s: pouet
    >>> with warnings.catch_warnings(record=True) as w:
    ...     c.s = 'val'
    ...
    >>> w[0].message.opt == s
    True
    >>> print str(w[0].message)
    invalid value val for option s: pouet
    """
    def __init__(self, msg, opt):
        self.opt = opt
        super(ValueWarning, self).__init__(msg)
