# -*- coding: utf-8 -*-
"default plugin for setting: set it in a simple dictionary"
# Copyright (C) 2014 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________
from ..util import Cache
from .util import SqlAlchemyBase
import util
from sqlalchemy import Column, Integer, String, PickleType, ForeignKey
from sqlalchemy.orm import relationship
from sqlalchemy.ext.associationproxy import association_proxy
from sqlalchemy.orm.collections import attribute_mapped_collection


#____________________________________________________________
#
# properties|permissives
class _Property(SqlAlchemyBase):
    __tablename__ = 'property'
    id = Column(Integer, primary_key=True)
    setting = Column(Integer, ForeignKey('settings.id'), nullable=False)
    path = Column(String)
    properties = Column(PickleType)

    def __init__(self, path, properties):
        self.path = path
        self.properties = properties


class _Permissive (SqlAlchemyBase):
    __tablename__ = 'permissive'
    id = Column(Integer, primary_key=True)
    setting = Column(Integer, ForeignKey('settings.id'), nullable=False)
    path = Column(String)
    permissives = Column(PickleType)

    def __init__(self, path, permissives):
        self.path = path
        self.permissives = permissives


#____________________________________________________________
#FIXME marche pas le cache ... de toute facon je vais faire un storage separe !
class Settings(Cache, SqlAlchemyBase):
    __tablename__ = 'settings'
    id = Column(Integer, primary_key=True)
    session_id = Column(String, index=True)
    _props = relationship("_Property",
                          collection_class=attribute_mapped_collection('path'),
                          cascade="all, delete-orphan")
    _properties = association_proxy("_props", "properties")
    _perms = relationship("_Permissive",
                          collection_class=attribute_mapped_collection('path'),
                          cascade="all, delete-orphan")
    _permissives = association_proxy("_perms", "permissives")

    def __init__(self, session_id, storage):
        session = self.getsession()
        self.session_id = session_id
        super(Settings, self).__init__(storage)
        session.commit()

    def getsession(self):
        return util.Session()

    # properties
    def setproperties(self, path, properties):
        session = self.getsession()
        self._properties[path] = properties
        session.commit()
        del(session)

    def getproperties(self, path, default_properties):
        return self._properties.get(path, set(default_properties))

    def hasproperties(self, path):
        return path in self._properties

    def reset_all_properties(self):
        session = self.getsession()
        self._properties.clear()
        session.commit()
        del(session)

    def delproperties(self, path):
        try:
            session = self.getsession()
            del(self._properties[path])
            session.commit()
            del(session)
        except KeyError:
            pass

    # permissive
    def setpermissive(self, path, permissive):
        session = self.getsession()
        self._permissives[path] = frozenset(permissive)
        session.commit()

    def getpermissive(self, path=None):
        ret = self._permissives.get(path, frozenset())
        #replace None by a frozenset()
        return {None: frozenset()}.get(ret, ret)

    def get_modified_properties(self):
        """return all modified settings in a dictionary
        example: {'path1': set(['prop1', 'prop2'])}
        """
        return self._properties

    def get_modified_permissives(self):
        """return all modified permissives in a dictionary
        example: {'path1': set(['perm1', 'perm2'])}
        """
        return self._permissives


def delete_session(session_id, session):
    print session.query(_Property).all()
    print session.query(_Permissive).all()
    settings_id = session.query(Settings).filter_by(session_id=session_id).first().id
    for val in session.query(_Property).filter_by(settings=settings_id).all():
        session.delete(val)
    for val in session.query(_Permissive).filter_by(settings=settings_id).all():
        session.delete(val)
