# -*- coding: utf-8 -*-
"plugin for value: set it in sqlalchemy"
# Copyright (C) 2013-2014 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________


from ..util import Cache
from .util import SqlAlchemyBase
import util
from ...setting import undefined
from sqlalchemy import Column, Integer, String, PickleType
from sqlalchemy import func
from tiramisu.setting import owners


#____________________________________________________________
#
# information
class _Vinformation(SqlAlchemyBase):
    __tablename__ = 'vinformation'
    id = Column(Integer, primary_key=True)
    session_id = Column(String, index=True)
    path = Column(String, index=True)
    key = Column(String)
    value = Column(PickleType)

    def __init__(self, session_id, key, value):
        self.session_id = session_id
        self.key = key
        self.value = value


class Value(SqlAlchemyBase):
    __tablename__ = 'value'
    id = Column(Integer, primary_key=True)
    session_id = Column(String, index=True)
    path = Column(String, index=True)
    key = Column(String)
    value = Column(PickleType)
    owner = Column(String)
    indx = Column(Integer, index=True)

    def __init__(self, session_id, path, value, owner, index):
        self.session_id = session_id
        self.path = path
        self.value = value
        self.owner = owner
        self.indx = index


class Values(Cache):

    def getsession(self):
        return util.Session()

    # value
    def setvalue(self, path, value, owner, index, session):
        """set value for a path
        a specified value must be associated to an owner
        """
        #if it's a multi
        if isinstance(value, list):
            value = list(value)
        val = session.query(Value).filter_by(
            path=path, indx=index, session_id=self._storage.session_id).first()
        if val is None:
            session.add(Value(self._storage.session_id, path, value,
                                   owner, index))
        else:
            val.value = value
            val.owner = owner
        session.commit()

    def getvalue(self, path, session, index=None):
        """get value for a path
        return: only value, not the owner
        """
        val = session.query(Value).filter_by(
            path=path, indx=index, session_id=self._storage.session_id).first()
        if not val:
            raise KeyError('no value found')
        return val.value

    def hasvalue(self, path, session):
        """if path has a value
        return: boolean
        """
        return session.query(Value).filter_by(
                  path=path, session_id=self._storage.session_id).first() is not None

    def resetvalue(self, path, session):
        """remove value means delete value in storage
        """
        vals = session.query(Value).filter_by(
            path=path, session_id=self._storage.session_id).all()
        if vals != []:
            for val in vals:
                session.delete(val)
            session.commit()

    def get_modified_values(self):
        """return all values in a dictionary
        example: {'path1': (owner, 'value1'), 'path2': (owner, 'value2')}
        """
        session = self.getsession()
        ret = {}
        for val in session.query(Value).filter_by(
                session_id=self._storage.session_id).all():
            value = val.value
            if isinstance(val.value, list):
                value = tuple(val.value)
            ret[val.path] = (val.owner, value)
        del(session)
        return ret

    # owner
    def setowner(self, path, owner, session, index=None):
        """change owner for a path
        """
        val = session.query(Value).filter_by(
            path=path, indx=index, session_id=self._storage.session_id).first()
        if val is None:
            raise KeyError('no value found')
        else:
            val.owner = owner
        session.commit()

    def get_max_length(self, path, session):
        val = session.query(Value, func.max(Value.indx)).filter_by(
                    path=path, session_id=self._storage.session_id).first()
        if val[1] is None:
            maxval = 0
        else:
            maxval = val[1] + 1
        return maxval

    def getowner(self, path, default, session, index=None, only_default=False):
        #FIXME support de only_default
        """get owner for a path
        return: owner object
        """
        session.commit()
        val = session.query(Value).filter_by(
            path=path, session_id=self._storage.session_id,
            indx=index).first()
        if val is None:
            return default
        else:
            owner = val.owner
            # autocreate owners
            try:
                return getattr(owners, owner)
            except AttributeError:
                owners.addowner(owner)
                return getattr(owners, owner)

    def set_information(self, key, value):
        """updates the information's attribute
        (which is a dictionary)

        :param key: information's key (ex: "help", "doc"
        :param value: information's value (ex: "the help string")
        """
        session = self.getsession()
        val = session.query(_Vinformation).filter_by(
            key=key, session_id=self._storage.session_id).first()
        if val is None:
            session.add(_Vinformation(self._storage.session_id, key,
                                           value))
        else:
            val.value = value
        session.commit()
        del(session)

    def get_information(self, key, default):
        """retrieves one information's item

        :param key: the item string (ex: "help")
        """
        session = self.getsession()
        val = session.query(_Vinformation).filter_by(
            key=key, session_id=self._storage.session_id).first()
        del(session)
        if not val:
            if default is not undefined:
                return default
            raise ValueError("not found")
        return val.value

    def exportation(self, session, fake=False):
        if fake:
            #(('path1',), (index1,), (value1,), ('owner1'))
            paths = []
            indexes = []
            values = []
            owners_ = []
            slaves = {}
            for val in session.query(Value).filter_by(
                    session_id=self._storage.session_id).all():
                if val.indx is not None:
                    slaves.setdefault(val.path, []).append((val.indx, val.value, getattr(owners, val.owner)))
                else:
                    paths.append(val.path)
                    indexes.append(val.indx)
                    values.append(val.value)
                    owners_.append(getattr(owners, val.owner))
            for path, vals in slaves.items():
                paths.append(path)
                t_idxes = []
                t_vals = []
                t_owners = []
                for val in vals:
                    t_idxes.append(val[0])
                    t_vals.append(val[1])
                    t_owners.append(val[2])
                indexes.append(tuple(t_idxes))
                values.append(t_vals)
                owners_.append(t_owners)
            return (paths, indexes, values, owners_)
        pass

    def importation(self, value):
        pass


def delete_session(session_id, session):
    for val in session.query(_Vinformation).filter_by(session_id=session_id).all():
        session.delete(val)
    for val in session.query(Value).filter_by(session_id=session_id).all():
        session.delete(val)
