# -*- coding: utf-8 -*-
"default plugin for value: set it in a simple dictionary"
# Copyright (C) 2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________

from .sqlite3db import Sqlite3DB
from tiramisu.setting import owners


class Values(Sqlite3DB):
    __slots__ = ('__weakref__',)

    def __init__(self, storage):
        """init plugin means create values storage
        """
        # should init cache too
        super(Values, self).__init__(storage)
        values_table = 'CREATE TABLE IF NOT EXISTS value(path text primary '
        values_table += 'key, value text, owner text)'
        self._storage.execute(values_table, commit=False)
        informations_table = 'CREATE TABLE IF NOT EXISTS information(key text primary '
        informations_table += 'key, value text)'
        self._storage.execute(informations_table)
        for owner in self._storage.select("SELECT DISTINCT owner FROM value", tuple(), False):
            try:
                getattr(owners, owner[0])
            except AttributeError:
                owners.addowner(owner[0])

    # sqlite
    def _sqlite_select(self, path):
        return self._storage.select("SELECT value FROM value WHERE path = ?",
                                   (path,))

    # value
    def setvalue(self, path, value, owner):
        """set value for an option
        a specified value must be associated to an owner
        """
        self.resetvalue(path)
        path = self._sqlite_encode_path(path)
        self._storage.execute("INSERT INTO value(path, value, owner) VALUES "
                             "(?, ?, ?)", (path, self._sqlite_encode(value),
                                           str(owner)))

    def getvalue(self, path):
        """get value for an option
        return: only value, not the owner
        """
        path = self._sqlite_encode_path(path)
        return self._sqlite_decode(self._sqlite_select(path)[0])

    def hasvalue(self, path):
        """if opt has a value
        return: boolean
        """
        path = self._sqlite_encode_path(path)
        return self._sqlite_select(path) is not None

    def resetvalue(self, path):
        """remove value means delete value in storage
        """
        path = self._sqlite_encode_path(path)
        self._storage.execute("DELETE FROM value WHERE path = ?", (path,))

    def get_modified_values(self):
        """return all values in a dictionary
        example: {option1: (owner, 'value1'), option2: (owner, 'value2')}
        """
        ret = {}
        for path, value, owner in self._storage.select("SELECT * FROM value",
                                                      only_one=False):
            path = self._sqlite_decode_path(path)
            owner = getattr(owners, owner)

            value = self._sqlite_decode(value)
            ret[path] = (owner, value)
        return ret

    # owner
    def setowner(self, path, owner):
        """change owner for an option
        """
        path = self._sqlite_encode_path(path)
        self._storage.execute("UPDATE value SET owner = ? WHERE path = ?",
                             (str(owner), path))

    def getowner(self, path, default):
        """get owner for an option
        return: owner object
        """
        path = self._sqlite_encode_path(path)
        owner = self._storage.select("SELECT owner FROM value WHERE path = ?",
                                    (path,))
        if owner is None:
            return default
        else:
            owner = owner[0]
            # autocreate owners
            try:
                return getattr(owners, owner)
            except AttributeError:
                owners.addowner(owner)
                return getattr(owners, owner)

    def set_information(self, key, value):
        """updates the information's attribute
        (which is a dictionary)

        :param key: information's key (ex: "help", "doc"
        :param value: information's value (ex: "the help string")
        """
        self._storage.execute("DELETE FROM information WHERE key = ?", (key,),
                             False)
        self._storage.execute("INSERT INTO information(key, value) VALUES "
                             "(?, ?)", (key, self._sqlite_encode(value)))

    def get_information(self, key):
        """retrieves one information's item

        :param key: the item string (ex: "help")
        """
        value = self._storage.select("SELECT value FROM information WHERE key = ?",
                                    (key,))
        if value is None:
            raise ValueError("not found")
        else:
            return self._sqlite_decode(value[0])
