from .autopath import do_autopath
do_autopath()

import warnings, sys
from py.test import raises

from tiramisu.config import Config
from tiramisu.option import DomainnameOption, EmailOption, URLOption, OptionDescription
from tiramisu.error import ValueWarning
from tiramisu.i18n import _


def test_domainname():
    d = DomainnameOption('d', '')
    f = DomainnameOption('f', '', allow_without_dot=True)
    g = DomainnameOption('g', '', allow_ip=True)
    od = OptionDescription('a', '', [d, f, g])
    c = Config(od)
    c.read_write()
    c.d = 'toto.com'
    raises(ValueError, "c.d = 'toto'")
    c.d = 'toto3.com'
    raises(ValueError, "c.d = 'toto_super.com'")
    c.d = 'toto-.com'
    raises(ValueError, "c.d = 'toto..com'")
    #
    c.f = 'toto.com'
    c.f = 'toto'
    c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamea'
    raises(ValueError, "c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamean'")
    c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamea.nd'
    c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamea.nditsnoteasytogeneratesolongdomainnamewithoutrepeatdomainnameto.olongthathavemorethanmaximumsizeforatruedomainnameanditsnoteas.ytogeneratesolongdomainnamewithoutrepeatbutimnotabletodoitnowie'
    raises(ValueError, "c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamea.nditsnoteasytogeneratesolongdomainnamewithoutrepeatdomainnameto.olongthathavemorethanmaximumsizeforatruedomainnameanditsnoteas.ytogeneratesolongdomainnamewithoutrepeatbutimnotabletodoitnowien'")
    c.f = 'd'
    c.f = 'd.t'
    #
    raises(ValueError, "c.f = '192.168.1.1'")
    c.g = 'toto.com'
    c.g = '192.168.1.0'
    c.g = '192.168.1.29'


def test_domainname_upper():
    d = DomainnameOption('d', '')
    od = OptionDescription('a', '', [d])
    c = Config(od)
    c.read_write()
    c.d = 'toto.com'
    msg = _('some characters are uppercase')
    has_error = False
    try:
        c.d = 'TOTO.COM'
    except ValueError as err:
        if sys.version_info[0] >= 3:  # pragma: optional cover
            assert msg in str(err)
        else:
            assert msg in str(err)
        has_error = True
    assert has_error is True
    has_error = False
    try:
        c.d = 'toTo.com'
    except ValueError as err:
        if sys.version_info[0] >= 3:  # pragma: optional cover
            assert msg in str(err)
        else:
            assert msg in str(err)
        has_error = True
    assert has_error is True


def test_domainname_warning():
    d = DomainnameOption('d', '', warnings_only=True)
    f = DomainnameOption('f', '', allow_without_dot=True, warnings_only=True)
    g = DomainnameOption('g', '', allow_ip=True, warnings_only=True)
    od = OptionDescription('a', '', [d, f, g])
    warnings.simplefilter("always", ValueWarning)
    c = Config(od)
    c.read_write()
    c.d = 'toto.com'
    raises(ValueError, "c.d = 'toto'")
    c.d = 'toto3.com'
    with warnings.catch_warnings(record=True) as w:
        c.d = 'toto_super.com'
    assert len(w) == 1
    c.d = 'toto-.com'
    raises(ValueError, "c.d = 'toto..com'")
    #
    c.f = 'toto.com'
    c.f = 'toto'
    c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamea'
    raises(ValueError, "c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamean'")
    c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamea.nd'
    c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainnamea.nditsnoteasytogeneratesolongdomainnamewithoutrepeatdomainnameto.olongthathavemorethanmaximumsizeforatruedomainnameanditsnoteas.ytogeneratesolongdomainnamewithoutrepeatbutimnotabletodoitnowie'
    raises(ValueError, "c.f = 'domainnametoolongthathavemorethanmaximumsizeforatruedomainname.nditsnoteasytogeneratesolongdomainnamewithoutrepeatdomainnamet.olongthathavemorethanmaximumsizeforatruedomainnameanditsnotea.ytogeneratesolongdomainnamewithoutrepeatbutimnotabletodoitnowie.xxxx'")
    c.f = 'd'
    c.f = 'd.t'
    #
    raises(ValueError, "c.f = '192.168.1.1'")
    c.g = 'toto.com'
    c.g = '192.168.1.0'
    c.g = '192.168.1.29'


def test_special_domain_name():
    """domain name option that starts with a number or not
    """
    d = DomainnameOption('d', '')
    e = DomainnameOption('e', '', type_='netbios')
    od = OptionDescription('a', '', [d, e])
    c = Config(od)
    c.read_write()
    c.d = '1toto.com'
    c.d = '123toto.com'
    c.e = 'toto'
    c.e = '1toto'


def test_domainname_netbios():
    d = DomainnameOption('d', '', type_='netbios')
    e = DomainnameOption('e', '', "toto", type_='netbios')
    od = OptionDescription('a', '', [d, e])
    c = Config(od)
    c.read_write()
    raises(ValueError, "c.d = 'toto.com'")
    c.d = 'toto'
    raises(ValueError, "c.d = 'domainnametoolong'")


def test_domainname_hostname():
    d = DomainnameOption('d', '', type_='hostname')
    e = DomainnameOption('e', '', "toto", type_='hostname')
    od = OptionDescription('a', '', [d, e])
    c = Config(od)
    c.read_write()
    raises(ValueError, "c.d = 'toto.com'")
    c.d = 'toto'
    c.d = 'domainnametoolong'


def test_email():
    e = EmailOption('e', '')
    od = OptionDescription('a', '', [e])
    c = Config(od)
    c.read_write()
    c.e = u'foo-bar.baz@example.com'
    c.e = u'root@foo.com'
    c.e = u'root@domain'
    raises(ValueError, "c.e = 1")
    raises(ValueError, "c.e = u'root'")
    raises(ValueError, "c.e = u'root[]@domain'")


def test_url():
    u = URLOption('u', '')
    od = OptionDescription('a', '', [u])
    c = Config(od)
    c.read_write()
    c.u = 'http://foo.com'
    c.u = 'https://foo.com'
    c.u = 'https://foo.com/'
    raises(ValueError, "c.u = 1")
    raises(ValueError, "c.u = 'ftp://foo.com'")
    raises(ValueError, "c.u = 'foo.com'")
    raises(ValueError, "c.u = ':/foo.com'")
    raises(ValueError, "c.u = 'foo.com/http://'")
    c.u = 'https://foo.com/index.html'
    c.u = 'https://foo.com/index.html?var=value&var2=val2'
    raises(ValueError, "c.u = 'https://foo.com/index\\n.html'")
    c.u = 'https://foo.com:8443'
    c.u = 'https://foo.com:8443/'
    c.u = 'https://foo.com:8443/index.html'
    raises(ValueError, "c.u = 'https://foo.com:84438989'")
    c.u = 'https://foo.com:8443/INDEX'
    raises(ValueError, "c.u = 'https://FOO.COM:8443'")
