from .autopath import do_autopath
do_autopath()

from py.test import raises

from tiramisu.setting import groups, owners
from tiramisu.config import Config, GroupConfig, MetaConfig
from tiramisu.option import IntOption, StrOption, NetworkOption, NetmaskOption, OptionDescription
from tiramisu.error import ConfigError, ConflictError

owners.addowner('meta')


def raise_exception():
    raise Exception('test')


def make_description():
    i1 = IntOption('i1', '')
    i2 = IntOption('i2', '', default=1)
    i3 = IntOption('i3', '')
    i4 = IntOption('i4', '', default=2)
    i5 = IntOption('i5', '', default=[2], multi=True)
    i6 = IntOption('i6', '', properties=('disabled',))
    od1 = OptionDescription('od1', '', [i1, i2, i3, i4, i5, i6])
    od2 = OptionDescription('od2', '', [od1])
    conf1 = Config(od2, name='conf1')
    conf2 = Config(od2, name='conf2')
    conf1.read_write()
    conf2.read_write()
    meta = MetaConfig([conf1, conf2], name='meta')
    meta.cfgimpl_get_settings().setowner(owners.meta)
    return meta


#FIXME ne pas mettre 2 meta dans une config
#FIXME ne pas mettre 2 OD differents dans un meta
#FIXME serialization
def test_none():
    meta = make_description()
    conf1, conf2 = meta.cfgimpl_get_children()
    assert conf1.od1.i3 is conf2.od1.i3 is None
    assert conf1.getowner(conf1.unwrap_from_path('od1.i3')) is conf2.getowner(conf2.unwrap_from_path('od1.i3')) is owners.default
    meta.od1.i3 = 3
    assert conf1.od1.i3 == conf2.od1.i3 == 3
    assert conf1.getowner(conf1.unwrap_from_path('od1.i3')) is conf2.getowner(conf2.unwrap_from_path('od1.i3')) is owners.meta
    meta.od1.i3 = 3
    conf1.od1.i3 = 2
    assert conf1.od1.i3 == 2
    assert conf2.od1.i3 == 3
    assert conf1.getowner(conf1.unwrap_from_path('od1.i3')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i3')) is owners.meta
    meta.od1.i3 = 4
    assert conf1.od1.i3 == 2
    assert conf2.od1.i3 == 4
    assert conf1.getowner(conf1.unwrap_from_path('od1.i3')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i3')) is owners.meta
    del(meta.od1.i3)
    assert conf1.od1.i3 == 2
    assert conf2.od1.i3 is None
    assert conf1.getowner(conf1.unwrap_from_path('od1.i3')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i3')) is owners.default
    del(conf1.od1.i3)
    assert conf1.od1.i3 is conf2.od1.i3 is None
    assert conf1.getowner(conf1.unwrap_from_path('od1.i3')) is conf2.getowner(conf2.unwrap_from_path('od1.i3')) is owners.default


def test_default():
    meta = make_description()
    conf1, conf2 = meta.cfgimpl_get_children()
    assert conf1.od1.i2 == conf2.od1.i2 == 1
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.default
    meta.od1.i2 = 3
    assert conf1.od1.i2 == conf2.od1.i2 == 3
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.meta
    meta.od1.i2 = 3
    conf1.od1.i2 = 2
    assert conf1.od1.i2 == 2
    assert conf2.od1.i2 == 3
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.meta
    meta.od1.i2 = 4
    assert conf1.od1.i2 == 2
    assert conf2.od1.i2 == 4
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.meta
    del(meta.od1.i2)
    assert conf1.od1.i2 == 2
    assert conf2.od1.i2 == 1
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.default
    del(conf1.od1.i2)
    assert conf1.od1.i2 == conf2.od1.i2 == 1
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.default


def test_contexts():
    meta = make_description()
    conf1, conf2 = meta.cfgimpl_get_children()
    assert conf1.od1.i2 == conf2.od1.i2 == 1
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.default
    meta.set_value('od1.i2', 6, only_config=True)
    assert meta.od1.i2 == 1
    assert conf1.od1.i2 == conf2.od1.i2 == 6
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.user


def test_find():
    meta = make_description()
    i2 = meta.unwrap_from_path('od1.i2')
    assert [i2] == meta.find(byname='i2')
    assert i2 == meta.find_first(byname='i2')
    assert meta.make_dict() == {'od1.i4': 2, 'od1.i1': None, 'od1.i3': None,
                                'od1.i2': 1, 'od1.i5': [2], 'od1.i6': None}


def test_group_error():
    raises(ValueError, "GroupConfig('str')")
    raises(ValueError, "GroupConfig(['str'])")


def test_meta_meta():
    meta1 = make_description()
    meta2 = MetaConfig([meta1])
    meta2.cfgimpl_get_settings().setowner(owners.meta)
    conf1, conf2 = meta1.cfgimpl_get_children()
    assert conf1.od1.i2 == conf2.od1.i2 == 1
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.default
    meta2.od1.i2 = 3
    assert conf1.od1.i2 == conf2.od1.i2 == 3
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.meta
    meta2.od1.i2 = 3
    conf1.od1.i2 = 2
    assert conf1.od1.i2 == 2
    assert conf2.od1.i2 == 3
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.meta
    meta2.od1.i2 = 4
    assert conf1.od1.i2 == 2
    assert conf2.od1.i2 == 4
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.meta
    del(meta2.od1.i2)
    assert conf1.od1.i2 == 2
    assert conf2.od1.i2 == 1
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is owners.user
    assert conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.default
    del(conf1.od1.i2)
    assert conf1.od1.i2 == conf2.od1.i2 == 1
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.default
    meta1.od1.i2 = 6
    assert conf1.od1.i2 == conf2.od1.i2 == 6
    assert conf1.getowner(conf1.unwrap_from_path('od1.i2')) is conf2.getowner(conf2.unwrap_from_path('od1.i2')) is owners.meta


def test_meta_meta_set():
    meta1 = make_description()
    meta2 = MetaConfig([meta1])
    meta2.cfgimpl_get_settings().setowner(owners.meta)
    conf1, conf2 = meta1.cfgimpl_get_children()
    meta2.set_value('od1.i1', 7, only_config=True)
    meta2.set_value('od1.i6', 7, only_config=True)
    assert conf1.od1.i1 == conf2.od1.i1 == 7
    assert conf1.getowner(conf1.unwrap_from_path('od1.i1')) is conf2.getowner(conf2.unwrap_from_path('od1.i1')) is owners.user
    assert [conf1, conf2] == meta2.find_firsts(byname='i1', byvalue=7).cfgimpl_get_children()
    conf1.od1.i1 = 8
    assert [conf1, conf2] == meta2.find_firsts(byname='i1').cfgimpl_get_children()
    assert [conf2] == meta2.find_firsts(byname='i1', byvalue=7).cfgimpl_get_children()
    assert [conf1] == meta2.find_firsts(byname='i1', byvalue=8).cfgimpl_get_children()
    assert [conf1, conf2] == meta2.find_firsts(byname='i5', byvalue=2).cfgimpl_get_children()
    raises(AttributeError, "meta2.find_firsts(byname='i1', byvalue=10)")
    raises(AttributeError, "meta2.find_firsts(byname='not', byvalue=10)")
    raises(AttributeError, "meta2.find_firsts(byname='i6')")
    raises(ValueError, "meta2.set_value('od1.i6', 7, only_config=True, force_default=True)")
    raises(ValueError, "meta2.set_value('od1.i6', 7, only_config=True, force_default_if_same=True)")
    raises(ValueError, "meta2.set_value('od1.i6', 7, only_config=True, force_dont_change_value=True)")


def test_not_meta():
    i1 = IntOption('i1', '')
    od1 = OptionDescription('od1', '', [i1])
    od2 = OptionDescription('od2', '', [od1])
    conf1 = Config(od2, name='conf1')
    conf2 = Config(od2, name='conf2')
    conf3 = Config(od2)
    conf4 = Config(od2, name='conf2')
    conf3, conf4
    raises(ValueError, "GroupConfig(conf1)")
    #same name
    raises(ConflictError, "GroupConfig([conf2, conf4])")
    grp = GroupConfig([conf1, conf2])
    raises(ConfigError, 'grp.od1.i1')
    conf1, conf2 = grp.cfgimpl_get_children()
    grp.set_value('od1.i1', 7)
    assert grp.conf1.od1.i1 == conf2.od1.i1 == 7
    assert grp.conf1.getowner(grp.conf1.unwrap_from_path('od1.i1')) is grp.conf2.getowner(grp.conf2.unwrap_from_path('od1.i1')) is owners.user


def test_group_find_firsts():
    i1 = IntOption('i1', '')
    od1 = OptionDescription('od1', '', [i1])
    od2 = OptionDescription('od2', '', [od1])
    conf1 = Config(od2, name='conf1')
    conf2 = Config(od2, name='conf2')
    grp = GroupConfig([conf1, conf2])
    assert [conf1, conf2] == grp.find_firsts(byname='i1').cfgimpl_get_children()


def test_group_group():
    i1 = IntOption('i1', '')
    od1 = OptionDescription('od1', '', [i1])
    od2 = OptionDescription('od2', '', [od1])
    conf1 = Config(od2, name='conf1')
    conf2 = Config(od2, name='conf2')
    grp = GroupConfig([conf1, conf2])
    raises(ValueError, "GroupConfig([grp])")
    grp = GroupConfig([conf1, conf2], 'grp')
    grp2 = GroupConfig([grp])
    grp2.set_value('od1.i1', 2)
    assert grp2.grp.conf1.od1.i1 == 2
    assert grp2.grp.conf1.getowner(i1) == owners.user


def test_meta_path():
    meta = make_description()
    assert meta._impl_path is None
    assert meta.od1._impl_path == 'od1'


def test_meta_unconsistent():
    i1 = IntOption('i1', '')
    i2 = IntOption('i2', '', default=1)
    i3 = IntOption('i3', '')
    i4 = IntOption('i4', '', default=2)
    od1 = OptionDescription('od1', '', [i1, i2, i3, i4])
    od2 = OptionDescription('od2', '', [od1])
    od3 = OptionDescription('od3', '', [od1])
    conf1 = Config(od2, name='conf1')
    conf2 = Config(od2, name='conf2')
    conf3 = Config(od2, name='conf3')
    conf4 = Config(od3, name='conf4')
    conf3, conf4
    meta = MetaConfig([conf1, conf2])
    meta.cfgimpl_get_settings().setowner(owners.meta)
    raises(TypeError, 'MetaConfig("string")')
    #same descr but conf1 already in meta
    raises(ValueError, "MetaConfig([conf1, conf3])")
    #not same descr
    raises(ValueError, "MetaConfig([conf3, conf4])")


def test_meta_master_slaves():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf2 = Config(interface1, name='conf2')
    meta = MetaConfig([conf1, conf2])
    meta.conf1.read_only()
    meta.conf2.read_only()
    assert [conf1, conf2] == meta.find_firsts(byname='netmask_admin_eth0').cfgimpl_get_children()
    meta.conf1.read_write()
    meta.conf2.read_only()
    assert [conf2] == meta.find_firsts(byname='netmask_admin_eth0').cfgimpl_get_children()
    meta.conf2.read_write()
    raises(AttributeError, "meta.find_firsts(byname='netmask_admin_eth0')")
    assert [conf1, conf2] == meta.find_firsts(byname='netmask_admin_eth0',
                                              check_properties=None).cfgimpl_get_children()
    meta.conf1.read_only()
    meta.conf2.read_only()
    meta.read_write()
    assert [conf1, conf2] == meta.find_firsts(byname='netmask_admin_eth0').cfgimpl_get_children()


def test_meta_master_slaves_value():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf2 = Config(interface1, name='conf2')
    meta = MetaConfig([conf1, conf2])
    meta.conf1.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.netmask_admin_eth0 == [None]
    del(meta.conf1.ip_admin_eth0)
    assert meta.conf1.netmask_admin_eth0 == []
    meta.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.netmask_admin_eth0 == [None]
    meta.netmask_admin_eth0 = ['255.255.255.0']
    assert meta.conf1.netmask_admin_eth0 == ['255.255.255.0']
    meta.netmask_admin_eth0 = ['255.255.0.0']
    assert meta.conf1.netmask_admin_eth0 == ['255.255.0.0']
    meta.conf1.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.netmask_admin_eth0 == [None]


def test_meta_master_slaves_value_default():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True, default=['192.168.1.1'])
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf2 = Config(interface1, name='conf2')
    meta = MetaConfig([conf1, conf2])
    assert meta.conf1.netmask_admin_eth0 == [None]
    meta.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.netmask_admin_eth0 == [None]
    meta.netmask_admin_eth0 = ['255.255.255.0']
    assert meta.conf1.netmask_admin_eth0 == ['255.255.255.0']
    meta.netmask_admin_eth0 = ['255.255.0.0']
    assert meta.conf1.netmask_admin_eth0 == ['255.255.0.0']
    meta.conf1.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.netmask_admin_eth0 == [None]


def test_meta_master_slaves_owners():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf2 = Config(interface1, name='conf2')
    meta = MetaConfig([conf1, conf2])
    meta.cfgimpl_get_settings().setowner(owners.meta)
    assert meta.conf1.getowner(ip_admin_eth0) == owners.default
    assert meta.conf1.getowner(netmask_admin_eth0) == owners.default
    meta.conf1.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.getowner(ip_admin_eth0) == owners.user
    assert meta.conf1.getowner(netmask_admin_eth0) == owners.default
    del(meta.conf1.ip_admin_eth0)
    assert meta.conf1.getowner(ip_admin_eth0) == owners.default
    assert meta.conf1.getowner(netmask_admin_eth0) == owners.default
    meta.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.getowner(ip_admin_eth0) == owners.meta
    assert meta.conf1.getowner(netmask_admin_eth0) == owners.default
    meta.netmask_admin_eth0 = ['255.255.255.0']
    assert meta.conf1.getowner(ip_admin_eth0) == owners.meta
    assert meta.conf1.getowner(netmask_admin_eth0, 0) == owners.meta
    meta.netmask_admin_eth0 = ['255.255.0.0']
    assert meta.conf1.getowner(ip_admin_eth0) == owners.meta
    assert meta.conf1.getowner(netmask_admin_eth0, 0) == owners.meta
    meta.conf1.ip_admin_eth0 = ['192.168.1.1']
    assert meta.conf1.getowner(ip_admin_eth0) == owners.user
    assert meta.conf1.getowner(netmask_admin_eth0, 0) == owners.default


def test_meta_force_default():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf1.read_write()
    conf2 = Config(interface1, name='conf2')
    conf2.read_write()
    meta = MetaConfig([conf1, conf2])
    meta.read_write()
    meta.cfgimpl_get_settings().setowner(owners.meta)
    assert meta.ip_admin_eth0 == []
    assert meta.conf1.ip_admin_eth0 == []
    assert meta.conf2.ip_admin_eth0 == []
    meta.set_value('ip_admin_eth0', ['192.168.1.1'])
    assert meta.ip_admin_eth0 == ['192.168.1.1']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.1']
    assert meta.conf2.ip_admin_eth0 == ['192.168.1.1']
    meta.conf1.ip_admin_eth0 = ['192.168.1.2']
    assert meta.ip_admin_eth0 == ['192.168.1.1']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.2']
    assert meta.conf2.ip_admin_eth0 == ['192.168.1.1']
    meta.set_value('ip_admin_eth0', ['192.168.1.3'])
    assert meta.ip_admin_eth0 == ['192.168.1.3']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.2']
    assert meta.conf2.ip_admin_eth0 == ['192.168.1.3']
    meta.set_value('ip_admin_eth0', ['192.168.1.4'], force_default=True)
    assert meta.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf2.ip_admin_eth0 == ['192.168.1.4']


def test_meta_force_dont_change_value():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf1.read_write()
    conf2 = Config(interface1, name='conf2')
    conf2.read_write()
    meta = MetaConfig([conf1, conf2])
    meta.read_write()
    meta.cfgimpl_get_settings().setowner(owners.meta)
    assert meta.ip_admin_eth0 == []
    assert meta.conf1.ip_admin_eth0 == []
    assert meta.conf2.ip_admin_eth0 == []
    meta.conf1.ip_admin_eth0 = ['192.168.1.4']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf2.ip_admin_eth0 == []
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.default
    meta.set_value('ip_admin_eth0', ['192.168.1.4'], force_dont_change_value=True)
    assert meta.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf2.ip_admin_eth0 == []
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.user


def test_meta_force_default_if_same():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf1.read_write()
    conf2 = Config(interface1, name='conf2')
    conf2.read_write()
    meta = MetaConfig([conf1, conf2])
    meta.read_write()
    meta.cfgimpl_get_settings().setowner(owners.meta)
    #
    assert meta.ip_admin_eth0 == []
    assert meta.conf1.ip_admin_eth0 == []
    assert meta.conf2.ip_admin_eth0 == []
    #
    meta.conf1.ip_admin_eth0 = ['192.168.1.4']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf2.ip_admin_eth0 == []
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.default
    meta.set_value('ip_admin_eth0', ['192.168.1.4'], force_default_if_same=True)
    assert meta.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf2.ip_admin_eth0 == ['192.168.1.4']
    assert conf1.getowner(ip_admin_eth0) is owners.meta
    assert conf2.getowner(ip_admin_eth0) is owners.meta
    #
    meta.conf1.ip_admin_eth0 = ['192.168.1.3']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.3']
    assert meta.conf2.ip_admin_eth0 == ['192.168.1.4']
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.meta
    meta.set_value('ip_admin_eth0', ['192.168.1.5'], force_default_if_same=True)
    assert meta.ip_admin_eth0 == ['192.168.1.5']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.3']
    assert meta.conf2.ip_admin_eth0 == ['192.168.1.5']
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.meta


def test_meta_force_default_if_same_and_dont_change():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf1.read_write()
    conf2 = Config(interface1, name='conf2')
    conf2.read_write()
    meta = MetaConfig([conf1, conf2])
    meta.read_write()
    meta.cfgimpl_get_settings().setowner(owners.meta)
    #
    assert meta.ip_admin_eth0 == []
    assert meta.conf1.ip_admin_eth0 == []
    assert meta.conf2.ip_admin_eth0 == []
    #
    meta.conf1.ip_admin_eth0 = ['192.168.1.4']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf2.ip_admin_eth0 == []
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.default
    meta.set_value('ip_admin_eth0', ['192.168.1.4'], force_default_if_same=True, force_dont_change_value=True)
    assert meta.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.4']
    assert meta.conf2.ip_admin_eth0 == []
    assert conf1.getowner(ip_admin_eth0) is owners.meta
    assert conf2.getowner(ip_admin_eth0) is owners.user
    #
    meta.conf1.ip_admin_eth0 = ['192.168.1.3']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.3']
    assert meta.conf2.ip_admin_eth0 == []
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.user
    meta.set_value('ip_admin_eth0', ['192.168.1.5'], force_default_if_same=True, force_dont_change_value=True)
    assert meta.ip_admin_eth0 == ['192.168.1.5']
    assert meta.conf1.ip_admin_eth0 == ['192.168.1.3']
    assert meta.conf2.ip_admin_eth0 == []
    assert conf1.getowner(ip_admin_eth0) is owners.user
    assert conf2.getowner(ip_admin_eth0) is owners.user


def test_meta_force_default_and_dont_change():
    ip_admin_eth0 = StrOption('ip_admin_eth0', "ip", multi=True)
    netmask_admin_eth0 = StrOption('netmask_admin_eth0', "mask", multi=True, properties=('hidden',))
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf1.read_write()
    conf2 = Config(interface1, name='conf2')
    conf2.read_write()
    meta = MetaConfig([conf1, conf2])
    meta.read_write()
    meta.cfgimpl_get_settings().setowner(owners.meta)
    raises(ValueError, "meta.set_value('ip_admin_eth0', ['192.168.1.4'], force_default=True, force_dont_change_value=True)")


def test_meta_properties_meta():
    ip_admin_eth0 = NetworkOption('ip_admin_eth0', "ip", multi=True, default=['192.168.1.1'])
    netmask_admin_eth0 = NetmaskOption('netmask_admin_eth0', "mask", multi=True, properties=('disabled',))
    netmask_admin_eth0.impl_add_consistency('network_netmask', ip_admin_eth0)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf1.read_write()
    conf2 = Config(interface1, name='conf2')
    conf2.read_write()
    meta = MetaConfig([conf1, conf2])
    meta.read_write()
    assert conf1.make_dict() == {}


def test_meta_exception_meta():
    ip_admin_eth0 = NetworkOption('ip_admin_eth0', "ip", multi=True, default=['192.168.1.1'])
    netmask_admin_eth0 = NetmaskOption('netmask_admin_eth0', "mask", multi=True, callback=raise_exception)
    netmask_admin_eth0.impl_add_consistency('network_netmask', ip_admin_eth0)
    interface1 = OptionDescription('ip_admin_eth0', '', [ip_admin_eth0, netmask_admin_eth0])
    interface1.impl_set_group_type(groups.master)
    conf1 = Config(interface1, name='conf1')
    conf1.read_write()
    conf2 = Config(interface1, name='conf2')
    conf2.read_write()
    meta = MetaConfig([conf1, conf2])
    meta.read_write()
    raises(Exception, "conf1.make_dict()")
