from .autopath import do_autopath
do_autopath()

from py.test import raises

from tiramisu.setting import owners, groups
from tiramisu.config import Config
from tiramisu.option import ChoiceOption, BoolOption, IntOption, FloatOption, \
    StrOption, OptionDescription, SymLinkOption
from tiramisu.error import ConfigError, ConstError, PropertiesOptionError


def make_description():
    gcoption = ChoiceOption('name', 'GC name', ['ref', 'framework'], 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ['std', 'thunk'], 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=True)
    wantref_option = BoolOption('wantref', 'Test requires', default=False)
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False)

    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption])
    descr = OptionDescription('tiram', '', [gcgroup, booloption, objspaceoption,
                                            wantref_option, stroption,
                                            wantframework_option,
                                            intoption, boolop])
    return descr


def test_default_owner():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    assert cfg.dummy is False
    assert cfg.getowner(gcdummy) == 'default'
    raises(TypeError, "cfg.getowner('gcdummy')")
    cfg.dummy = True
    assert cfg.getowner(gcdummy) == owners.user


def test_hidden_owner():
    gcdummy = BoolOption('dummy', 'dummy', default=False, properties=('hidden',))
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    cfg.read_write()
    raises(PropertiesOptionError, "cfg.getowner(gcdummy)")
    raises(PropertiesOptionError, "cfg.getowner(gcdummy, force_permissive=True)")
    raises(PropertiesOptionError, "cfg.cfgimpl_get_values().is_default_owner(gcdummy)")
    raises(PropertiesOptionError, "cfg.cfgimpl_get_values().is_default_owner(gcdummy, force_permissive=True)")
    cfg.cfgimpl_get_settings().setpermissive(('hidden',))
    cfg.getowner(gcdummy, force_permissive=True)
    cfg.cfgimpl_get_values().is_default_owner(gcdummy, force_permissive=True)


def test_addowner():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    assert cfg.dummy is False
    assert cfg.getowner(gcdummy) == 'default'
    assert cfg.cfgimpl_get_values().is_default_owner(gcdummy)
    owners.addowner("gen_config")
    cfg.cfgimpl_get_settings().setowner(owners.gen_config)
    cfg.dummy = True
    assert cfg.getowner(gcdummy) == owners.gen_config
    assert not cfg.cfgimpl_get_values().is_default_owner(gcdummy)


def test_addowner_multiple_time():
    owners.addowner("testowner")
    raises(ConstError, 'owners.addowner("testowner")')


def test_delete_owner():
    owners.addowner('deleted')
    raises(ConstError, 'del(owners.deleted)')
    raises(ValueError, 'del(owners.deleted2)')


def test_owner_is_not_a_string():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    assert cfg.dummy is False
    assert cfg.getowner(gcdummy) == owners.default
    assert cfg.getowner(gcdummy) == 'default'
    assert isinstance(cfg.getowner(gcdummy), owners.Owner)
    cfg.dummy = True
    assert cfg.getowner(gcdummy) == 'user'


def test_setowner_without_valid_owner():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    assert cfg.dummy is False
    assert cfg.getowner(gcdummy) == 'default'
    raises(TypeError, "cfg.cfgimpl_get_settings().setowner('gen_config')")


def test_setowner_for_value():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    assert cfg.dummy is False
    assert cfg.getowner(gcdummy) == 'default'
    owners.addowner("new")
    raises(ConfigError, "cfg.cfgimpl_get_values().setowner(gcdummy, owners.new)")
    cfg.dummy = False
    assert cfg.getowner(gcdummy) == owners.user
    cfg.cfgimpl_get_values().setowner(gcdummy, owners.new)
    assert cfg.getowner(gcdummy) == owners.new
    raises(TypeError, "cfg.cfgimpl_get_values().setowner(gcdummy, 'new')")


def test_setowner_read_only():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr = OptionDescription('tiramisu', '', [gcdummy])
    cfg = Config(descr)
    cfg.read_write()
    assert cfg.dummy is False
    assert cfg.getowner(gcdummy) == 'default'
    owners.addowner("readonly")
    cfg.dummy = False
    assert cfg.getowner(gcdummy) == owners.user
    cfg.read_only()
    raises(PropertiesOptionError,
           "cfg.cfgimpl_get_values().setowner(gcdummy, owners.readonly)")
    assert cfg.getowner(gcdummy) == owners.user


def test_setowner_optiondescription():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    descr1 = OptionDescription('tiramisu', '', [gcdummy])
    descr = OptionDescription('tiramisu', '', [descr1])
    cfg = Config(descr)
    cfg
    raises(ConfigError, 'cfg.cfgimpl_get_values().getowner(descr1)')
    raises(ConfigError, 'cfg.cfgimpl_get_values().setowner(descr1, owners.user)')


def test_setowner_symlinkoption():
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    s = SymLinkOption('symdummy', gcdummy)
    descr1 = OptionDescription('tiramisu', '', [gcdummy, s])
    descr = OptionDescription('tiramisu', '', [descr1])
    cfg = Config(descr)
    assert cfg.cfgimpl_get_values().getowner(s) == owners.default
    cfg.tiramisu.dummy = True
    assert cfg.cfgimpl_get_values().getowner(s) == owners.user
    raises(ConfigError, 'cfg.cfgimpl_get_values().setowner(s, owners.user)')


def test_owner_masterslaves():
    b = IntOption('int', 'Test int option', default=[0], multi=True)
    c = StrOption('str', 'Test string option', multi=True)
    descr = OptionDescription("int", "", [b, c])
    descr.impl_set_group_type(groups.master)
    cfg = Config(descr)
    raises(ConfigError, 'cfg.cfgimpl_get_values().setowner(c, owners.user)')

    cfg.int.append(1)
    cfg.str[0] = 'yes'
    assert cfg.cfgimpl_get_values().getowner(c, 0) == owners.user
    assert cfg.cfgimpl_get_values().getowner(c, 1) == owners.default
    cfg.cfgimpl_get_values().setowner(c, owners.user, 0)
    assert cfg.cfgimpl_get_values().getowner(c, 0) == owners.user
    assert cfg.cfgimpl_get_values().getowner(c, 1) == owners.default
    assert cfg.str == ['yes', None]
