"config.set() or config.setoption() or option.setoption()"
from .autopath import do_autopath
do_autopath()

from py.test import raises

from tiramisu.i18n import _
from tiramisu.error import display_list
from tiramisu.setting import owners, groups
from tiramisu.config import Config
from tiramisu.option import ChoiceOption, BoolOption, IntOption, FloatOption, \
    StrOption, OptionDescription
from tiramisu.error import PropertiesOptionError


def make_description():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=True)
    wantref_option = BoolOption('wantref', 'Test requires', default=False)
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False)
    gcgroup = OptionDescription('gc', '', [gcoption, gcdummy, floatoption])
    descr = OptionDescription('tiramisu', '', [gcgroup, booloption, objspaceoption,
                                               wantref_option, stroption,
                                               wantframework_option,
                                               intoption, boolop])
    return descr


#____________________________________________________________
# change with __setattr__
def test_attribute_access():
    "Once set, option values can't be changed again by attribute access"
    s = StrOption("string", "", default="string")
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    # let's try to change it again
    config.string = "foo"
    assert config.string == "foo"


def test_setitem():
    s = StrOption("string", "", default=["string", "sdfsdf"], default_multi="prout", multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    config.string[1] = "titi"


def test_reset():
    "if value is None, resets to default owner"
    s = StrOption("string", "", default="string")
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    config.string = "foo"
    assert config.string == "foo"
    assert config.getowner(s) == owners.user
    del(config.string)
    assert config.string == 'string'
    assert config.getowner(s) == owners.default


def test_reset_with_multi():
    s = StrOption("string", "", default=["string"], default_multi="string", multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
#    config.string = []
    del(config.string)
    assert config.string == ["string"]
    assert config.getowner(s) == 'default'
    config.string = ["eggs", "spam", "foo"]
    assert config.getowner(s) == 'user'
    config.string = []
    del(config.string)
#    assert config.string == ["string"]
    assert config.getowner(s) == 'default'
    raises(ValueError, "config.string = None")


def test_default_with_multi():
    "default with multi is a list"
    s = StrOption("string", "", default=[], default_multi="string", multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    assert config.string == []
    s = StrOption("string", "", default=None, default_multi="string", multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    assert config.string == []


def test_idontexist():
    descr = make_description()
    cfg = Config(descr)
    cfg
    raises(AttributeError, "cfg.idontexist")


# ____________________________________________________________
def test_attribute_access_with_multi():
    s = StrOption("string", "", default=["string"], default_multi="string", multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    config.string = ["foo", "bar"]
    assert config.string == ["foo", "bar"]


def test_item_access_with_multi():
    s = StrOption("string", "", default=["string"], multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    config.string = ["foo", "bar"]
    assert config.string == ["foo", "bar"]
    assert config.string[0] == "foo"
    config.string[0] = 'changetest'
    assert config.string[0] == 'changetest'


def test_access_with_multi_default():
    s = StrOption("string", "", default=["string"], multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    assert config.getowner(s) == 'default'
    config.string = ["foo", "bar"]
    assert config.string == ["foo", "bar"]
    assert config.getowner(s) == 'user'


def test_multi_with_requires():
    s = StrOption("string", "", default=["string"], default_multi="string", multi=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default=["abc"], default_multi="abc",
                          requires=[{'option': intoption, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr = OptionDescription("options", "", [s, intoption, stroption])
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    assert not 'hidden' in setting[stroption]
    config.int = 1
    raises(PropertiesOptionError, "config.str = ['a', 'b']")
    assert 'hidden' in setting[stroption]


def test__requires_with_inverted():
    s = StrOption("string", "", default=["string"], multi=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default=["abc"], default_multi="abc",
                          requires=[{'option': intoption, 'expected': 1, 'action': 'hide', 'inverse': True}], multi=True)
    descr = OptionDescription("options", "", [s, intoption, stroption])
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    assert not 'hidden' in setting[stroption]
    config.int = 1
    assert not 'hidden' in setting[stroption]


def test_multi_with_requires_in_another_group():
    s = StrOption("string", "", default=["string"], multi=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default=["abc"],
                          requires=[{'option': intoption, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr = OptionDescription("opt", "", [stroption])
    descr2 = OptionDescription("opt2", "", [intoption, s, descr])
    config = Config(descr2)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    assert not 'hidden' in setting[stroption]
    config.int = 1
    raises(PropertiesOptionError,  "config.opt.str = ['a', 'b']")
    assert 'hidden' in setting[stroption]


def test_multi_with_requires_in_another_group_inverse():
    s = StrOption("string", "", default=["string"], multi=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default=["abc"],
            requires=[{'option': intoption, 'expected': 0, 'action': 'hidden', 'inverse': True}], multi=True)
    descr = OptionDescription("opt", "", [stroption])
    descr2 = OptionDescription("opt2", "", [intoption, s, descr])
    config = Config(descr2)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    assert not 'hidden' in setting[stroption]
    config.int = 1
    raises(PropertiesOptionError,  "config.opt.str = ['a', 'b']")
    assert 'hidden' in setting[stroption]


def test_apply_requires_from_config():
    s = StrOption("string", "", default=["string"], multi=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default=["abc"],
                          requires=[{'option': intoption, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr = OptionDescription("opt", "", [stroption])
    descr2 = OptionDescription("opt2", "", [intoption, s, descr])
    config = Config(descr2)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    assert not 'hidden' in setting[stroption]
    config.int = 1
    raises(PropertiesOptionError, 'config.opt.str')
    assert 'hidden' in setting[stroption]


def test_apply_requires_with_disabled():
    s = StrOption("string", "", default=["string"], multi=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default=["abc"],
                          requires=[{'option': intoption, 'expected': 1, 'action': 'disabled'}], multi=True)
    descr = OptionDescription("opt", "", [stroption])
    descr2 = OptionDescription("opt2", "", [intoption, s, descr])
    config = Config(descr2)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    assert not 'disabled' in setting[stroption]
    config.int = 1
    raises(PropertiesOptionError, 'config.opt.str')
    assert 'disabled' in setting[stroption]


def test_multi_with_requires_with_disabled_in_another_group():
    s = StrOption("string", "", default=["string"], multi=True)
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default=["abc"],
                          requires=[{'option': intoption, 'expected': 1, 'action': 'disabled'}], multi=True)
    descr = OptionDescription("opt", "", [stroption])
    descr2 = OptionDescription("opt2", "", [intoption, s, descr])
    config = Config(descr2)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    assert not 'disabled' in setting[stroption]
    config.int = 1
    raises(PropertiesOptionError,  "config.opt.str = ['a', 'b']")
    assert 'disabled' in setting[stroption]


def test_multi_with_requires_that_is_multi():
    b = IntOption('int', 'Test int option', default=[0], multi=True)
    c = StrOption('str', 'Test string option', default=['abc'], requires=[{'option': b, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr = OptionDescription("opt", "", [b, c])
    descr
    raises(ValueError, "Config(descr)")


def test_multi_with_requires_that_is_multi_inverse():
    b = IntOption('int', 'Test int option', default=[0], multi=True)
    c = StrOption('str', 'Test string option', default=['abc'], requires=[{'option': b, 'expected': 0, 'action': 'hidden', 'inverse': True}], multi=True)
    descr = OptionDescription("opt", "", [b, c])
    descr
    raises(ValueError, "Config(descr)")


def test_multi_with_requires_that_is_masterslave():
    b = IntOption('int', 'Test int option', default=[0], multi=True)
    c = StrOption('str', 'Test string option', requires=[{'option': b, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr = OptionDescription("int", "", [b, c])
    descr.impl_set_group_type(groups.master)
    Config(descr)


def test_multi_with_requires_that_is_masterslave_master():
    b = IntOption('int', 'Test int option', multi=True)
    c = StrOption('str', 'Test string option', requires=[{'option': b, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr = OptionDescription("str", "", [c, b])
    descr.impl_set_group_type(groups.master)
    raises(ValueError, "Config(descr)")


def test_multi_with_requires_that_is_masterslave_slave():
    b = IntOption('int', 'Test int option', default=[0], multi=True)
    c = StrOption('str', 'Test string option', multi=True)
    d = StrOption('str1', 'Test string option', requires=[{'option': c, 'expected': '1', 'action': 'hidden'}], multi=True)
    descr = OptionDescription("int", "", [b, c, d])
    descr.impl_set_group_type(groups.master)
    config = Config(descr)
    config.read_write()
    assert config.int == [0]
    assert config.str == [None]
    assert config.str1 == [None]
    config.int = [0, 1]
    assert config.int == [0, 1]
    assert config.str == [None, None]
    assert config.str1 == [None, None]
    config.str = [None, '1']
    config.read_only()
    assert config.str1 == [None, None]
    config.read_write()
    assert config.str1[0] is None
    raises(PropertiesOptionError, 'config.str1[1]')


def test_multi_with_requires_that_is_masterslave_slave_inverse():
    b = IntOption('int', 'Test int option', default=[0], multi=True)
    c = StrOption('str', 'Test string option', multi=True)
    d = StrOption('str1', 'Test string option', requires=[{'option': c, 'expected': None, 'action': 'hidden', 'inverse': True}], multi=True)
    descr = OptionDescription("int", "", [b, c, d])
    descr.impl_set_group_type(groups.master)
    config = Config(descr)
    config.read_write()
    assert config.int == [0]
    assert config.str == [None]
    assert config.str1 == [None]
    config.int = [0, 1]
    assert config.int == [0, 1]
    assert config.str == [None, None]
    assert config.str1 == [None, None]
    config.str = [None, '1']
    config.read_only()
    assert config.str1 == [None, None]
    config.read_write()
    assert config.str1[0] is None
    raises(PropertiesOptionError, 'config.str1[1]')


def test_multi_with_requires_that_is_not_same_masterslave():
    b = IntOption('int', 'Test int option', default=[0], multi=True)
    c = StrOption('str', 'Test string option', requires=[{'option': b, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr1 = OptionDescription("int", "", [b, c])
    descr1.impl_set_group_type(groups.master)
    d = IntOption('int1', 'Test int option', default=[0], multi=True)
    e = StrOption('str', 'Test string option', requires=[{'option': b, 'expected': 1, 'action': 'hidden'}], multi=True)
    descr2 = OptionDescription("int1", "", [d, e])
    descr2.impl_set_group_type(groups.master)
    descr3 = OptionDescription('val', '', [descr1, descr2])
    descr3
    raises(ValueError, "Config(descr3)")


def test_multi_with_bool():
    s = BoolOption("bool", "", default=[False], multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    assert descr.bool.impl_is_multi() is True
    config.bool = [True, False]
    assert config.cfgimpl_get_values()[s] == [True, False]
    assert config.bool == [True, False]


def test_multi_with_bool_two():
    s = BoolOption("bool", "", default=[False], multi=True)
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    config
    assert descr.bool.impl_is_multi() is True
    raises(ValueError, "config.bool = True")


def test_choice_access_with_multi():
    ch = ChoiceOption("t1", "", ("a", "b"), default=["a"], multi=True)
    descr = OptionDescription("options", "", [ch])
    config = Config(descr)
    config.t1 = ["a", "b", "a", "b"]
    assert config.t1 == ["a", "b", "a", "b"]


#____________________________________________________________
def test_accepts_multiple_changes_from_option():
    s = StrOption("string", "", default="string")
    descr = OptionDescription("options", "", [s])
    config = Config(descr)
    config.string = "egg"
    assert s.impl_getdefault() == "string"
    assert config.string == "egg"
    config.string = 'blah'
    assert s.impl_getdefault() == "string"
    assert config.string == "blah"
    config.string = 'bol'
    assert config.string == 'bol'


def test_allow_multiple_changes_from_config():
    """
    a `setoption` from the config object is much like the attribute access,
    except the fact that value owner can bet set
    """
    s = StrOption("string", "", default="string")
    s2 = StrOption("string2", "", default="string")
    suboption = OptionDescription("bip", "", [s2])
    descr = OptionDescription("options", "", [s, suboption])
    config = Config(descr)
    config.string = "oh"
    assert config.string == "oh"
    config.string = "blah"
    assert config.string == "blah"


# ____________________________________________________________
# accessing a value by the get method
def test_access_by_get():
    descr = make_description()
    cfg = Config(descr)
    raises(AttributeError, "cfg.find(byname='idontexist')")
    assert cfg.find_first(byname='wantref', type_='value') is False
    assert cfg.gc.dummy is False
    assert cfg.find_first(byname='dummy', type_='value') is False


def test_access_by_get_whith_hide():
    b1 = BoolOption("b1", "", properties=(('hidden'),))
    descr = OptionDescription("opt", "",
                              [OptionDescription("sub", "",
                                                 [b1, ChoiceOption("c1", "", ('a', 'b', 'c'), 'a'),
                                                  BoolOption("d1", "")]),
                               BoolOption("b2", ""),
                               BoolOption("d1", "")])
    c = Config(descr)
    c.read_write()
    raises(AttributeError, "c.find(byname='b1')")


def test_extend_config_properties():
    descr = make_description()
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    assert set(eval(str(setting))) == set(['cache', 'expire', 'validator', 'warnings'])
    setting.extend(['test', 'test2'])
    assert set(eval(str(setting))) == set(['test', 'cache', 'test2', 'expire', 'validator', 'warnings'])


def test_append_properties():
    descr = make_description()
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    option = cfg.cfgimpl_get_description().gc.dummy
    assert tuple(option._properties) == tuple()
    assert not 'test' in setting[option]
    setting[option].append('test')
    assert tuple(option._properties) == tuple()
    assert 'test' in setting[option]


def test_reset_properties():
    descr = make_description()
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    option = cfg.cfgimpl_get_description().gc.dummy
    assert setting._p_.get_modified_properties() == {}
    setting.append('frozen')
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'cache', 'validator', 'warnings'))}
    setting.reset()
    assert setting._p_.get_modified_properties() == {}
    setting[option].append('test')
    assert setting._p_.get_modified_properties() == {'gc.dummy': set(('test',))}
    setting.reset()
    assert setting._p_.get_modified_properties() == {'gc.dummy': set(('test',))}
    setting.append('frozen')
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'validator', 'cache', 'warnings')), 'gc.dummy': set(('test',))}
    setting.reset(option)
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'validator', 'cache', 'warnings'))}
    setting[option].append('test')
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'validator', 'cache', 'warnings')), 'gc.dummy': set(('test',))}
    setting.reset(all_properties=True)
    assert setting._p_.get_modified_properties() == {}
    raises(ValueError, 'setting.reset(all_properties=True, opt=option)')
    a = descr.wantref
    setting[a].append('test')
    setting[a].reset()


def test_reset_multiple():
    descr = make_description()
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    option = cfg.cfgimpl_get_description().gc.dummy
    setting.append('frozen')
    setting[option].append('test')
    setting.reset()
    setting.reset()
    setting.append('frozen')
    setting[option].append('test')
    setting.reset(option)
    setting.reset(option)
    setting.append('frozen')
    setting[option].append('test')
    setting.reset(all_properties=True)
    setting.reset(all_properties=True)


def test_properties_cached():
    b1 = BoolOption("b1", "", properties=('test',))
    descr = OptionDescription("opt", "", [OptionDescription("sub", "", [b1])])
    c = Config(descr)
    c.read_write()
    setting = c.cfgimpl_get_settings()
    option = c.cfgimpl_get_description().sub.b1
    option
    c._setattr('sub.b1', True, force_permissive=True)
    assert str(setting[b1]) in ["['test']", "[u'test']"]



def test_append_properties_force_store_value():
    gcdummy = BoolOption('dummy', 'dummy', default=False, properties=('force_store_value',))
    gcgroup = OptionDescription('gc', '', [gcdummy])
    descr = OptionDescription('tiramisu', '', [gcgroup])
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    option = cfg.cfgimpl_get_description().gc.dummy
    assert tuple(option._properties) == tuple(['force_store_value'])
    assert not 'test' in setting[option]
    setting[option].append('test')
    assert tuple(option._properties) == tuple(['force_store_value'])
    assert 'test' in setting[option]


def test_reset_properties_force_store_value():
    gcdummy = BoolOption('dummy', 'dummy', default=False, properties=('force_store_value',))
    gcgroup = OptionDescription('gc', '', [gcdummy])
    descr = OptionDescription('tiramisu', '', [gcgroup])
    cfg = Config(descr)
    setting = cfg.cfgimpl_get_settings()
    option = cfg.cfgimpl_get_description().gc.dummy
    assert setting._p_.get_modified_properties() == {}
    setting.append('frozen')
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'cache', 'validator', 'warnings'))}
    setting.reset()
    assert setting._p_.get_modified_properties() == {}
    setting[option].append('test')
    assert setting._p_.get_modified_properties() == {'gc.dummy': set(('test', 'force_store_value'))}
    setting.reset()
    assert setting._p_.get_modified_properties() == {'gc.dummy': set(('test', 'force_store_value'))}
    setting.append('frozen')
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'validator', 'cache', 'warnings')), 'gc.dummy': set(('test', 'force_store_value'))}
    setting.reset(option)
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'validator', 'cache', 'warnings'))}
    setting[option].append('test')
    assert setting._p_.get_modified_properties() == {None: set(('frozen', 'expire', 'validator', 'cache', 'warnings')), 'gc.dummy': set(('test', 'force_store_value'))}
    setting.reset(all_properties=True)
    assert setting._p_.get_modified_properties() == {}
    raises(ValueError, 'setting.reset(all_properties=True, opt=option)')


def test_pprint():
    msg_error = _("cannot access to {0} \"{1}\" because has {2} {3}")
    msg_is = _('the value of "{0}" is "{1}"')
    msg_is_not = _('the value of "{0}" is not "{1}"')
    properties = _('properties')
    prop = _('property')

    s = StrOption("string", "", default=["string"], default_multi="string", multi=True, properties=('hidden', 'disabled'))
    s2 = StrOption("string2", "", default="string")
    s3 = StrOption("string3", "", default=["string"], default_multi="string", multi=True, properties=('hidden',))
    intoption = IntOption('int', 'Test int option', default=0)
    stroption = StrOption('str', 'Test string option', default="abc",
                          requires=[{'option': intoption, 'expected': 2, 'action': 'hidden', 'inverse': True},
                    {'option': intoption, 'expected': 3, 'action': 'hidden', 'inverse': True},
                    {'option': intoption, 'expected': 4, 'action': 'hidden', 'inverse': True},
                    {'option': intoption, 'expected': 1, 'action': 'disabled'},
                    {'option': s2, 'expected': 'string', 'action': 'disabled'}])

    val2 = StrOption('val2', "")
    descr2 = OptionDescription("options", "", [val2], requires=[{'option': intoption, 'expected': 1, 'action': 'hidden'}])

    val3 = StrOption('val3', "", requires=[{'option': stroption, 'expected': '2', 'action': 'hidden', 'inverse': True}])

    descr = OptionDescription("options", "", [s, s2, s3, intoption, stroption, descr2, val3])
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    config.int = 1
    try:
        config.str
    except Exception as error:
        err = error
        pass

    list_disabled_1 = 'disabled (' + display_list([msg_is.format('string2', 'string'), msg_is.format('Test int option', '1')]) + ')'
    list_disabled_2 = 'disabled (' + display_list([msg_is.format('Test int option', '1'), msg_is.format('string2', 'string')]) + ')'
    list_hidden = 'hidden (' + msg_is_not.format('Test int option', display_list([2, 3, 4], 'or')) + ')'
    comp1 = str(err) == _(msg_error.format('option', 'Test string option', properties, display_list([list_disabled_1, list_hidden])))
    comp2 = str(err) == _(msg_error.format('option', 'Test string option', properties, display_list([list_disabled_2, list_hidden])))
    comp3 = str(err) == _(msg_error.format('option', 'Test string option', properties, display_list([list_hidden, list_disabled_1])))
    comp4 = str(err) == _(msg_error.format('option', 'Test string option', properties, display_list([list_hidden, list_disabled_2])))
    assert comp1 or comp2 or comp3 or comp4

    try:
        config.options.val2
    except Exception as error:
        err = error
        pass

    assert str(err) == msg_error.format('optiondescription', 'options', prop, 'hidden (' + msg_is.format('Test int option', 1) + ')')

    try:
        config.val3
    except Exception as error:
        err = error
        pass
    msg_1 = msg_is.format('string2', 'string')
    msg_2 = msg_is.format('Test int option', 1)
    msg_3 = msg_is_not.format('Test int option', display_list([2, 3, 4], 'or'))

    list_hidden = 'hidden (' + display_list([msg_1, msg_2, msg_3]) + ')'
    comp1 = str(err) == msg_error.format('option', 'val3', prop, list_hidden)
    list_hidden = 'hidden (' + display_list([msg_1, msg_3, msg_2]) + ')'
    comp2 = str(err) == msg_error.format('option', 'val3', prop, list_hidden)
    list_hidden = 'hidden (' + display_list([msg_2, msg_1, msg_3]) + ')'
    comp3 = str(err) == msg_error.format('option', 'val3', prop, list_hidden)
    list_hidden = 'hidden (' + display_list([msg_2, msg_3, msg_1]) + ')'
    comp4 = str(err) == msg_error.format('option', 'val3', prop, list_hidden)
    list_hidden = 'hidden (' + display_list([msg_3, msg_1, msg_2]) + ')'
    comp5 = str(err) == msg_error.format('option', 'val3', prop, list_hidden)
    list_hidden = 'hidden (' + display_list([msg_3, msg_2, msg_1]) + ')'
    comp6 = str(err) == msg_error.format('option', 'val3', prop, list_hidden)
    assert comp1 or comp2 or comp3 or comp4 or comp5 or comp6

    try:
        config.string
    except Exception as error:
        err = error
        pass

    assert str(err) == msg_error.format('option', 'string', properties, display_list(['disabled', 'hidden']))

    try:
        config.string3
    except Exception as error:
        err = error
        pass

    assert str(err) == msg_error.format('option', 'string3', prop, 'hidden')
