# Copyright (C) 2013-2014 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________

"""Config's informations are, by default, volatiles. This means, all values and
settings changes will be lost.

The storage is the system Tiramisu uses to communicate with various DB.
You can specified a persistent storage.

Storage is basic components used to set Config informations in DB.
The primary "entry point" class is the StorageType and it's public
configurator ``set_storage()``.
"""


from time import time
from random import randint
import os
from ..error import ConfigError
from ..i18n import _


MODULE_PATH = os.path.split(os.path.split(os.path.split(__file__)[0])[0])[1]


class StorageType(object):
    """Object to store storage's type. If a Config is already set,
    default storage is store as selected storage. You cannot change it
    after.
    """
    default_storage = os.environ.get('TIRAMISU_STORAGE', 'dictionary')
    storage_type = None
    mod = None

    def set(self, name):  # pragma: optional cover
        if self.storage_type is not None:
            if self.storage_type == name:
                return
            raise ConfigError(_('storage_type is already set, cannot rebind it'))
        self.storage_type = name

    def get(self):
        if self.storage_type is None:
            self.storage_type = self.default_storage
        if self.mod is None:
            modulepath = '{0}.storage.{1}'.format(MODULE_PATH, self.storage_type)
            try:
                mod = __import__(modulepath)
            except ImportError:
                raise SystemError(_('cannot import the storage {0}').format(
                    self.default_storage))
            for token in modulepath.split(".")[1:]:
                mod = getattr(mod, token)
            self.mod = mod
        return self.mod


storage_type = StorageType()
storage_option_type = StorageType()
storage_validation = StorageType()
storage_validation.set('dictionary')


def set_storage(type_, name, **kwargs):  # pragma: optional cover
    """Change storage's configuration

    :params name: is the storage name. If storage is already set, cannot
        reset storage name

    Other attributes are differents according to the selected storage's name
    """
    if type_ == 'option':
        storage_option_type.set(name)
        setting = storage_option_type.get().setting
    else:
        storage_type.set(name)
        setting = storage_type.get().setting
    for option, value in kwargs.items():
        try:
            getattr(setting, option)
            setattr(setting, option, value)
        except AttributeError:
            raise ValueError(_('option {0} not already exists in storage {1}'
                               '').format(option, name))


def _impl_getstate_setting():
    setting = storage_type.get().setting
    state = {'name': storage_type.storage_type}
    for var in dir(setting):
        if not var.startswith('_'):
            state[var] = getattr(setting, var)
    return state


def get_storage(type_, session_id, persistent, test):  # pragma: optional cover
    """all used when __setstate__ a Config
    """
    #FIXME ca sert ???
    if type_ == 'option':
        return storage_option_type.get().Storage(session_id, persistent, test)
    elif type_ == 'config':
        return storage_type.get().Storage(session_id, persistent, test)
    else:
        return storage_validation.get().Storage(session_id, persistent, test)


def get_storages(context, session_id, persistent):
    def gen_id(config):
        return str(id(config)) + str(time()) + str(randint(0, 500))

    if session_id is None:
        session_id = gen_id(context)
    imp = storage_type.get()
    storage = imp.Storage(session_id, persistent)
    settings = imp.Settings(session_id, storage)
    values = imp.Values(storage)
    return settings, values


def get_storages_option(type_):
    imp = storage_option_type.get()
    if type_ == 'base':
        return imp.StorageBase
    elif type_ == 'masterslaves':
        return imp.StorageMasterSlaves
    else:
        return imp.StorageOptionDescription


def get_storages_validation():
    imp = storage_validation.get()
    storage = imp.Storage('__validator_storage', persistent=False, test=True)
    return imp.Values(storage)


def list_sessions(type_):  # pragma: optional cover
    """List all available session (persistent or not persistent)
    """
    if type_ == 'option':
        return storage_option_type.get().list_sessions()
    else:
        return storage_type.get().list_sessions()


def delete_session(type_, session_id):  # pragma: optional cover
    """Delete a selected session, be careful, you can deleted a session
    use by an other instance
    :params session_id: id of session to delete
    """
    if type_ == 'option':
        storage_option_type.get().delete_session(session_id)
    else:
        storage_module = storage_type.get()
        session = storage_module.storage.getsession()
        storage_module.value.delete_session(session_id, session)
        storage_module.storage.delete_session(session_id, session)
        session.commit()
        del(session)


__all__ = (set_storage, list_sessions, delete_session)
