"configuration objects global API"
from .autopath import do_autopath
do_autopath()

from py.test import raises

from tiramisu.config import Config
from tiramisu.option import IntOption, FloatOption, StrOption, ChoiceOption, \
    BoolOption, FilenameOption, UnicodeOption, SymLinkOption, IPOption, \
    PortOption, NetworkOption, NetmaskOption, BroadcastOption, \
    DomainnameOption, OptionDescription
from tiramisu.error import PropertiesOptionError


def make_description():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False)
    prop = BoolOption('prop', '', properties=('disabled',))
    prop2 = BoolOption('prop', '', properties=('hidden',))
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), 'std')
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    booloption2 = BoolOption('bool', 'Test boolean option', default=False)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption2 = FloatOption('float', 'Test float option', default=2.3)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")
    boolop = BoolOption('boolop', 'Test boolean option op', default=True)
    wantref_option = BoolOption('wantref', 'Tests', default=False)
    wantframework_option = BoolOption('wantframework', 'Test', default=False)
    gcgroup2 = OptionDescription('gc2', '', [booloption2, prop])
    gcgroup = OptionDescription('gc', '', [gcgroup2, gcoption, gcdummy, floatoption, prop2])
    descr = OptionDescription('tiramisu', '', [gcgroup, booloption, objspaceoption,
                                               wantref_option, stroption,
                                               wantframework_option,
                                               intoption, boolop, floatoption2])
    return descr


def _is_same_opt(opt1, opt2):
    if "id" in dir(opt1):
        assert opt1.id == opt2.id
    else:
        assert opt1 == opt2


def test_iter_config():
    "iteration on config object"
    s = StrOption("string", "", default="string")
    s2 = StrOption("string2", "", default="string2")
    descr = OptionDescription("options", "", [s, s2])
    config = Config(descr)
    assert [(name, value) for name, value in config] == \
        [('string', 'string'), ('string2', 'string2')]


def test_iter_config_property():
    "iteration on config object"
    s = StrOption("string", "", default="string", properties=('disabled',))
    s2 = StrOption("string2", "", default="string2")
    descr = OptionDescription("options", "", [s, s2])
    config = Config(descr)
    config.read_only()
    assert [(name, value) for name, value in config] == \
        [('string2', 'string2')]


def test_iter_subconfig():
    "iteration on config sub object"
    descr = make_description()
    conf = Config(descr)
    for (name, value), (gname, gvalue) in \
            zip(conf.gc, [("name", "ref"), ("dummy", False)]):
        assert name == gname
        assert value == gvalue


def test_str():
    descr = make_description()
    c = Config(descr)
    c  # does not crash


def test_make_dict():
    "serialization of the whole config to a dict"
    descr = OptionDescription("opt", "", [
        OptionDescription("s1", "", [
            BoolOption("a", "", default=False),
            BoolOption("b", "", default=False, properties=('hidden',))]),
        IntOption("int", "", default=42)])
    config = Config(descr)
    config.read_write()
    config.cfgimpl_get_settings().setpermissive(('hidden',))
    d = config.make_dict()
    assert d == {"s1.a": False, "int": 42}
    config.int = 43
    config.s1.a = True
    d = config.make_dict()
    assert d == {"s1.a": True, "int": 43}
    d2 = config.make_dict(flatten=True)
    assert d2 == {'a': True, 'int': 43}
    raises(ValueError, 'd2 = config.make_dict(withvalue="3")')
    d = config.make_dict(force_permissive=True)
    assert d == {"s1.a": True, "s1.b": False, "int": 43}


def test_make_dict_with_disabled():
    descr = OptionDescription("opt", "", [
        OptionDescription("s1", "", [
            BoolOption("a", "", default=False),
            BoolOption("b", "", default=False, properties=('disabled',))]),
        OptionDescription("s2", "", [
            BoolOption("a", "", default=False),
            BoolOption("b", "", default=False)], properties=('disabled',)),
        IntOption("int", "", default=42)])
    config = Config(descr)
    config.read_only()
    d = config.make_dict()
    assert d == {"s1.a": False, "int": 42}


def test_make_dict_with_disabled_in_callback():
    descr = OptionDescription("opt", "", [
        OptionDescription("s1", "", [
            BoolOption("a", "", default=False),
            BoolOption("b", "", default=False, properties=('disabled',))]),
        OptionDescription("s2", "", [
            BoolOption("a", "", default=False),
            BoolOption("b", "", default=False)], properties=('disabled',)),
        IntOption("int", "", default=42)])
    config = Config(descr)
    config.read_only()
    d = config.make_dict()
    assert d == {"s1.a": False, "int": 42}


def test_make_dict_fullpath():
    descr = OptionDescription("root", "", [
        OptionDescription("opt", "", [
            OptionDescription("s1", "", [
                BoolOption("a", "", default=False),
                BoolOption("b", "", default=False, properties=('disabled',))]),
            OptionDescription("s2", "", [
                BoolOption("a", "", default=False),
                BoolOption("b", "", default=False)], properties=('disabled',)),
            IntOption("int", "", default=42)]),
        IntOption("introot", "", default=42)])
    config = Config(descr)
    config.read_only()
    assert config.make_dict() == {"opt.s1.a": False, "opt.int": 42, "introot": 42}
    assert config.opt.make_dict() == {"s1.a": False, "int": 42}
    assert config.make_dict(fullpath=True) == {"opt.s1.a": False, "opt.int": 42, "introot": 42}
    assert config.opt.make_dict(fullpath=True) == {"opt.s1.a": False, "opt.int": 42}


def test_find_in_config():
    "finds option in config"
    descr = make_description()
    conf = Config(descr)
    conf.read_only()
    conf.cfgimpl_get_settings().setpermissive(('hidden',))
    ret = conf.find(byname='dummy')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.dummy'))
    #
    ret = conf.find_first(byname='dummy')
    _is_same_opt(ret, conf.unwrap_from_path('gc.dummy'))
    #
    ret = conf.find(byname='float')
    assert len(ret) == 2
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.float'))
    _is_same_opt(ret[1], conf.unwrap_from_path('float'))
    _is_same_opt(conf.find_first(byname='bool'), conf.unwrap_from_path('gc.gc2.bool'))
    _is_same_opt(conf.find_first(byname='bool', byvalue=True), conf.unwrap_from_path('bool'))
    _is_same_opt(conf.find_first(byname='dummy'), conf.unwrap_from_path('gc.dummy'))
    _is_same_opt(conf.find_first(byname='float'), conf.unwrap_from_path('gc.float'))
    ret = conf.find(bytype=ChoiceOption)
    assert len(ret) == 2
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.name'))
    _is_same_opt(ret[1], conf.unwrap_from_path('objspace'))
    _is_same_opt(conf.find_first(bytype=ChoiceOption), conf.unwrap_from_path('gc.name'))
    ret = conf.find(byvalue='ref')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.name'))
    _is_same_opt(conf.find_first(byvalue='ref'), conf.unwrap_from_path('gc.name'))
    ret = conf.find(byname='prop')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.prop'))
    conf.read_write()
    raises(AttributeError, "assert conf.find(byname='prop')")
    ret = conf.find(byname='prop', check_properties=False)
    assert len(ret) == 2
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.gc2.prop'))
    _is_same_opt(ret[1], conf.unwrap_from_path('gc.prop'))
    ret = conf.find(byname='prop', force_permissive=True)
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.prop'))
    _is_same_opt(conf.find_first(byname='prop', force_permissive=True), conf.unwrap_from_path('gc.prop'))
    #assert conf.find_first(byname='prop') == conf.unwrap_from_path('gc.prop')
    # combinaison of filters
    ret = conf.find(bytype=BoolOption, byname='dummy')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.dummy'))
    _is_same_opt(conf.find_first(bytype=BoolOption, byname='dummy'), conf.unwrap_from_path('gc.dummy'))
    ret = conf.find(byvalue=False, byname='dummy')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.dummy'))
    _is_same_opt(conf.find_first(byvalue=False, byname='dummy'), conf.unwrap_from_path('gc.dummy'))
    #subconfig
    ret = conf.gc.find(byname='dummy')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.dummy'))
    ret = conf.gc.find(byname='float')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.float'))
    ret = conf.gc.find(byname='bool')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.gc2.bool'))
    _is_same_opt(conf.gc.find_first(byname='bool', byvalue=False), conf.unwrap_from_path('gc.gc2.bool'))
    raises(AttributeError, "assert conf.gc.find_first(byname='bool', byvalue=True)")
    raises(AttributeError, "conf.gc.find(byname='wantref').first()")
    ret = conf.gc.find(byname='prop', check_properties=False)
    assert len(ret) == 2
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.gc2.prop'))
    _is_same_opt(ret[1], conf.unwrap_from_path('gc.prop'))
    conf.read_only()
    ret = conf.gc.find(byname='prop')
    assert len(ret) == 1
    _is_same_opt(ret[0], conf.unwrap_from_path('gc.prop'))
    # not OptionDescription
    raises(AttributeError, "conf.find_first(byname='gc')")
    raises(AttributeError, "conf.gc.find_first(byname='gc2')")
    raises(ValueError, "conf.find(byname='bool', type_='unknown')")


def test_find_multi():
    b = BoolOption('bool', '', multi=True)
    o = OptionDescription('od', '', [b])
    conf = Config(o)
    raises(AttributeError, "conf.find(byvalue=True)")
    raises(AttributeError, "conf.find_first(byvalue=True)")
    conf.bool.append(False)
    raises(AttributeError, "conf.find(byvalue=True)")
    raises(AttributeError, "conf.find_first(byvalue=True)")
    conf.bool.append(False)
    raises(AttributeError, "conf.find(byvalue=True)")
    raises(AttributeError, "conf.find_first(byvalue=True)")
    conf.bool.append(True)
    ret = conf.find(byvalue=True)
    assert len(ret) == 1
    _is_same_opt(ret[0], b)
    _is_same_opt(conf.find_first(byvalue=True), b)


def test_does_not_find_in_config():
    descr = make_description()
    conf = Config(descr)
    conf
    raises(AttributeError, "conf.find(byname='IDontExist')")


def test_filename():
    a = FilenameOption('a', '')
    o = OptionDescription('o', '', [a])
    c = Config(o)
    c.a = u'/'
    c.a = u'/tmp'
    c.a = u'/tmp/'
    c.a = u'/tmp/text.txt'
    c.a = u'tmp'
    c.a = u'tmp/'
    c.a = u'tmp/text.txt'
    raises(ValueError, "c.a = u'/tmp/with space.txt'")
    raises(ValueError, "c.a = u'/tmp/with$.txt'")


def test_iter_all():
    s = StrOption("string", "", default="string")
    s2 = StrOption("string2", "", default="string2")
    descr = OptionDescription("options", "", [s, s2])
    config = Config(descr)
    assert list(config.iter_all()) == [('string', 'string'), ('string2', 'string2')]
    for i in config.iter_all():
        #test StopIteration
        break


def test_iter_all_force_permissive():
    s = StrOption("string", "", default="string")
    s2 = StrOption("string2", "", default="string2")
    s3 = StrOption("string3", "", default="string3", properties=('hidden',))
    descr = OptionDescription("options", "", [s, s2, s3])
    config = Config(descr)
    config.read_write()
    config.cfgimpl_get_settings().setpermissive(('hidden',))
    assert list(config.iter_all()) == [('string', 'string'), ('string2', 'string2')]
    assert list(config.iter_all(force_permissive=True)) == [('string', 'string'),
                                                            ('string2', 'string2'),
                                                            ('string3', 'string3')]


def test_iter_all_prop():
    s = StrOption("string", "", default="string", properties=('disabled',))
    s2 = StrOption("string2", "", default="string2")
    descr = OptionDescription("options", "", [s, s2])
    config = Config(descr)
    config.read_only()
    assert list(config.iter_all()) == [('string2', 'string2')]


def test_impl_getpaths():
    s = StrOption("string", "", default="string", properties=('disabled',))
    s2 = StrOption("string2", "", default="string2")
    s3 = StrOption("string3", "", default="string3")
    s4 = StrOption("string4", "", default="string4", properties=('hidden',))
    od = OptionDescription('od', '', [s3, s4])
    descr = OptionDescription("options", "", [s, s2, od])
    config = Config(descr)
    assert ['string', 'string2', 'od.string3', 'od.string4'] == config.cfgimpl_get_description().impl_getpaths()
    assert ['string', 'string2', 'od', 'od.string3', 'od.string4'] == config.cfgimpl_get_description().impl_getpaths(include_groups=True)
    config.read_write()
    raises(PropertiesOptionError, "config.od.string4")
    assert ['string', 'string2', 'od.string3', 'od.string4'] == config.cfgimpl_get_description().impl_getpaths()
    assert ['string', 'string2', 'od', 'od.string3', 'od.string4'] == config.cfgimpl_get_description().impl_getpaths(include_groups=True)


def test_invalid_option():
    ChoiceOption('a', '', ('1', '2'))
    raises(TypeError, "ChoiceOption('a', '', [1, 2])")
    raises(TypeError, "ChoiceOption('a', '', 1)")
    raises(ValueError, "ChoiceOption('a', '', (1,), 3)")
    FloatOption('a', '')
    raises(ValueError, "FloatOption('a', '', 'string')")
    UnicodeOption('a', '')
    raises(ValueError, "UnicodeOption('a', '', 1)")
    u = UnicodeOption('a', '')
    SymLinkOption('a', u)
    raises(ValueError, "SymLinkOption('a', 'string')")
    IPOption('a', '')
    raises(ValueError, "IPOption('a', '', 1)")
    raises(ValueError, "IPOption('a', '', 'string')")
    PortOption('a', '')
    raises(ValueError, "PortOption('a', '', 'string')")
    raises(ValueError, "PortOption('a', '', '11:12:13', allow_range=True)")
    raises(ValueError, "PortOption('a', '', 11111111111111111111)")
    raises(ValueError, "PortOption('a', '', allow_zero=True, allow_wellknown=False, allow_registred=True, allow_private=False)")
    raises(ValueError, "PortOption('a', '', allow_zero=True, allow_wellknown=True, allow_registred=False, allow_private=True)")
    raises(ValueError, "PortOption('a', '', allow_zero=True, allow_wellknown=False, allow_registred=False, allow_private=True)")
    raises(ValueError, "PortOption('a', '', allow_zero=True, allow_wellknown=False, allow_registred=True, allow_private=True)")
    raises(ValueError, "PortOption('a', '', allow_zero=False, allow_wellknown=False, allow_registred=False, allow_private=False)")
    NetworkOption('a', '')
    raises(ValueError, "NetworkOption('a', '', 'string')")
    NetmaskOption('a', '')
    raises(ValueError, "NetmaskOption('a', '', 'string')")
    BroadcastOption('a', '')
    raises(ValueError, "BroadcastOption('a', '', 'string')")
    DomainnameOption('a', '')
    raises(ValueError, "DomainnameOption('a', '', 'string')")
    raises(ValueError, "DomainnameOption('a', '', type_='string')")
    raises(ValueError, "DomainnameOption('a', '', allow_ip='string')")
    raises(ValueError, "DomainnameOption('a', '', allow_without_dot='string')")
    raises(ValueError, "DomainnameOption('a', '', 1)")
    #
    ChoiceOption('a', '', (1,), multi=True, default_multi=1)
    raises(ValueError, "ChoiceOption('a', '', (1,), default_multi=1)")
    raises(ValueError, "ChoiceOption('a', '', (1,), multi=True, default=[1,], default_multi=2)")
    raises(ValueError, "FloatOption('a', '', multi=True, default_multi='string')")
    raises(ValueError, "UnicodeOption('a', '', multi=True, default_multi=1)")
    raises(ValueError, "IPOption('a', '', multi=True, default_multi=1)")
    raises(ValueError, "IPOption('a', '', multi=True, default_multi='string')")
    raises(ValueError, "PortOption('a', '', multi=True, default_multi='string')")
    raises(ValueError, "PortOption('a', '', multi=True, default_multi='11:12:13', allow_range=True)")
    raises(ValueError, "PortOption('a', '', multi=True, default_multi=11111111111111111111)")
    raises(ValueError, "NetworkOption('a', '', multi=True, default_multi='string')")
    raises(ValueError, "NetmaskOption('a', '', multi=True, default_multi='string')")
    raises(ValueError, "BroadcastOption('a', '', multi=True, default_multi='string')")
    raises(ValueError, "DomainnameOption('a', '', multi=True, default_multi='string')")
    raises(ValueError, "DomainnameOption('a', '', multi=True, default_multi=1)")
