# coding: utf-8
"frozen and hidden values"
from .autopath import do_autopath
do_autopath()

from py.test import raises

from tiramisu.config import Config
from tiramisu.option import ChoiceOption, BoolOption, IntOption, FloatOption, \
    PasswordOption, StrOption, DateOption, OptionDescription
from tiramisu.error import PropertiesOptionError


def make_description():
    gcoption = ChoiceOption('name', 'GC name', ('ref', 'framework'), 'ref')
    gcdummy = BoolOption('dummy', 'dummy', default=False, properties=(('hidden'),))
    objspaceoption = ChoiceOption('objspace', 'Object space',
                                  ('std', 'thunk'), ['std'], multi=True)
    booloption = BoolOption('bool', 'Test boolean option', default=True)
    intoption = IntOption('int', 'Test int option', default=0)
    floatoption = FloatOption('float', 'Test float option', default=2.3)
    stroption = StrOption('str', 'Test string option', default="abc")

    wantref_option = BoolOption('wantref', 'Test requires', default=False,
                                requires=({'option': gcoption, 'expected': 'ref', 'action': 'hidden'},))
    wantframework_option = BoolOption('wantframework', 'Test requires',
                                      default=False,
                                      requires=({'option': gcoption, 'expected': 'framework', 'action': 'hidden'},))

    # ____________________________________________________________
    booloptiontwo = BoolOption('booltwo', 'Test boolean option two', default=False)
    subgroup = OptionDescription('subgroup', '', [booloptiontwo])
    # ____________________________________________________________

    gcgroup = OptionDescription('gc', '', [subgroup, gcoption, gcdummy, floatoption])
    descr = OptionDescription('trs', '', [gcgroup, booloption, objspaceoption,
                                          wantref_option, stroption,
                                          wantframework_option,
                                          intoption])
    return descr


# ____________________________________________________________
def test_is_hidden():
    descr = make_description()
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    dummy = config.unwrap_from_path('gc.dummy')
    assert not 'frozen' in setting[dummy]
    # setattr
    raises(PropertiesOptionError, "config.gc.dummy == False")
    # getattr
    raises(PropertiesOptionError, "config.gc.dummy")


def test_group_is_hidden():
    descr = make_description()
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    gc = config.unwrap_from_path('gc')
    config.unwrap_from_path('gc.dummy')
    setting[gc].append('hidden')
    raises(PropertiesOptionError, "config.gc.dummy")
    assert 'hidden' in setting[gc]
    raises(PropertiesOptionError, "config.gc.float")
    # manually set the subconfigs to "show"
    setting[gc].remove('hidden')
    assert not 'hidden' in setting[gc]
    assert config.gc.float == 2.3
    #dummy est en hide
    prop = []
    try:
        config.gc.dummy = False
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'hidden' in prop

def test_extend_properties():
    descr = make_description()
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    gc = config.unwrap_from_path('gc')
    config.unwrap_from_path('gc.dummy')
    setting[gc].extend(['hidden', 'user_defined_property'])
    assert 'hidden' in setting[gc]
    assert 'user_defined_property' in setting[gc]

def test_group_is_hidden_multi():
    descr = make_description()
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    obj = config.unwrap_from_path('objspace')
    objspace = config.objspace
    setting[obj].append('hidden')
    raises(PropertiesOptionError, "config.objspace")
    assert 'hidden' in setting[obj]
    prop = []
    try:
        objspace.append('std')
    except PropertiesOptionError as err:
        prop = err.proptype
    assert 'hidden' in prop
    setting[obj].remove('hidden')
    assert not 'hidden' in setting[obj]
    config.objspace.append('std')


def test_global_show():
    descr = make_description()
    config = Config(descr)
    setting = config.cfgimpl_get_settings()
    config.read_write()
    dummy = config.unwrap_from_path('gc.dummy')
    setting[dummy].append('hidden')
    assert 'hidden' in setting[dummy]
    raises(PropertiesOptionError, "config.gc.dummy == False")


def test_with_many_subgroups():
    descr = make_description()
    config = Config(descr)
    booltwo = config.unwrap_from_path('gc.subgroup.booltwo')
    setting = config.cfgimpl_get_settings()
    assert not 'hidden' in setting[booltwo]
    assert config.gc.subgroup.booltwo is False
    setting[booltwo].append('hidden')
    path = 'gc.subgroup.booltwo'
    homeconfig, name = config.cfgimpl_get_home_by_path(path)
    assert name == "booltwo"
    getattr(homeconfig.cfgimpl_get_description(), name)
    assert 'hidden' in setting[booltwo]


def test_password_option():
    o = PasswordOption('o', '')
    d = OptionDescription('d', '', [o])
    c = Config(d)

    c.o = u'a_valid_password'
    raises(ValueError, "c.o = 1")


def test_date_option():
    o = DateOption('o', '')
    d = OptionDescription('d', '', [o])
    c = Config(d)

    c.o = u'2017-02-04'
    c.o = u'2017-2-4'
    raises(ValueError, "c.o = 1")
    raises(ValueError, "c.o = u'2017-13-20'")
    raises(ValueError, "c.o = u'2017-11-31'")
    raises(ValueError, "c.o = u'2017-12-32'")
    raises(ValueError, "c.o = u'2017-2-29'")
    raises(ValueError, "c.o = u'2-2-2017'")
    raises(ValueError, "c.o = u'2017/2/2'")
