# -*- coding: utf-8 -*-
""
# Copyright (C) 2014 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ____________________________________________________________
from tiramisu.i18n import _
from tiramisu.setting import groups, undefined
from tiramisu.error import ConfigError
from .util import SqlAlchemyBase
import util

from sqlalchemy import not_, or_, and_, inspect
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.ext.associationproxy import association_proxy
from sqlalchemy import Column, Integer, String, Boolean, PickleType, \
    ForeignKey, Table
from sqlalchemy.orm import relationship, backref
from sqlalchemy.orm.collections import attribute_mapped_collection

from itertools import chain


def load_requires(collection_type, proxy):
    def getter(obj):
        if obj is None:
            return None
        ret = []
        requires = getattr(obj, proxy.value_attr)
        session = util.Session()
        for require in requires:
            option = session.query(_Base).filter_by(id=require.option).first()
            ret.append(tuple([option, require.expected, require.action, require.inverse, require.transitive, require.same_action]))
        return tuple(ret)

    def setter(obj, value):
        setattr(obj, proxy.value_attr, value)
    return getter, setter


class _Require(SqlAlchemyBase):
    __tablename__ = "require"
    id = Column(Integer, primary_key=True)
    requires_id = Column(Integer, ForeignKey("baseoption.id"), nullable=False)
    requires = relationship('_RequireOption')

    def __init__(self, requires):
        for require in requires:
            self.requires.append(_RequireOption(require))


class _RequireOption(SqlAlchemyBase):
    __tablename__ = 'requireoption'
    id = Column(Integer, primary_key=True)
    require_id = Column(Integer, ForeignKey("require.id"), nullable=False)
    option = Column(Integer, nullable=False)
    _expected = relationship("_RequireExpected", collection_class=list,
                             cascade="all, delete-orphan")
    expected = association_proxy("_expected", "expected")
    #expected = Column(String)
    action = Column(String, nullable=False)
    inverse = Column(Boolean, default=False)
    transitive = Column(Boolean, default=True)
    same_action = Column(Boolean, default=True)

    def __init__(self, values):
        option, expected, action, inverse, transitive, same_action = values
        self.option = option.id
        self.expected = expected
        self.action = action
        self.inverse = inverse
        self.transitive = transitive
        self.same_action = same_action


class _RequireExpected(SqlAlchemyBase):
    __tablename__ = 'expected'
    id = Column(Integer, primary_key=True)
    require = Column(Integer, ForeignKey('requireoption.id'), nullable=False)
    expected = Column(PickleType)

    def __init__(self, expected):
        #FIXME ne pas creer plusieurs fois la meme _expected_
        #FIXME pareil avec calc_properties
        self.expected = expected


class _CalcProperties(SqlAlchemyBase):
    __tablename__ = 'calcproperty'
    id = Column(Integer, primary_key=True)
    require = Column(Integer, ForeignKey('baseoption.id'), nullable=False)
    name = Column(PickleType)

    def __init__(self, name):
        #FIXME ne pas creer plusieurs fois la meme _expected_
        #FIXME pareil avec calc_properties
        self.name = name


#____________________________________________________________
#
# properties
class _PropertyOption(SqlAlchemyBase):
    __tablename__ = 'propertyoption'
    id = Column(Integer, primary_key=True)
    option = Column(Integer, ForeignKey('baseoption.id'), nullable=False)
    name = Column(String)

    def __init__(self, name):
        self.name = name


#____________________________________________________________
#
# information
class _Information(SqlAlchemyBase):
    __tablename__ = 'information'
    id = Column(Integer, primary_key=True)
    option = Column(String, ForeignKey('baseoption.id'), nullable=False)
    key = Column(String)
    value = Column(PickleType)

#    def __init__(self, option, key, value):
#        self.option = option
#        self.key = key
#        self.value = value


#____________________________________________________________
#
# callback
def load_callback_parm(collection_type, proxy):
    def getter(obj):
        if obj is None:
            return None
        ret = []
        requires = getattr(obj, proxy.value_attr)
        session = util.Session()
        for require in requires:
            if require.value is not None:
                ret.append(require.value)
            else:
                option = session.query(_Base).filter_by(id=require.option).first()
                ret.append((option, require.force_permissive))
        return tuple(ret)

    def setter(obj, value):
        setattr(obj, proxy.value_attr, value)
    return getter, setter


class _CallbackParamOption(SqlAlchemyBase):
    __tablename__ = 'callback_param_option'
    id = Column(Integer, primary_key=True)
    callback_param = Column(Integer, ForeignKey('callback_param.id'))
    option = Column(Integer)
    force_permissive = Column(Boolean)
    value = Column(PickleType)

    def __init__(self, option=undefined, force_permissive=undefined,  value=undefined):
        if value is not undefined:
            self.value = value
        elif option is not undefined:
            self.option = option.id
            self.force_permissive = force_permissive


class _CallbackParam(SqlAlchemyBase):
    __tablename__ = 'callback_param'
    id = Column(Integer, primary_key=True)
    callback = Column(Integer, ForeignKey('baseoption.id'))
    key = Column(String)
    params = relationship('_CallbackParamOption')

    def __init__(self, key, params):
        self.key = key
        for param in params:
            if isinstance(param, tuple):
                if param == (None,):
                    self.params.append(_CallbackParamOption())
                else:
                    self.params.append(_CallbackParamOption(option=param[0],
                                                            force_permissive=param[1]))
            else:
                self.params.append(_CallbackParamOption(value=param))


#____________________________________________________________
#
# choice
class _ChoiceParamOption(SqlAlchemyBase):
    __tablename__ = 'choice_param_option'
    id = Column(Integer, primary_key=True)
    choice = Column(Integer, index=True)
    option = Column(Integer)
    force_permissive = Column(Boolean)
    value = Column(PickleType)

    def __init__(self, choice, option=undefined, force_permissive=undefined,  value=undefined):
        self.choice = choice.id
        if value is not undefined:
            self.value = value
        elif option is not undefined:
            self.option = option.id
            self.force_permissive = force_permissive


class _ChoiceParam(SqlAlchemyBase):
    __tablename__ = 'choice_param'
    id = Column(Integer, primary_key=True)
    option = Column(Integer, index=True)
    key = Column(String)

    def __init__(self, option, key):
        self.option = option.id
        self.key = key


#def load_choice_parm(collection_type, proxy):
#    def getter(obj):
#        if obj is None:
#            return None
#        ret = []
#        requires = getattr(obj, proxy.value_attr)
#        session = util.Session()
#        for require in requires:
#            if require.value is not None:
#                ret.append(require.value)
#            else:
#                option = session.query(_Base).filter_by(id=require.option).first()
#                ret.append((option, require.force_permissive))
#        return tuple(ret)
#
#    def setter(obj, value):
#        setattr(obj, proxy.value_attr, value)
#    return getter, setter
#
#
#class _ChoiceParamOption(SqlAlchemyBase):
#    __tablename__ = 'choice_param_option'
#    id = Column(Integer, primary_key=True)
#    valid_param = Column(Integer, ForeignKey('choice_param.id'))
#    option = Column(Integer)
#    force_permissive = Column(Boolean)
#    value = Column(PickleType)
#
#    def __init__(self, option=undefined, force_permissive=undefined,  value=undefined):
#        if value is not undefined:
#            self.value = value
#        elif option is not undefined:
#            self.option = option.id
#            self.force_permissive = force_permissive
#
#
#class _ChoiceParam(SqlAlchemyBase):
#    __tablename__ = 'choice_param'
#    id = Column(Integer, primary_key=True)
#    choice = Column(Integer, ForeignKey('baseoption.id'))
#    key = Column(String)
#    params = relationship('_ChoiceParamOption')
#
#    def __init__(self, key, params):
#        self.key = key
#        for param in params:
#            if isinstance(param, tuple):
#                if param == (None,):
#                    self.params.append(_ChoiceParamOption())
#                else:
#                    self.params.append(_ChoiceParamOption(option=param[0],
#                                                            force_permissive=param[1]))
#            else:
#                self.params.append(_ChoiceParamOption(value=param))


#____________________________________________________________
#
# validator
def load_validator_parm(collection_type, proxy):
    def getter(obj):
        if obj is None:
            return None
        ret = []
        requires = getattr(obj, proxy.value_attr)
        session = util.Session()
        for require in requires:
            if require.value is not None:
                ret.append(require.value)
            else:
                option = session.query(_Base).filter_by(id=require.option).first()
                ret.append((option, require.force_permissive))
        return tuple(ret)

    def setter(obj, value):
        setattr(obj, proxy.value_attr, value)
    return getter, setter


class _ValidatorParamOption(SqlAlchemyBase):
    __tablename__ = 'validator_param_option'
    id = Column(Integer, primary_key=True)
    validator_param = Column(Integer, ForeignKey('validator_param.id'))
    option = Column(Integer)
    force_permissive = Column(Boolean)
    value = Column(PickleType)

    def __init__(self, option=undefined, force_permissive=undefined,  value=undefined):
        if value is not undefined:
            self.value = value
        elif option is not undefined:
            self.option = option.id
            self.force_permissive = force_permissive


class _ValidatorParam(SqlAlchemyBase):
    __tablename__ = 'validator_param'
    id = Column(Integer, primary_key=True)
    validator = Column(Integer, ForeignKey('baseoption.id'))
    key = Column(String)
    params = relationship('_ValidatorParamOption')

    def __init__(self, key, params):
        self.key = key
        for param in params:
            if isinstance(param, tuple):
                if param == (None,):
                    self.params.append(_ValidatorParamOption())
                else:
                    self.params.append(_ValidatorParamOption(option=param[0],
                                                            force_permissive=param[1]))
            else:
                self.params.append(_ValidatorParamOption(value=param))


#____________________________________________________________
#
# consistency
consistency_table = Table('consistencyopt', SqlAlchemyBase.metadata,
                          Column('id', Integer, primary_key=True),
                          Column('left_id', Integer, ForeignKey('consistency.id')),
                          Column('right_id', Integer, ForeignKey('baseoption.id'))
                          )


class _Consistency(SqlAlchemyBase):
    __tablename__ = 'consistency'
    id = Column(Integer, primary_key=True)
    func = Column(PickleType)
    params = Column(PickleType)

    def __init__(self, func, all_cons_opts, params):
        self.func = func
        for option in all_cons_opts[1:]:
            option._consistencies.append(self)
        self.params = params


class _Parent(SqlAlchemyBase):
    __tablename__ = 'parent'
    id = Column(Integer, primary_key=True)
    child_id = Column(Integer)
    child_name = Column(String)
    parent_id = Column(Integer)

    def __init__(self, parent, child):
        self.parent_id = parent.id
        self.child_id = child.id
        self.child_name = child._name


#____________________________________________________________
#
# Base
class _Base(SqlAlchemyBase):
    __tablename__ = 'baseoption'
    id = Column(Integer, primary_key=True)
    _name = Column(String)
    #FIXME not autoload
#    _infos = relationship("_Information",
#                          collection_class=attribute_mapped_collection('key'),
#                          cascade="all, delete-orphan")
#    _informations = association_proxy("_infos", "value")
    _informations = relationship("_Information")
    _default = Column(PickleType)
    _default_multi = Column(PickleType)
    _subdyn = Column(Integer)
    _dyn = Column(String)
    _opt = Column(Integer)
    _master_slaves = Column(Integer)
    _choice_values = Column(PickleType)
    #_cho_params = relationship('_ChoiceParam',
    #                           collection_class=attribute_mapped_collection('key'))
    #_choice_values_params = association_proxy("_cho_params", "params",
    #                                   getset_factory=load_choice_parm)
    _reqs = relationship("_Require", collection_class=list)
    _requires = association_proxy("_reqs", "requires", getset_factory=load_requires)
    _multi = Column(Integer)
    ######
    _callback = Column(PickleType)
    _call_params = relationship('_CallbackParam',
                                collection_class=attribute_mapped_collection('key'))
    _callback_params = association_proxy("_call_params", "params",
                                         getset_factory=load_callback_parm)
    _validator = Column(PickleType)
    _val_params = relationship('_ValidatorParam',
                               collection_class=attribute_mapped_collection('key'))
    _validator_params = association_proxy("_val_params", "params",
                                          getset_factory=load_validator_parm)
    ######
    #FIXME not autoload
    _props = relationship("_PropertyOption", collection_class=set)
    _properties = association_proxy("_props", "name")
    _calc_props = relationship("_CalcProperties", collection_class=set)
    _calc_properties = association_proxy("_calc_props", "name")
    _warnings_only = Column(Boolean)
    _allow_empty_list = Column(Boolean)
    _readonly = Column(Boolean, default=False)
    _consistencies = relationship('_Consistency', secondary=consistency_table,
                                  backref=backref('options',
                                                  enable_typechecks=False))
    _stated = Column(Boolean)
    _type = Column(String(50))
    __mapper_args__ = {
        'polymorphic_identity': 'optionsql',
        'polymorphic_on': _type
    }
    _extra = Column(PickleType)
    #FIXME devrait etre une table
    _group_type = Column(String)
    _is_build_cache = Column(Boolean, default=False)

    def __init__(self, name, multi, warnings_only, doc, extra, calc_properties,
                 requires, properties, allow_empty_list, opt=undefined, session=None):
        self._name = name
        if multi is not undefined:
            self._multi = multi
        if warnings_only is not undefined:
            self._warnings_only = warnings_only
        if allow_empty_list is not undefined:
            self._allow_empty_list = allow_empty_list
        if doc is not undefined:
            self._informations = [_Information(key='doc', value=doc)]
            #self._informations = {'doc': doc}
        if opt is not undefined:
            self._opt = opt.id
        if extra is not undefined:
            self._extra = extra
        if calc_properties is not undefined:
            self._calc_properties = calc_properties
        if requires is not undefined:
            self._requires = requires
        if properties is not undefined:
            self._properties = properties
        session.add(self)

    def getsession(self):
        return util.Session()

    def commit(self, session):
        session.commit()
        del(session)

    def _add_consistency(self, func, all_cons_opts, params):
        _Consistency(func, all_cons_opts, params)

    def _set_default_values(self, default, default_multi, is_multi):
        self._default = default
        if self.impl_is_multi() and default_multi is not None:
            err = self._validate(default_multi)
            if err:
                raise err
            self._default_multi = default_multi

    def _get_consistencies(self):
        return [(consistency.func, consistency.options, consistency.params)
                for consistency in self._consistencies]

    def _get_id(self):
        return self.id

    def impl_get_callback(self):
        a=self.getsession().query(_Base).filter_by(id=self.id).first()
        ret = self._callback
        if ret is None:
            return (None, {})
        params = self._callback_params
        if params is None:
            params = {}
        return ret, params

    def impl_get_validator(self):
        ret = self._validator
        if ret is None:
            return (None, {})
        return ret, self._validator_params

    def _impl_getsubdyn(self):
        session = self.getsession()
        return session.query(_Base).filter_by(id=self._subdyn).first()

    def _impl_getopt(self):
        session = self.getsession()
        return session.query(_Base).filter_by(id=self._opt).first()

    def impl_getname(self):
        return self._name

    def impl_getrequires(self):
        session = self.getsession()
        requires = session.query(_Require).filter_by(requires_id=self.id).all()
        for require in requires:
            _ret = []
            for req in require.requires:
                _ret.append((session.query(_Base).filter_by(id=req.option).first(),
                       req.expected,
                       req.action,
                       req.inverse,
                       req.transitive,
                       req.same_action))
            yield(_ret)
            

    def impl_getdefault(self):
        ret = self._default
        if self.impl_is_multi():
            if ret is None:
                return []
            return list(ret)
        return ret

    def impl_getdefault_multi(self):
        if self.impl_is_multi():
            return self._default_multi

    def _get_extra(self, key):
        return self._extra[key]

    def _impl_setopt(self, opt):
        self._opt = opt.id

    def _impl_setsubdyn(self, subdyn):
        session = self.getsession()
        self._subdyn = subdyn.id
        self.commit(session)

    def _set_readonly(self, has_extra):
        session = self.getsession()
        opt = session.query(_Base).filter_by(id=self.id).first()
        opt._readonly = True
        session.commit()
        self._readonly = True

    def _set_callback(self, callback, callback_params):
        self._callback = callback
        if callback_params is not None:
            opt._callback_params = callback_params
        #session = self.getsession()
        #opt = session.query(_Base).filter_by(id=self.id).first()
        #opt._callback = callback
        #if callback_params is not None:
        #    opt._callback_params = callback_params
        #session.commit()

    def impl_set_choice_values_params(self, values, values_params, session):
        self._choice_values = values
        if values_params is not None:
            for key, params in values_params.items():
                choice = _ChoiceParam(self, key)
                session.add(choice)
                session.commit()
                for param in params:
                    if isinstance(param, tuple):
                        if param == (None,):
                            session.add(_ChoiceParamOption(choice))
                        else:
                            session.add(_ChoiceParamOption(choice, option=param[0], force_permissive=param[1]))
                    else:
                        session.add(_ChoiceParamOption(choice, value=param))
        session.commit()

    def impl_get_choice_values_params(self):
        session = self.getsession()
        params = {}
        for param in session.query(_ChoiceParam).filter_by(option=self.id).all():
            _params = []
            for _param in session.query(_ChoiceParamOption).filter_by(choice=param.id).all():
                if _param.value:
                    _params.append(_param.value)
                elif _param.option:
                    _params.append((session.query(_Base).filter_by(id=_param.option).first(),
                                         _param.force_permissive))
                else:
                    _params.append((None,))
            params[param.key] = _params
        return params

    def _set_validator(self, validator, validator_params):
        self._validator = validator
        if validator_params is not None:
            self._validator_params = validator_params

    def impl_is_readonly(self):
        session = self.getsession()
        opt = session.query(_Base).filter_by(id=self.id).first()
        if opt is None or opt._readonly is None:
            return False
        return opt._readonly

    def impl_is_multi(self):
        return self._multi == 0 or self._multi == 2

    def impl_is_submulti(self):
        return self._multi == 2

    def impl_allow_empty_list(self):
        try:
            return self._allow_empty_list
        except AttributeError:
            return undefined

    def _is_warnings_only(self):
        return self._warnings_only

    def impl_get_calc_properties(self):
        session = self.getsession()
        return session.query(_CalcProperties).filter_by(require=self.id).all()
        #try:
        #    return self._calc_properties
        #except AttributeError:
        #    return frozenset()

    # information
    def impl_set_information(self, key, value):
        session = self.getsession()
        val = session.query(_Information).filter_by(
            option=self.id, key=key).first()
        if val is None:
            session.add(_Information(self, key, value))
        else:
            val.value = value
        session.commit()

    def impl_get_information(self, key, default=undefined):
        """retrieves one information's item

        :param key: the item string (ex: "help")
        """
        session = self.getsession()
        val = session.query(_Information).filter_by(
            option=self.id, key=key).first()
        if not val:
            if default is not undefined:
                return default
            raise ValueError(_("information's item not found: {0}").format(
                               key))
        return val.value

    def _impl_getattributes(self):
        slots = set()
        mapper = inspect(self)
        for column in mapper.attrs:
                slots.add(column.key)
        return slots

    def impl_getproperties(self):
        session = self.getsession()
        for prop in session.query(_PropertyOption).filter_by(option=self.id).all():
            yield prop.name

    def _set_master_slaves(self, option):
        session = self.getsession()
        opt = session.query(_Base).filter_by(id=self.id).first()
        opt._master_slaves = option._p_.id
        self.commit(session)

    def _get_master_slave(self):
        session = self.getsession()
        return session.query(StorageMasterSlaves).filter_by(id=self._master_slaves).first()


class Cache(SqlAlchemyBase):
    __tablename__ = 'cache'
    id = Column(Integer, primary_key=True)
    path = Column(String, nullable=False, index=True)
    descr = Column(Integer, nullable=False, index=True)
    parent = Column(Integer, nullable=False, index=True)
    option = Column(Integer, nullable=False, index=True)
    opt_type = Column(String, nullable=False, index=True)
    is_subdyn = Column(Boolean, nullable=False, index=True)
    subdyn_path = Column(String)

    def __init__(self, descr, parent, option, path, subdyn_path):
        #context
        self.descr = descr.id
        self.parent = parent.id
        self.option = option.id
        self.path = path
        self.opt_type = option.__class__.__name__
        if subdyn_path:
            self.is_subdyn = True
            self.subdyn_path = subdyn_path
        else:
            self.is_subdyn = False
            self.subdyn_path = None


class StorageOptionDescription(object):
    def impl_already_build_caches(self):
        cache = self._is_build_cache
        if cache is None:
            cache = False
        return cache

    def impl_get_opt_by_path(self, path):
        session = self.getsession()
        ret = session.query(Cache).filter_by(descr=self.id, path=path).first()
        if ret is None:
            raise AttributeError(_('no option for path {0}').format(path))
        return session.query(_Base).filter_by(id=ret.option).first()

    def impl_get_path_by_opt(self, opt):
        session = self.getsession()
        ret = session.query(Cache).filter_by(descr=self.id,
                                                  option=opt.id).first()
        if ret is None:
            ret = session.query(Cache).filter_by(descr=self.id).first()
            if ret is None:
                raise ConfigError(_('use impl_get_path_by_opt only with root OptionDescription'))
            raise AttributeError(_('no option {0} found').format(opt))
        return ret.path

    def impl_get_group_type(self):
        return getattr(groups, self._group_type)

    def impl_build_cache_option(self, descr=None, _currpath=None,
                                subdyn_path=None, session=None):
        if self.impl_is_readonly() or (_currpath is None and getattr(self, '_cache_paths', None) is not None):
            # cache already set
            return
        if descr is None:
            save = True
            descr = self
            _currpath = []
            session = self.getsession()
        else:
            save = False
        for option in self._impl_getchildren(dyn=False):
            attr = option.impl_getname()
            if isinstance(option, StorageOptionDescription):
                sub = subdyn_path
                if option.impl_is_dynoptiondescription():
                    sub = '.'.join(_currpath)
                session.add(Cache(descr, self, option,
                                       str('.'.join(_currpath + [attr])),
                                       sub))
                _currpath.append(attr)
                option.impl_build_cache_option(descr,
                                               _currpath,
                                               sub, session)
                _currpath.pop()
            else:
                if subdyn_path:
                    subdyn_path = '.'.join(_currpath)
                session.add(Cache(descr, self, option,
                                       str('.'.join(_currpath + [attr])),
                                       subdyn_path))
        if save:
            self._is_build_cache = True
            self.commit(session)

    def impl_get_options_paths(self, bytype, byname, _subpath, only_first,
                               context):
        def _build_ret_opt(opt, option, suffix, name):
            subdyn_path = opt.subdyn_path
            dynpaths = opt.path[len(subdyn_path):].split('.')

            path = subdyn_path
            dot = False
            for dynpath in dynpaths:
                if dot:
                    path += '.'
                path += dynpath + suffix
                dot = True
            _opt = option._impl_to_dyn(name + suffix, path)
            return (path, _opt)

        session = self.getsession()
        sqlquery = session.query(Cache).filter_by(descr=self.id)
        if bytype is None:
            sqlquery = sqlquery.filter(and_(not_(
                Cache.opt_type == 'OptionDescription'),
                not_(Cache.opt_type == 'DynOptionDescription')))
        else:
            sqlquery = sqlquery.filter_by(opt_type=bytype.__name__)

        query = ''
        or_query = ''
        if _subpath is not None:
            query += _subpath + '.%'
        #if byname is not None:
        #    or_query = query + byname
        #    query += '%.' + byname
        if query != '':
            filter_query = Cache.path.like(query)
            if or_query != '':
                filter_query = or_(Cache.path == or_query, filter_query)
            sqlquery = sqlquery.filter(filter_query)
        #if only_first:
        #    opt = sqlquery.first()
        #    if opt is None:
        #        return tuple()
        #    option = util.session.query(_Base).filter_by(id=opt.option).first()
        #    return ((opt.path, option),)
        #else:
        ret = []
        for opt in sqlquery.all():
            option = session.query(_Base).filter_by(id=opt.option).first()
            if opt.is_subdyn:
                name = option.impl_getname()
                if byname is not None:
                    if byname.startswith(name):
                        found = False
                        dynoption = option._impl_getsubdyn()
                        for suffix in dynoption._impl_get_suffixes(
                                context):
                            if byname == name + suffix:
                                found = True
                                break
                        if not found:
                            continue
                        ret_opt = _build_ret_opt(opt, option, suffix, name)
                    else:
                        ret_opt = _build_ret_opt(opt, option, suffix, name)
                else:
                    if not only_first:
                        ret_opt = []
                    dynoption = option._impl_getsubdyn()
                    for suffix in dynoption._impl_get_suffixes(context):
                        val = _build_ret_opt(opt, option, suffix, name)
                        if only_first:
                            ret_opt = val
                        else:
                            ret_opt.append(val)
            else:
                if byname is not None and byname != option.impl_getname():
                    continue
                ret_opt = (opt.path, option)
            if only_first:
                return ret_opt
            if isinstance(ret_opt, list):
                if ret_opt != []:
                    ret.extend(ret_opt)
            else:
                ret.append(ret_opt)
        return ret

    def _add_children(self, child_names, children):
        session = self.getsession()
        for child in children:
            session.add(_Parent(self, child))
        self.commit(session)

    def _impl_st_getchildren(self, context, only_dyn=False):
        session = self.getsession()
        if only_dyn is False or context is undefined:
            for child in session.query(_Parent).filter_by(
                    parent_id=self.id).all():
                yield(session.query(_Base).filter_by(id=child.child_id
                                                          ).first())
        else:
            descr = context.cfgimpl_get_description().id
            for child in session.query(Cache).filter_by(descr=descr,
                                                             parent=self.id
                                                             ).all():
                yield(session.query(_Base).filter_by(id=child.option).first())

    def _getattr(self, name, suffix=undefined, context=undefined, dyn=True):
        error = False
        if suffix is not undefined:
            try:
                if undefined in [suffix, context]:  # pragma: optional cover
                    raise ConfigError(_("suffix and context needed if "
                                        "it's a dyn option"))
                if name.endswith(suffix):
                    session = self.getsession()
                    oname = name[:-len(suffix)]
                    #child = self._children[1][self._children[0].index(oname)]
                    child = session.query(_Parent).filter_by(
                        parent_id=self.id, child_name=oname).first()
                    if child is None:
                        error = True
                    else:
                        opt = session.query(_Base).filter_by(
                            id=child.child_id).first()
                        return self._impl_get_dynchild(opt, suffix)
                else:
                    error = True
            except ValueError:  # pragma: optional cover
                error = True
        else:
            session = self.getsession()
            child = session.query(_Parent).filter_by(parent_id=self.id,
                                                          child_name=name
                                                          ).first()
            if child is None:
                child = self._impl_search_dynchild(name, context=context)
                if child != []:
                    return child
                error = True
            if error is False:
                return session.query(_Base).filter_by(id=child.child_id
                                                           ).first()
        if error:
            raise AttributeError(_('unknown Option {0} in OptionDescription {1}'
                                 '').format(name, self.impl_getname()))

    def _get_force_store_value(self):
        #only option in current tree
        session = self.getsession()
        current_ids = tuple(chain(*session.query(Cache.option).filter_by(
            descr=self.id).all()))
        for prop in session.query(_PropertyOption).filter(
                _PropertyOption.option.in_(current_ids),
                _PropertyOption.name == 'force_store_value').all():
            opt = session.query(_Base).filter_by(id=prop.option).first()
            path = self.impl_get_path_by_opt(opt)
            yield (opt, path)


class StorageBase(_Base):
    @declared_attr
    def __mapper_args__(self):
        return {'polymorphic_identity': self.__name__.lower()}


class _Slave(SqlAlchemyBase):
    __tablename__ = 'slaves'
    id = Column(Integer, primary_key=True)
    master_id = Column(Integer, index=True, nullable=False)
    slave_id = Column(Integer)

    def __init__(self, master, slave):
        self.master_id = master.id
        self.slave_id = slave.id


class StorageMasterSlaves(SqlAlchemyBase):
    __tablename__ = 'masterslaves2'
    id = Column(Integer, primary_key=True)
    master = Column(Integer)

    def __init__(self, master, slaves):
        session = util.Session()
        self.master = master.id
        session.add(self)
        session.commit()
        for slave in slaves:
            sl = _Slave(self, slave)
            session.add(sl)
        session.commit()

    def _sm_getslaves(self):
        session = util.Session()
        for slave in session.query(_Slave).filter_by(master_id=self.master).all():
            yield(session.query(_Base).filter_by(id=slave.slave_id).first())

    def _sm_getmaster(self):
        session = util.Session()
        return session.query(_Base).filter_by(id=self.master).first()
