# -*- coding: utf-8 -*-
"default plugin for setting: set it in a simple dictionary"
# Copyright (C) 2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________
from .sqlite3db import Sqlite3DB


class Settings(Sqlite3DB):
    __slots__ = tuple()

    def __init__(self, storage):
        settings_table = 'CREATE TABLE IF NOT EXISTS property(path text '
        settings_table += 'primary key, properties text)'
        permissives_table = 'CREATE TABLE IF NOT EXISTS permissive(path text '
        permissives_table += 'primary key, permissives text)'
        # should init cache too
        super(Settings, self).__init__(storage)
        self._storage.execute(settings_table, commit=False)
        self._storage.execute(permissives_table)

    # properties
    def setproperties(self, path, properties):
        path = self._sqlite_encode_path(path)
        self._storage.execute("DELETE FROM property WHERE path = ?", (path,),
                              False)
        self._storage.execute("INSERT INTO property(path, properties) VALUES "
                              "(?, ?)", (path,
                                         self._sqlite_encode(properties)))

    def getproperties(self, path, default_properties):
        path = self._sqlite_encode_path(path)
        value = self._storage.select("SELECT properties FROM property WHERE "
                                     "path = ?", (path,))
        if value is None:
            return set(default_properties)
        else:
            return set(self._sqlite_decode(value[0]))

    def hasproperties(self, path):
        path = self._sqlite_encode_path(path)
        return self._storage.select("SELECT properties FROM property WHERE "
                                    "path = ?", (path,)) is not None

    def reset_all_properties(self):
        self._storage.execute("DELETE FROM property")

    def reset_properties(self, path):
        path = self._sqlite_encode_path(path)
        self._storage.execute("DELETE FROM property WHERE path = ?", (path,))

    # permissive
    def setpermissive(self, path, permissive):
        path = self._sqlite_encode_path(path)
        self._storage.execute("DELETE FROM permissive WHERE path = ?", (path,),
                              False)
        self._storage.execute("INSERT INTO permissive(path, permissives) "
                              "VALUES (?, ?)", (path,
                                                self._sqlite_encode(permissive)
                                                ))

    def getpermissive(self, path='_none'):
        permissives = self._storage.select("SELECT permissives FROM "
                                           "permissive WHERE path = ?",
                                           (path,))
        if permissives is None:
            return frozenset()
        else:
            return frozenset(self._sqlite_decode(permissives[0]))

    def get_modified_properties(self):
        """return all modified settings in a dictionary
        example: {'path1': set(['prop1', 'prop2'])}
        """
        ret = {}
        for path, properties in self._storage.select("SELECT * FROM property",
                                                     only_one=False):
            path = self._sqlite_decode_path(path)
            ret[path] = self._sqlite_decode(properties)
        return ret

    def get_modified_permissives(self):
        """return all modified permissives in a dictionary
        example: {'path1': set(['perm1', 'perm2'])}
        """
        ret = {}
        for path, permissives in self._storage.select("SELECT * FROM permissive",
                                                      only_one=False):
            path = self._sqlite_decode_path(path)
            ret[path] = self._sqlite_decode(permissives)
        return ret
