# -*- coding: utf-8 -*-
"utils used by storage"
# Copyright (C) 2013 Team tiramisu (see AUTHORS for all contributors)
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ____________________________________________________________
from ..setting import owners


class SerializeObject(object):
    def __getstate__(self):  # pragma: no cover
        ret = {}
        for key in dir(self):
            if not key.startswith('__'):
                ret[key] = getattr(self, key)
        return ret


class Cache(object):
    __slots__ = ('_cache', '_storage')
    key_is_path = False

    def __init__(self, storage):
        self._cache = {}
        self._storage = storage

    def __getstate__(self):
        slots = set()
        for subclass in self.__class__.__mro__:
            if subclass is not object:
                slots.update(subclass.__slots__)
        slots -= frozenset(['__weakref__', '_storage'])
        states = {}
        for slot in slots:
            try:
                value = getattr(self, slot)
                #value has owners object, need 'str()' it
                if slot == '_values':
                    _value = []
                    _value.append(value[0])
                    _value.append(value[1])
                    str_owner = []
                    _value.append(value[2])
                    for owner in value[3]:
                        if isinstance(owner, list):  # pragma: no cover
                            str_owners = []
                            for subowner in owner:
                                str_owners.append(str(subowner))
                            str_owner.append(str_owners)
                        else:
                            str_owner.append(str(owner))
                    _value.append(str_owner)
                    states[slot] = _value
                else:
                    states[slot] = value
            except AttributeError:  # pragma: no cover
                pass
        return states

    def __setstate__(self, states):
        def convert_owner(owner):
            try:
                owner = getattr(owners, owner)
            except AttributeError:  # pragma: no cover
                owners.addowner(owner)
                owner = getattr(owners, owner)
            return owner

        for key, value in states.items():
            #value has owners object, need to reconstruct it
            if key == '_values':
                _value = []
                _value.append(value[0])
                _value.append(value[1])
                _value.append(value[2])
                obj_owner = []
                for owner in value[3]:
                    if isinstance(owner, list):  # pragma: no cover
                        obj_owners = []
                        for subowner in owner:
                            obj_owners.append(convert_owner(subowner))
                        obj_owner.append(tuple(obj_owners))
                    else:
                        obj_owner.append(convert_owner(owner))
                _value.append(tuple(obj_owner))
                value = tuple(_value)
            setattr(self, key, value)

    def setcache(self, path, val, time, index):
        self._cache.setdefault(path, {})[index] = (val, time)

    def getcache(self, path, exp, index):
        value, created = self._cache[path][index]
        if created is None or exp <= created:
            return True, value
        return False, None  # pragma: no cover

    def hascache(self, path, index):
        """ path is in the cache

        :param path: the path's option
        """
        return path in self._cache and index in self._cache[path]

    def reset_expired_cache(self, exp):
        cache_keys = list(self._cache.keys())
        for key in cache_keys:
            key_cache_keys = list(self._cache[key].keys())
            for index in key_cache_keys:
                val, created = self._cache[key][index]
                if created is not None and exp > created:
                    del(self._cache[key][index])
                    if self._cache[key] == {}:
                        del(self._cache[key])

    def reset_all_cache(self):
        "empty the cache"
        self._cache.clear()

    def get_cached(self, context):
        """return all values in a dictionary
        example: {'path1': {'index1': ('value1', 'time1')}, 'path2': {'index2': ('value2', 'time2', )}}
        """
        return self._cache
