# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Interface orientée objet aux commandes RRDtool.

Il est conseillé de consulter la documentation des outils RRDtool:
L{http://people.ee.ethz.ch/~oetiker/webtools/rrdtool/manual/index.html}.
"""

try: _ # localized string fetch function
except NameError: _ = str

from datetime import datetime, timedelta

import rrdtool, os

from zephir.monitor.agentmanager.util import TIME_ORIGIN, utcnow, log

DATASOURCE_TYPES = ['GAUGE', 'COUNTER', 'DERIVE', 'ABSOLUTE']
CONSOLIDATION_FUNCTIONS = ['AVERAGE', 'MIN', 'MAX', 'LAST']




def rrd_date_from_datetime(date):
    if date is None:
        timestamp = 'N'
    else:
        delta = date - TIME_ORIGIN
        timestamp = '%d' % (delta.days*24*3600 + delta.seconds)
    return timestamp




class Database:
    """Round-Robin Database (fichier C{.rrd})
    """
    
    def __init__(self, rrdfile, step=60):
        self.rrdfile = rrdfile # TODO path + basename without .rrd extension
        self.step = step
        self.datasources = []
        self.archives = []
        self.graphs = []


    def new_datasource(self, name, ds_type='GAUGE', heartbeat=None,
                       min_bound=None, max_bound=None):
        """Ajoute une nouvelle I{datasource} (DS) à la base de données
        """
        if heartbeat is None:
            heartbeat = 2 * self.step
        ds = Datasource(name, ds_type, heartbeat, min_bound, max_bound)
        self.datasources.append(ds)


    def new_archive(self, rows, consolidation='AVERAGE',
                    steps=1, xfiles_factor=0):
        """Ajoute une nouvelle archive I{round-robin} (RRA) à la base
        de données.
        """
        rra = Archive(rows, consolidation, steps, xfiles_factor)
        self.archives.append(rra)


    def new_graph(self, pngname, vnamedefs, options):
        """Ajoute un nouveau graphe à la base de données.

        @param vnamedefs: {vname: (ds_name, CF)}
        """
        #TODO default options
        defs = [ "DEF:%s=%s:%s:%s" % (vname, self.rrdfile, ds, cf)
                 for vname, (ds, cf) in vnamedefs.items() ]
        graph = Graph(pngname, defs, *options)
        self.graphs.append(graph)


    def create(self):
        """Crée le fichier C{.rrd} une fois que les datasources,
        archives et graphes ont été configurés.
        """
        # check if already created
        if not os.path.exists(self.rrdfile):
            begin = rrd_date_from_datetime(utcnow()) # - timedelta(seconds=10))
            args = [self.rrdfile,
                    "-b%s" % begin,
                    "-s%d" % self.step]
            args += map(str, self.datasources)
            args += map(str, self.archives)
            #log.msg('RRDtool create: %s' % ' '.join(args))
            rrdtool.create(*args)


    def update(self, values, date=None): #values = list | dict
        """Insère une nouvelle valeur dans la base de données.

        @param values: soit une liste de valeurs (données dans l'ordre
        de déclaration des champs), soit un dictionnaire C{{champ:
        valeur}}.
        """
        if date is None:
            date = utcnow()
        args = [self.rrdfile]
        if type(values) is dict:
            items = values.items()
            args.append('-t' + ':'.join([ str(i[0]) for i in items]))
            values = [ str(i[1]) for i in items]
        args.append(rrd_date_from_datetime(date)+ ':' + ':'.join(values))
        #log.msg(' '.join(args))
        try:
            rrdtool.update(*args)
        except rrdtool.error, e:
            log.msg(_('RRDtool warning: ') + str(e))


    def graph_all(self, additional_args = None):
        """Génère ou met à jour tous les graphes de cette base de
        données.
        """
        for g in self.graphs:
            g.graph(additional_args)

    def info(self):
        """retourne des informations sur le fichier rrd associé
        """
        if os.path.exists(self.rrdfile):
            return rrdtool.info(self.rrdfile)
        else:
            return None

class Datasource:

    def __init__(self, name, ds_type, heartbeat,
                 min_bound=None, max_bound=None):
        assert not name is ""
        assert ds_type in DATASOURCE_TYPES
        assert not heartbeat is None
        self.name = name
        self.type = ds_type
        self.heartbeat = heartbeat
        self.min_bound = min_bound
        self.max_bound = max_bound

    def __str__(self):
        minb, maxb = "U", "U"
        if self.min_bound is not None: minb = str(self.min_bound)
        if self.max_bound is not None: maxb = str(self.max_bound)
        return "DS:%(ds-name)s:%(DST)s:%(heartbeat)d:%(min)s:%(max)s" % {
            'ds-name': self.name, 'DST': self.type,
            'heartbeat': self.heartbeat,
            'min': minb, 'max': maxb
            }
    

    

class Archive:

    def __init__(self, rows, consolidation='AVERAGE',
                 steps=1, xfiles_factor=0):
        assert consolidation in CONSOLIDATION_FUNCTIONS
        assert xfiles_factor >= 0.0 and xfiles_factor < 1
        self.rows = rows
        self.consolidation = consolidation
        self.steps = steps
        self.xfiles_factor = xfiles_factor

    def __str__(self):
        return "RRA:%(CF)s:%(xff)f:%(steps)d:%(rows)d" % {
            'CF': self.consolidation, 'xff': self.xfiles_factor,
            'steps': self.steps,
            'rows': self.rows
            }


    

class Graph:

    def __init__(self, imgname, defs, *options):
        self.imgname = imgname
        self.defs = defs
        self.options = list(options)
        # self.options.append('-z')
        
    def graph(self, additional_args = None):
        #log.msg("*** rrd.graph %s" % self.imgname)
        args = [self.imgname]
        args += map(str, self.defs)
        args += self.options
        if additional_args is not None:
            args += additional_args
        return rrdtool.graph(*args)




# def test_main():
#     test_support.run_unittest(UserStringTest)
 
# if __name__ == "__main__":
#     test_main()
