# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour les cartes réseau
"""

import os
from ifconfig import ifconfig
from creole.client import CreoleClient
from twisted.internet.utils import getProcessValue
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img, boolean_to_onoff
from zephir.monitor.agentmanager.util import log

class Network(Agent):

    def __init__(self, name,
                 **params):
        Agent.__init__(self, name, **params)
        self.table = TableData([
            ('status', "état", {'align':'center'}, status_to_img),
            ('interf', "Interface", {'align':'left'}, None),
            ])
        self.data = [self.table]
        self.dico = CreoleClient().get_creole()

    def measure(self):
        self.mesures = []
        # on regarde si on est sur Amon/Sphynx ou autre
        bin_ipsec = "/usr/sbin/ipsec"
        if os.path.exists(bin_ipsec) and self.dico.get('install_rvp', 'non') == 'oui':
            if self.dico.get('activer_haute_dispo', 'non') == 'non':
            # Si Amon ou Sphynx sans haute dispo, on mesure rvp
                return self.measure_rvp(bin_ipsec)
            else:
            # Si Sphynx Haute dispo, on teste le node actif
                cmd = "ip route|grep default|grep -q src"
                errcode = os.system(cmd)
                if errcode == 0:
                # Si Sphynx haute dispo et node actif, on mesure rvp
                    return self.measure_rvp(bin_ipsec)
                else:
                # Si Sphynx haute dispo et node inactif, on mesure uniquement network
                    return self.measure_network()
        else:
            return self.measure_network()

    def measure_rvp(self, cmd_rvp):
        update = getProcessValue(cmd_rvp, args=['statusall'],
                                 env={'ModeTxt': 'yes'})
        return update.addCallbacks(self.callback_rvp, self.errback_rvp)

    def callback_rvp(self, data):
        #log.msg('agent network : traitement ipsec')
        status_ipsec = 'On'
        if data != 0:
            status_ipsec = 'Off'
        self.mesures.append({ 'interf': 'ipsec',
                              'status': status_ipsec })
        self.measure_data['ipsec'] = status_ipsec
        return self.measure_network()

    def errback_rvp(self, err):
        log.msg('agent network : erreur, ipsec arrêté : ')
        self.mesures.append({ 'interf': 'ipsec',
                              'status': 'Off' })
        self.measure_data['ipsec'] = 'Off'
        return self.measure_network()

    def measure_network(self):
        """
        test des interfaces réseau
        """
        d = self.dico
        ic = ifconfig()
        # affiche seulement les interfaces ok (sans réslolution DNS)
        try:
            options = ['-a', '-A', '-q']
            interfaces = {}
            index_int = 0
            nb_int = d['nombre_interfaces']
            search = True
            while search:
                if nb_int and (index_int >= int(nb_int)):
                    search = False
                elif ('eth%d_method' % index_int in d) and (d['eth%d_method' % index_int] == 'dhcp'):
                    pass
                elif ('adresse_ip_eth%d' % index_int in d) and (d['adresse_ip_eth%d' % index_int] != ''):
                    pass
                else:
                    search = False
                if search:
                    try:
                        interfaces['eth%d' % index_int] = ic.getStatus('eth%d' % index_int)
                    except:
                        import traceback
                        traceback.print_exc()
                        interfaces['eth%d' % index_int] = False
                index_int += 1
            # amonecole: adresse_ip_dmz
            interfaces_keys = interfaces.keys()
            interfaces_keys.sort()
            if 'adresse_ip_dmz' in d:
                interfaces['dmz'] = ic.getStatus('dmz')
                interfaces_keys.append('dmz')
            for interf in interfaces_keys:
                self.mesures.append({'interf': interf,
                                    'status': boolean_to_onoff(interfaces[interf])})
                self.measure_data[interf] = boolean_to_onoff(interfaces[interf])
        except:
            import traceback
            traceback.print_exc()
        return {'networks': self.mesures}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['networks']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None:
            for service in self.last_measure.value['networks']:
                if service['status'] != 'On':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()
