# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007  
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr 
#  
# prelude_options.py
#  
# options pour l'enregistrement des sondes prelude
#       
# les options sont stockées dans un dictionnaire de la forme:
#
#       {<nom_sonde>: [uid, gid, services]}
#
#   uid/gid : utilisateur à utiliser à l'appel de add-user
#   services : liste des services à relancer après enregistrement
#
# Les options snt chargées depuis des fichiers de type .ini 
#
# par défaut : /usr/share/zephir/zephir_conf/prelude_eole.ini
# si d'autres fichiers de ce type sont présents dans le répertoire,
# ils seront lus aussi. ordre de priorité des fichiers (croissant):
# - prelude_eole.ini (installé par défaut)
# - prelude_acad.ini (si présent)
# - autres fichiers du type prelude_XXXX.ini
#   (ordre de chargement déterminé par le nom de fichier,
#   le dernier chargé est prioritaire)

from glob import glob
from ConfigParser import ConfigParser
import os

DEFAULT_UID = 'root'
DEFAULT_GID = 'root'
DEFAULT_PERMS = 'idmef:w admin:r'

def get_conf_files(zephir_dir):
    """détermine les fichiers à charger et l'ordre de prise en compte"""
    conf_files = []
    fics = glob(os.path.join(zephir_dir, 'zephir_conf', 'prelude_*.ini'))
    fics.sort()
    for std_f in ('eole', 'acad'):
        std_file = os.path.join(zephir_dir, 'zephir_conf', 'prelude_%s.ini' % std_f)
        if std_file in fics:
            conf_files.append(std_file)
            fics.remove(std_file)
    # on ajoute les éventuels fichiers restants
    conf_files.extend(fics)
    return conf_files

def get_options(zephir_dir):
    """création de la liste des fichiers de définition"""
    conf_files = get_conf_files(zephir_dir)
    # calcul du dictionnaire des options
    options = {}

    for conf_file in conf_files:
        conf_p = ConfigParser()
        conf_p.read(conf_file)
        for sonde in conf_p.sections():
            perms = uid = gid = services = None
            # vérification des options déjà chargées
            if sonde in options:
                uid, gid, services = options[sonde]
            if conf_p.has_option(sonde, 'permissions'):
                perms = conf_p.get(sonde, 'permissions')
            else:
                perms = perms or DEFAULT_PERMS
            if conf_p.has_option(sonde, 'uid'):
                uid = conf_p.get(sonde, 'uid')
            else:
                uid = uid or DEFAULT_UID
            if conf_p.has_option(sonde, 'gid'):
                gid = conf_p.get(sonde, 'gid')
            else:
                gid = gid or DEFAULT_GID
            if conf_p.has_option(sonde, 'services'):
                services = conf_p.get(sonde, 'services').split(',')
            else:
                services = services or [sonde]
            # prise en compte des options pour cette sonde (écrase si déjà présent)
            options[sonde] = [perms, uid, gid, services]
    return options
