#!/usr/bin/env python
# -*- coding: utf-8 -*-
################################################################################
# Eole NG - 2008
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# upgrade_distrib.py
#
# script de mise à jour du serveur dans zephir après changement de distribution
#
# - vérification du module actuel dans la base zephir
# - migration dans la base vers le module installé sur le serveur
#
################################################################################

import sys, os, getpass, time, socket, glob, termios
from pyeole.ihm import print_orange, print_red, print_green, print_line
from creole.loader import creole_loader, config_save_values, _list_extras
from creole.config import dicos_dir, configeol

def flushed_input(msg):
    sys.stdout.write(msg)
    sys.stdout.flush()
    return sys.stdin.readline().strip()

registered = False
# mode auto ou migration depuis 2.2/2.3
auto = "--auto" in sys.argv[1:]
msg_backup = ""

def message_fin(zephirok=False):
    print_line("")
    global msg_backup
    if msg_backup:
        print_green(msg_backup)
        print_line("")
    print_orange("Pour finaliser la procédure, veuillez :")
    print_line("")
    if auto:
        print_orange("- exécuter la commande 'reconfigure'")
    else:
        print_orange("- redémarrer le serveur")
        if not zephirok:
            print_orange("- mettre à jour la configuration (par exemple via gen_config)")
        if os.path.isdir("/opt/lxc/"):
            print_orange("- exécuter la commande gen_conteneurs")
        print_orange("- exécuter la commande 'instance'")
    print_line("")
    if os.path.exists('/etc/ha.d/.hdinitialized'):
        print_line("")
        print_red("Attention, exécuter la procédure d'upgrade sur le noeud esclave avant d'instancier")
        print_line("")

def msg_relance_enregistrement():
    print_line("- Ré-enregisrer le serveur via enregistrement_zephir")
    print_line("  * Choisir : 2 - relancer l'enregistrement")
    print_line("  * Ne pas créer de nouveau serveur et utiliser le même identifiant")

try:
    from zephir.zephir_conf.zephir_conf import id_serveur, adresse_zephir
    assert int(id_serveur) > 0
    from zephir.lib_zephir import lock
    from zephir.lib_zephir import unlock
    from zephir.lib_zephir import convert
    from zephir.lib_zephir import xmlrpclib
    from zephir.lib_zephir import update_sudoers
    from zephir.lib_zephir import get_module_var
    from zephir.lib_zephir import migrate_data
    from zephir.lib_zephir import exit_err
    from zephir.lib_zephir import zephir_dir
    from zephir.lib_zephir import EoleProxy
    from zephir.lib_zephir import TransportEole
    from zephir.lib_zephir import get_paqs_from_logs
    from zephir.lib_zephir import lock
    from zephir.lib_zephir import unlock
    from zephir.lib_zephir import public_dir
    registered = True
except:
    # serveur non enregistré sur zephir, on passe cette étape
    print_line("")
    print_line("""Pas d'enregistrement dans une base Zéphir détecté.""")

def backup_dicts():
    """ met de côté les dictionnaires locaux/de variante
    """
    global msg_backup
    dict_found = False
    for subdir in ('local', 'variante'):
        backup_dir = os.path.join("/root/backup_dicos", subdir)
        dicts = glob.glob(os.path.join(dicos_dir, subdir, '*.xml'))
        if dicts:
            dict_found = True
            if not os.path.isdir(backup_dir):
                os.makedirs(backup_dir)
            for dic_perso in dicts:
                dict_name = os.path.basename(dic_perso)
                os.rename(dic_perso, os.path.join(backup_dir, dict_name))
    if dict_found:
        msg_backup = "Les anciens dictionnaires personnalisés ont été archivés dans {0}"
        msg_backup = msg_backup.format(os.path.dirname(backup_dir))

def upgrade():
    """migration du serveur dans zephir"""

    # import des fonctions communes de Zéphir client
    from zephir.lib_zephir import creole_vars
    # on demande un rechargement de la configuration pour être sur que
    # creoled n'est pas en train de recharger les dictionnaires
    creole_vars.reload_config()

    print_orange("""
Cette fonctionnalité nécessite un compte ayant les permissions suivantes
dans l'application Zéphir gérant ce serveur (%s) :""" % adresse_zephir)
    print_line("""
- Lecture
- Actions sur les clients (avec ou sans modification de configuration) ou enregistrement
- Ecriture sur les serveurs et les modules (ou Migration de serveur + Ecriture (modules))

    """)


    authentified, data = get_pwd(adresse_zephir, 7080)
    if authentified == False:
        return data
    else:
        proxy = data
    id_new_mod = 0
    old_module = ""
    id_variante = ""
    # on récupère les informations sur le serveur dans la base zephir
    try:
        info_serveur = convert(proxy.serveurs.get_serveur(id_serveur))
    except xmlrpclib.ProtocolError:
        return """Erreur de permissions ou Zéphir non disponible"""
    except socket.error, e:
        return """Erreur de connexion au serveur Zéphir (%s)""" % str(e)
    if info_serveur[0] == 0:
        return "Erreur de recherche du serveur dans la base zephir : "+str(info_serveur[1])
    # récupération du module installé sur la machine
    try:
        module = creole_vars.get_creole('eole_module', '')
        version = creole_vars.get_creole('eole_version', '')
        release = creole_vars.get_creole('eole_release', '')
        if release != '2.4.0':
            # sur Zéphir, la release est considérée comme une version à
            # part entière de la distribution. Pour des raisons 'historiques'
            # la version 2.4.0 est vue comme 2.4 dans Zéphir
            version = release
        assert module, version
    except:
        raise Exception('impossible de trouver le nom de module')
    module_local = '%s-%s' % (module, version)
    # on vérifie le n° de module du serveur dans la base zephir
    id_module = info_serveur[1][0]['module_actuel']
    modules = convert(proxy.modules.get_module())
    if modules[0] == 0:
        return "Erreur de recherche du module dans la base zephir : "+str(modules[1])
    # on détermine le numéro dans zephir du module installé
    for mod in modules[1]:
        if mod['libelle'] == module_local:
            id_new_mod = mod['id']
        if mod['id'] == id_module:
            old_module = mod['libelle']
    # mise à jour de /etc/sudoers
    update_sudoers()

    # gestion de la configuration et migration du serveur dans l'application Zéphir
    #
    # Suivant le paramètre auto, 2 modes sont possibles:
    # - auto = True : Upgrade entre 2 release. La gestion de la configuration est automatique
    # - auto = False: Migration depuis un module 2.2/2.3. Une configuration de migration doit
    #                 être générée au préalable dans Zéphir ou mise à jour via gen_config
    #
    # Si la configuration de migration n'a pas été générée au préalable, on prévient l'utilisateur
    # qu'il devra valider la configuration localement, puis lancer enregistrement_zephir en sauvegardant
    # la configuration sur Zéphir
    #
    # Dans le cas inverse, on effectue une descente de la configuration de migration et des données de variante
    # Puis, on fait une sauvegarde complète pour remonter les fichiers locaux
    if auto:
        zephir_conf_ok = True
        id_variante = ""
    else:
        zephir_conf_ok = False
        try:
            # détection de l'état de migration dans Zéphir
            code, res = convert(proxy.serveurs.get_status(id_serveur))
            if code == 1 and res['migration_ok'] == 0:
                # une nouvelle configuration est présente sur Zéphir
                zephir_conf_ok = True
        except xmlrpclib.ProtocolError:
            return """Erreur de permissions ou zephir non disponible"""
        except socket.error:
            return """Erreur lors de la détection de l'état de migration dans l'application zephir"""
        if not zephir_conf_ok:
            print_line("")
            print_red("Attention ! La configuration de migration n'a pas été préparée sur le serveur Zéphir")
            print_line("")
            print_orange("Veuillez sélectionner une variante parmi celles proposées :")
            id_variante = get_module_var(proxy, id_new_mod)
    try:
        assert id_new_mod != 0 and old_module != ""
        # on indique à migrate_serveur qu'on est en mode Upgrade-Auto
        code, res = convert(proxy.serveurs.migrate_serveur(id_serveur, {}, id_new_mod, id_variante, True))
        if code == 0:
            return "Erreur lors de la migration dans l'application Zéphir: %s" % res
        # affichage de la variante du serveur migré
        try:
            code_ret, data = convert(proxy.serveurs.get_serveur(id_serveur))
            if code_ret == 1:
                code_ret, var_info = convert(proxy.modules.get_variante(data[0]['variante']))
                if code_ret == 1:
                    print_green("\nLe serveur a été basculé dans Zéphir sur la variante %d (%s) du module %s" \
                    % (data[0]['variante'], var_info[0]['libelle'], module_local))
        except:
            print_red("\n!! Erreur lors de la vérification des données du serveur sur Zéphir !!")

    except xmlrpclib.ProtocolError:
        return """Erreur de permissions ou zephir non disponible"""
    except socket.error:
        return """Erreur lors de la migration dans l'application zephir"""

    if auto:
        creole_vars.reload_config()
        # upgrade automatique du config.eol
        upgrade_config()

        # sauvegarde de la configuration locale sur zephir en fin de procédure
        print_line('\nSauvegarde de la configuration actuelle sur Zéphir')
        res = os.system('/usr/share/zephir/scripts/zephir_client del_lock >/dev/null')
        res = os.system('/usr/share/zephir/scripts/zephir_client save_files >/dev/null')
        print_line('\nRécupération des données de variante sur Zéphir')
    else:
        # migration depuis 2.2/2.3, les données ne sont pas reprises automatiquement
        # on regarde des données (fichiers divers) à migrer ont été déclarées
        try:
            code_ret, available = migrate_data(proxy, id_serveur, check=True)
        except socket.error, e:
            exit_err("Erreur lors de l'appel au serveur Zéphir (%s)" % str(e))
        if code_ret == 1 and available == True:
            print_orange("""\nCertaines données utilisateur peuvent être récupérées\n(fichiers divers définis sur Zéphir)\n""")
            rep_mig = flushed_input("""Voulez vous migrer ces données (O/N) ?""")
            if rep_mig.upper() in ['O','OUI']:
                try:
                    code_ret, infos = migrate_data(proxy, id_serveur)
                except socket.error, e:
                    exit_err("Erreur lors de l'appel au serveur Zéphir (%s)" % str(e))
                if code_ret == 0:
                    print_red("!! erreur lors de la migration des données suivantes :\n %s" % infos)
        # après migration, on descend les données et la configuration migrées
        if not zephir_conf_ok:
            print_line('\nRécupération des données de variante sur Zéphir')
        else:
            print_line('\nRécupération de la configuration et des données de variante sur Zéphir')
    res = proxy.uucp.configure(id_serveur)
    if res[0] == 1:
        res = os.system("%s/scripts/zephir_client call >/dev/null" % zephir_dir)
    else:
        exit_err("\n erreur lors de la récupération des données de configuration : %s \n" % res[1])
    print_line("** attente de la mise en place des fichiers **")
    # on attend que les fichiers soient mis en place
    # (timeout de 20 secondes au cas ou la procédure échoue)
    wait = 0
    while wait < 40:
        time.sleep(0.5)
        wait += 1
        if os.path.exists(public_dir+'/config-zephir.tar.ok'):
            wait = 100
    get_paqs_from_logs()
    if wait != 100:
        # Problème de configuration UUCP : Indications sur la démarche à suivre dans ce cas
        print_red("!!  La mise en place de la configuration s'est mal terminée !!")
        print_line("")
        print_orange("Le serveur a  été migré dans Zéphir, mais les échanges semblent non fonctionnels")
        print_orange("Vous pouvez essayer de:")
        print_line("")
        if auto:
            msg_relance_enregistrement()
            print_line("  * Choisir de sauvegarder la configuration locale sur Zéphir en fin d'enregistrement")
            print_line("- Suivre ensuite les instructions ci dessous pour finaliser la migration")
            message_fin(True)
            sys.exit(1)
        elif zephir_conf_ok:
            msg_relance_enregistrement()
            print_line("  * Choisir de récupérer la configuration définie sur Zéphir en fin d'enregistrement")
            print_line("- Suivre ensuite les instructions ci dessous pour finaliser la migration")
            print_line("- Demander une sauvegarde de l'état du serveur dans l'application Zéphir")
            message_fin(True)
            sys.exit(1)
        if not auto and not zephir_conf_ok:
            message_fin()
            msg_relance_enregistrement()
            print_line("  * Choisir de sauvegarder la configuration locale sur Zéphir en fin d'enregistrement")
            sys.exit(1)
    if auto:
        creole_vars.reload_config()
        # upgrade automatique du config.eol
        upgrade_config()
    if zephir_conf_ok:
        # sauvegarde des données locales sur zephir en fin de procédure
        print_line('\nSauvegarde de la configuration actuelle sur Zéphir')
        res = os.system('/usr/share/zephir/scripts/zephir_client del_lock >/dev/null')
        res = os.system('/usr/share/zephir/scripts/zephir_client save_files >/dev/null')
        message_fin(True)
    else:
        message_fin()
        # si pas de configuration définie, la sauvegarde devra être faite manuellement
        # après génération de la nouvelle configuration avec gen_config
        print_orange("Puis, utilisez une de ces 2 méthodes pour remonter la configuration sur Zéphir :")
        print_orange(" - demander une sauvegarde de configuration dans l'application Zéphir")
        print_orange("  ou")
        print_orange(" - lancez '/usr/share/zephir/scripts/zephir_client save_files' sur ce serveur")
        print_line("")
    return ""

def get_pwd(addr, port):
    """lecture d'un login/passwd pour l'application zephir
    """
    login_ok = 0
    user = "toto"
    while login_ok == 0 and user != "":
        # flush de l'entrée standard au cas où l'utilisateur aurait
        # tapé <entrée> pendant l'Upgrade
        termios.tcflush(sys.stdin, termios.TCIOFLUSH)
	user = flushed_input("Entrez votre login zephir (rien pour sortir) : ")
        if user != "":
            print_line("Mot de passe zephir pour %s : " % user)
            passwd = getpass.getpass("")
            # création du proxy avec zephir
            proxy = EoleProxy("https://%s:%s@%s:%s" % (user, passwd, addr, port), transport=TransportEole())
            login_ok = 1
            try:
                res = convert(proxy.get_permissions(user))
            except xmlrpclib.ProtocolError:
                login_ok = 0
                print_line("\n Erreur d'authentification \n")
        else:
            return False, "! Abandon de la procédure !"
    return True, proxy

def upgrade_config():
    """
    Mise à jour automatique du config.eol local
    """
    print
    print_line("Mise à jour des fichiers de configuration")
    load_extra = True
    try:
        loader = creole_loader(load_extra=True)
    except Exception, err:
        print_red("Erreur de chargement lors du chargement des configurations : {0}".format(err))
        try:
            loader = creole_loader(load_extra=False)
            print_line("Configuration chargée sans les extra...")
            load_extra = False
        except:
            return
    try:
        config_save_values(loader, namespace='creole', reload_config=False)
        print_line("* Fichier {0} mis à niveau".format(configeol))
    except Exception, err:
        print_red("Impossible d'enregistrer le fichier {0} : {1}".format(configeol, err))
        return
    if not load_extra:
        print_line("Les fichiers extra n'ont pas été mis à niveau")
        return
    for extra in _list_extras():
        try:
            config_save_values(loader, namespace=extra, reload_config=False)
            print_line("* Configuration extra {0} mise à niveau".format(extra))
        except Exception, err:
            print_red("Erreur de migration de la configuration extra {0} : {1}".format(extra, err))


if __name__ == '__main__':
    if os.path.isfile('/etc/init.d/zephir'):
        is_zephir = True
    else:
        is_zephir = False

    if registered:
        # mise de côté des dictionnaires locaux/de variante pour limiter
        # les risques de plantage de creoled :
        #
        #   https://dev-eole.ac-dijon.fr/issues/9787
        #
        backup_dicts()
        if not is_zephir:
            os.system('/etc/init.d/z_stats stop >/dev/null')
        else:
            os.system('/etc/init.d/zephir stop >/dev/null')
        # on bloque les fonctions zephir pendant la procédure
        lock(['maj','reconfigure'])
        res = upgrade()
        if auto and res != '':
            # L'ugrade de la config n'a pas été faite
            upgrade_config()
        # upgrade des infos enregistrées sur zephir pour ce client
        unlock(['maj','reconfigure'])
        if res == "NO CONF":
            # Serveur non upgradé, on ne relance pas z_stats
            # (enregistrement_zephir nécessaire)
            sys.exit(0)
        elif res not in ["", "NO CONF"]:
            print_line(res)
        if not is_zephir:
            os.system('/etc/init.d/z_stats start >/dev/null 2>&1')
        else:
            os.system('/etc/init.d/zephir start >/dev/null')
        if res not in ["", "NO CONF"]:
            message_fin()
    else:
        if auto:
            upgrade_config()
        message_fin()
