# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Various helper functions
"""

try: _ # localized string fetch function
except NameError: _ = str

import os
from hashlib import md5

from twisted.python import log
import time, calendar
from datetime import datetime, timedelta, tzinfo

from zephir.monitor.agentmanager import status

log.FileLogObserver.timeFormat = "%Y/%m/%d %H:%M:%S %Z"

def percent(quantity, total):
    """quantity/total rate, as a percentage
    """
    quantity, total = float(quantity), float(total)
    if total == 0:
        return 0
    else:
        return int((quantity * 100.0) / total)

def expand_dirname(dirname):
    """Expande C{~} et les variables d'environnement dans C{dirname},
    et ajoute un séparateur terminal si nécessaire."""
    result = os.path.expanduser(os.path.expandvars(dirname))
    assert os.path.isdir(result)
    if not result.endswith(os.path.sep):
        result += os.path.sep
    return result

def boolean_to_onoff(b):
    if b: return "On"
    else: return "Off"

def ensure_dirs(*dirs):
    """Crée les répertoires passés en argument si nécessaire"""
    for d in dirs:
        if not os.path.isdir(d):
            log.msg(_('Creating dir %s') % d)
            os.makedirs(d) # will die if there is a file, but it's ok


class UTC(tzinfo):
    """UTC timezone

    To be throwed away if one day standard timezones
    get implemented in python...
    """
    def utcoffset(self, dt): return timedelta(0)
    def tzname(self, dt): return "UTC"
    def dst(self, dt): return timedelta(0)

utc = UTC()
TIME_ORIGIN = datetime(1970,1,1,tzinfo=utc)
MONTHS = dict(JAN=1, FEV=2, FEB=2, MAR=3, APR=4, AVR=4, MAY=5, MAI=5,
         JUN=6, JUI=7, AOU=8, AUG=8, SEP=9, OCT=10, NOV=11, DEC=12)

def parse_date(date):
    month, day, hour, year = date.split()
    hour, min, sec = hour.split(':')
    # suppression des accents et mise en majuscule pour les mois
    month = month.replace('û','u').replace('é','e').upper()
    month = MONTHS[month]
    return datetime(int(year), int(month), int(day), int(hour), int(min), int(float(sec)))

def utcnow():
    """Aware UTC datetime"""
    #return datetime.utcnow().replace(tzinfo=utc)
    return datetime.now(utc)

def utcfromtimestamp(timestamp):
    #return TIME_ORIGIN + timedelta(seconds=timestamp)
    return datetime.utcfromtimestamp(timestamp).replace(tzinfo=utc)

def timestampfromutc(date):
    #return (date - TIME_ORIGIN).seconds
    return calendar.timegm(date.utctimetuple())

def status_to_img(s):
    if isinstance(s, status.Status):
        if s == status.Unknown():
            s = ""
        elif s == status.OK():
            s = "On"
        else:
            s = "Off"
    if s == "":
        return '<img src="/static/img/gris.gif" alt="??"/>'
    elif s == "On":
        return '<img src="/static/img/vert.gif" alt="On"/>'
    return '<img src="/static/img/rouge.gif" alt="Off"/>'

# fichiers à vérifier par version de distribution: (nom fichier/repertoire, equivalent sur zephir, extension ou fin de fichier)
md5files_default = [("variables.eol", "variables.eol",None),
               ("patch", "patchs",".patch"),
               ("patch/variante", "patchs/variante",".patch"),
               ("dicos/local", "dicos/local",".xml"),
               ("dicos/variante", "dicos/variante",".xml")]

md5files = {
            # eole 1.X
            1:[("variables.eol", "variables.eol",None),
               ("patch", "patchs",".patch"),
               ("patch/variante", "patchs/variante",".patch"),
               ("dicos", "dicos",".eol"),
               ("dicos/variante", "dicos/variante",".eol"),
              ],
            # eole 2.0
            2:[("zephir.eol", "zephir.eol",None),
               ("patch", "patchs",".patch"),
               ("patch/variante", "patchs/variante",".patch"),
               ("dicos/local", "dicos/local",".xml"),
               ("dicos/variante", "dicos/variante",".xml"),
              ],
            # eole 2.1
            3: md5files_default,
            # eole 2.2
            4: md5files_default,
            # eole 2.3
            5: md5files_default,
            # support eole 2.4 pour Zéphir 2.3
            6: md5files_default,
            7: md5files_default,
           }

def get_md5files(version):
    """renvoie les fichiers à vérifier en fonction de la distribution
    """
    # toutes les versions > 2.0 utilisent md5files_default
    return md5files.get(version, md5files_default)

def md5file(filename):
    """Return the hex digest of a file without loading it all into memory"""
    fh = open(filename)
    digest = md5()
    while 1:
        buf = fh.read(4096)
        if buf == "":
            break
        digest.update(buf)
    fh.close()
    return digest.hexdigest()
